package org.lsst.ccs.drivers.archon;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.ShortBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import static java.nio.file.StandardOpenOption.READ;
import java.util.HashSet;
import java.util.Set;

/*
 * Typically, a host program will issue the FRAME command, and check the
 * returned status data to see if there are any frame buffers holding
 * completed frames with a frame number greater than what the program has
 * already captured. If a new frame exists, the host program issues a LOCK
 * command to prevent that buffer from being overwritten, and then fetches
 * the pixel data using a FETCH command. The number of bytes to fetch for a
 * complete frame is the frame width times the frame height times 2 (for
 * 16-bit samples) or 4 (for 32-bit samples). Divide the number of bytes by
 * 1024 and round up to determine the number of blocks to fetch. The
 * starting address is 0x20000000 for frame buffer 1, 0x40000000 for frame
 * buffer 2, and 0x60000000 for frame buffer 3.
 */
public class RawImageData {

    public enum Sample {

        BIT16(2), BIT32(4);
        private final int bytes;

        private Sample(int bytes) {
            this.bytes = bytes;
        }

        public int bytes() {
            return bytes;
        }

        public int bits() {
            return 8 * bytes;
        }
    };

    public enum FrameMode {

        TOP, BOTTOM, SPLIT
    }

    int width;
    int height;
    Sample sample;
    FrameMode frameMode;
    long timestamp;
    long frame;
    byte[] data;

    public int getWidth() {
        return width;
    }

    public int getHeight() {
        return height;
    }

    public Sample getSample() {
        return sample;
    }

    public FrameMode getFrameMode() {
        return frameMode;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public byte[] getData() {
        return data;
    }
    
    public long getFrame() {
        return frame;
    }

    /**
     * Create a test image. Used for test cases and by the
     * DummyArchonController.
     *
     * @return
     */
    public static RawImageData createTestImage() {
        RawImageData testData = new RawImageData();
        // This image is set to be 35651584 bytes long, based on the test raw image Homer provided
        // Based on analysis by JimChiang we guess this represents:
        // Assuming the same number of pixels per segment and 16 segments, 
        // this implies 1114112 = 2^16 x 17 pixels.  So this looks like an e2v 
        // device, with 544 (=17*32) pixels in the serial direction (10 prescan 
        // + 512 physical + 22 overscan) and 2048 pixels in the parallel 
        // direction (2002 physical + 46 overscan). 
        testData.sample = Sample.BIT16;
        testData.frameMode = FrameMode.SPLIT;
        testData.timestamp = System.currentTimeMillis();
        testData.height = 2048 * 2;
        testData.width = 544 * 8;
        testData.data = new byte[testData.width * testData.height * 2];
        ByteBuffer bb = ByteBuffer.wrap(testData.data);
        bb.order(ByteOrder.LITTLE_ENDIAN);
        ShortBuffer sb = bb.asShortBuffer();
        for (int j = 0; j < testData.height; j++) {
            for (int i = 0; i < testData.width; i++) {
                sb.put((short) (isOverScan(i, j) ? 50 : 100 + 2 * i + 3 * j));
            }
        }
        return testData;
    }
    
    /**
     * Create a RawImageData object by reading a file of raw data.
     * @param rawFile The file to read
     * @return The raw image
     * @throws java.io.IOException
     */
    public static RawImageData createTestImage(Path rawFile) throws IOException {
        RawImageData testData = new RawImageData();
        // This image is set to be 35651584 bytes long, based on the test raw image Homer provided
        // Based on analysis by JimChiang we guess this represents:
        // Assuming the same number of pixels per segment and 16 segments, 
        // this implies 1114112 = 2^16 x 17 pixels.  So this looks like an e2v 
        // device, with 544 (=17*32) pixels in the serial direction (10 prescan 
        // + 512 physical + 22 overscan) and 2048 pixels in the parallel 
        // direction (2002 physical + 46 overscan). 
        testData.sample = Sample.BIT16;
        testData.frameMode = FrameMode.SPLIT;
        testData.timestamp = System.currentTimeMillis();
        testData.height = 2048 * 2;
        testData.width = 544 * 8;
        testData.data = new byte[testData.width * testData.height * 2];
        ByteBuffer bb = ByteBuffer.wrap(testData.data);
        bb.order(ByteOrder.LITTLE_ENDIAN);
        Set<OpenOption> options = new HashSet<>();
        options.add(READ);
        try (SeekableByteChannel newByteChannel = Files.newByteChannel(rawFile, options)) {
           newByteChannel.read(bb);
           return testData;
        }
    }

    private static boolean isOverScan(int i, int j) {
        //FIXME: Is all of the parallel overscan at the end??
        i %= 544;
        j %= 2048;
        return i < 10 || i >= 522 || j >= 2002;
    }
}
