package org.lsst.ccs.drivers.archon;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.Map;

public class ArchonStatus implements Serializable {

    public int valid;
    public int count;
    public int log;
    public int power;
    public float backplaneTemp;
    public float p2v5I;
    public float p2v5V;
    public float p5vV;
    public float p5vI;
    public float p6vV;
    public float p6vI;
    public float n6vV;
    public float n6vI;
    public float p17vV;
    public float p17vI;
    public float n17vV;
    public float n17vI;
    public float p35vV;
    public float p35vI;
    public float n35vV;
    public float n35vI;
    public float userV;
    public float userI;
    public float heaterV;
    public float heaterI;

    public BoardStatus[] boards;

    private static final long serialVersionUID = 4096596433181205957L;

    public static int pi(Map<String, String> map, String key) {
        return Integer.parseInt(map.get(key));
    }

    public static int pix(Map<String, String> map, String key) {
        return Integer.parseInt(map.get(key), 16);
    }

    public static int pim(Map<String, String> map, int id, String key) {
        return Integer.parseInt(map.get("MOD" + id + "/" + key));
    }

    public static float pf(Map<String, String> map, String key) {
        return Float.parseFloat(map.get(key));
    }

    public static float pfm(Map<String, String> map, int id, String key) {
        return Float.parseFloat(map.get("MOD" + id + "/" + key));
    }

    public static long pl(Map<String, String> map, String key) {
        return Long.parseLong(map.get(key));
    }

    public static long plx(Map<String, String> map, String key) {
        return Long.parseLong(map.get(key), 16);
    }

    public ArchonStatus() {

    }

    public ArchonStatus(ArchonConfig config, Map<String, String> map) {
        valid = pi(map, "VALID");
        count = pi(map, "COUNT");
        log = pi(map, "LOG");
        power = pi(map, "POWER");
        backplaneTemp = pf(map, "BACKPLANE_TEMP");
        p2v5V = pf(map, "P2V5_V");
        p2v5I = pf(map, "P2V5_I");
        p5vV = pf(map, "P5V_V");
        p5vI = pf(map, "P5V_I");
        p6vV = pf(map, "P6V_V");
        p6vI = pf(map, "P6V_I");
        n6vV = pf(map, "N6V_V");
        n6vI = pf(map, "N6V_I");
        p17vV = pf(map, "P17V_V");
        p17vI = pf(map, "P17V_I");
        n17vV = pf(map, "N17V_V");
        n17vI = pf(map, "N17V_I");
        p35vV = pf(map, "P35V_V");
        p35vI = pf(map, "P35V_I");
        n35vV = pf(map, "N35V_V");
        n35vI = pf(map, "N35V_I");
        userV = pf(map, "USER_V");
        userI = pf(map, "USER_I");
        heaterV = pf(map, "HEATER_V");
        heaterI = pf(map, "HEATER_I");

        boards = new BoardStatus[12];

        for (int i = 0; i < 12; i++) {
            switch (config.modules[i].type) {
            case AD:
                boards[i] = new ADStatus(i + 1, map);
                break;
            case DRIVER:
                boards[i] = new DriverStatus(i + 1, map);
                break;
            case HEATER:
                boards[i] = new HeaterStatus(i + 1, map);
                break;
            case HVBIAS:
                boards[i] = new HVBiasStatus(i + 1, map);
                break;
            case LVBIAS:
                boards[i] = new LVBiasStatus(i + 1, map);
                break;
            default:
                break;
            }

        }
    }

    public void dump() {
        Field[] ff = ArchonStatus.class.getDeclaredFields();
        try {
            for (Field f : ff) {
                if (f.getType().equals(float.class)) {
                    System.out.printf("%s = %f\n", f.getName(),
                            f.getFloat(this));
                } else if (f.getType().equals(int.class)) {
                    System.out.printf("%s = %d\n", f.getName(), f.getInt(this));
                }
            }
        } catch (Exception e) {
            // should not happen!
        }

        for (int i = 0; i < 12; i++) {
            if (boards[i] != null) {
                System.out.println("Board " + (i + 1));
                boards[i].dump();
            }
        }
    }

    /*
     * VALID=1 COUNT=103135 LOG=0 POWER=4 BACKPLANE_TEMP=33.062 P2V5_V=2.416
     * P2V5_I=2.031 P5V_V=4.673 P5V_I=5.017 P6V_V=6.141 P6V_I=0.584 N6V_V=-6.145
     * N6V_I=0.560 P17V_V=17.165 P17V_I=0.277 N17V_V=-17.212 N17V_I=0.199
     * P35V_V=35.281 P35V_I=0.065 N35V_V=0.000 N35V_I=0.000 USER_V=0.000
     * USER_I=-0.001 HEATER_V=28.096 HEATER_I=0.002
     */

    public static class BoardStatus implements Serializable {
        public float temperature;

        public void dump() {
            System.out.printf("   temperature %2.3f\n", temperature);
        }

        public BoardStatus(int id, Map<String, String> map) {
            temperature = pfm(map, id, "TEMP");
        }

        private static final long serialVersionUID = 210908887271052714L;

    }

    public static class HeaterStatus extends BoardStatus {
        public float tempA;
        public float tempB;
        public float heaterAOutput;
        public float heaterBOutput;
        public int vac;

        private static final long serialVersionUID = 4261393134690818187L;

        public HeaterStatus(int id, Map<String, String> map) {
            super(id, map);
            tempA = pfm(map, id, "TEMPA");
            tempB = pfm(map, id, "TEMPB");
            heaterAOutput = pfm(map, id, "HEATERAOUTPUT");
            heaterBOutput = pfm(map, id, "HEATERBOUTPUT");
            vac = pim(map, id, "VAC");
        }

        public void dump() {
            super.dump();
            Field[] ff = HeaterStatus.class.getDeclaredFields();
            try {
                for (Field f : ff) {
                    if (f.getType().equals(float.class)) {
                        System.out.printf("    %s = %f\n", f.getName(),
                                f.getFloat(this));
                    } else if (f.getType().equals(int.class)) {
                        System.out.printf("    %s = %d\n", f.getName(),
                                f.getInt(this));
                    }
                }
            } catch (Exception e) {
                // should not happen!
            }
        }

        /*
         * MOD2/TEMP=29.750 MOD2/TEMPA=-273.150 MOD2/TEMPB=-273.150
         * MOD2/HEATERAOUTPUT=0.900 MOD2/HEATERBOUTPUT=0.900 MOD2/VAC=-1
         */
    }

    public static class LVBiasStatus extends BoardStatus {
        public float[] lvlc_V = new float[24];
        public float[] lvlc_I = new float[24];
        public float[] lvhc_V = new float[6];
        public float[] lvhc_I = new float[6];

        private static final long serialVersionUID = 3237985632825573221L;

        public LVBiasStatus(int id, Map<String, String> map) {
            super(id, map);
            for (int i = 0; i < 24; i++) {
                lvlc_V[i] = pfm(map, id, "LVLC_V" + (i + 1));
                lvlc_I[i] = pfm(map, id, "LVLC_I" + (i + 1));
            }
            for (int i = 0; i < 6; i++) {
                lvhc_V[i] = pfm(map, id, "LVHC_V" + (i + 1));
                lvhc_I[i] = pfm(map, id, "LVHC_I" + (i + 1));
            }
        }

        @Override
        public void dump() {
            super.dump();
            for (int i = 0; i < 24; i++) {
                System.out.printf("    LVLC V%d = %f  I%d = %f\n", i,
                        lvlc_V[i], i, lvlc_I[i]);
            }
            for (int i = 0; i < 6; i++) {
                System.out.printf("    LVHC V%d = %f  I%d = %f\n", i,
                        lvhc_V[i], i, lvhc_I[i]);
            }

        }

        /*
         * MOD4/LVLC_V1=0.000 MOD4/LVLC_I1=0.001 MOD4/LVLC_V2=0.000
         * MOD4/LVLC_I2=0.000 MOD4/LVLC_V3=0.000 MOD4/LVLC_I3=0.000
         * MOD4/LVLC_V4=0.000 MOD4/LVLC_I4=0.000 MOD4/LVLC_V5=0.000
         * MOD4/LVLC_I5=0.000 MOD4/LVLC_V6=0.000 MOD4/LVLC_I6=0.000
         * MOD4/LVLC_V7=0.000 MOD4/LVLC_I7=0.000 MOD4/LVLC_V8=0.000
         * MOD4/LVLC_I8=0.000 MOD4/LVLC_V9=0.000 MOD4/LVLC_I9=0.000
         * MOD4/LVLC_V10=0.000 MOD4/LVLC_I10=0.000 MOD4/LVLC_V11=0.000
         * MOD4/LVLC_I11=0.000 MOD4/LVLC_V12=4.000 MOD4/LVLC_I12=0.110
         * MOD4/LVLC_V13=0.000 MOD4/LVLC_I13=-0.003 MOD4/LVLC_V14=0.000
         * MOD4/LVLC_I14=0.000 MOD4/LVLC_V15=0.000 MOD4/LVLC_I15=0.000
         * MOD4/LVLC_V16=0.000 MOD4/LVLC_I16=0.000 MOD4/LVLC_V17=0.000
         * MOD4/LVLC_I17=-0.001 MOD4/LVLC_V18=0.000 MOD4/LVLC_I18=0.000
         * MOD4/LVLC_V19=-0.001 MOD4/LVLC_I19=0.000 MOD4/LVLC_V20=0.000
         * MOD4/LVLC_I20=0.000 MOD4/LVLC_V21=0.000 MOD4/LVLC_I21=0.000
         * MOD4/LVLC_V22=0.000 MOD4/LVLC_I22=-0.001 MOD4/LVLC_V23=0.000
         * MOD4/LVLC_I23=-0.001 MOD4/LVLC_V24=0.000 MOD4/LVLC_I24=0.000
         * MOD4/LVHC_V1=7.500 MOD4/LVHC_I1=0.264 MOD4/LVHC_V2=-7.501
         * MOD4/LVHC_I2=-0.312 MOD4/LVHC_V3=-0.003 MOD4/LVHC_I3=-0.034
         * MOD4/LVHC_V4=7.501 MOD4/LVHC_I4=0.261 MOD4/LVHC_V5=-7.501
         * MOD4/LVHC_I5=-0.283 MOD4/LVHC_V6=-0.001 MOD4/LVHC_I6=0.001
         */
    }

    public static class HVBiasStatus extends BoardStatus {
        public float[] hvlc_V = new float[24];
        public float[] hvlc_I = new float[24];
        public float[] hvhc_V = new float[6];
        public float[] hvhc_I = new float[6];

        private static final long serialVersionUID = 5596477496468189559L;

        public HVBiasStatus(int id, Map<String, String> map) {
            super(id, map);
            for (int i = 0; i < 24; i++) {
                hvlc_V[i] = pfm(map, id, "HVLC_V" + (i + 1));
                hvlc_I[i] = pfm(map, id, "HVLC_I" + (i + 1));
            }
            for (int i = 0; i < 6; i++) {
                hvhc_V[i] = pfm(map, id, "HVHC_V" + (i + 1));
                hvhc_I[i] = pfm(map, id, "HVHC_I" + (i + 1));
            }
        }

        @Override
        public void dump() {
            super.dump();
            for (int i = 0; i < 24; i++) {
                System.out.printf("    HVLC V%d = %f  I%d = %f\n", i,
                        hvlc_V[i], i, hvlc_I[i]);
            }
            for (int i = 0; i < 6; i++) {
                System.out.printf("    HVHC V%d = %f  I%d = %f\n", i,
                        hvhc_V[i], i, hvhc_I[i]);
            }

        }

        /*
         * MOD9/HVLC_V1=29.501 MOD9/HVLC_I1=1.045 MOD9/HVLC_V2=29.500
         * MOD9/HVLC_I2=1.046 MOD9/HVLC_V3=29.501 MOD9/HVLC_I3=1.046
         * MOD9/HVLC_V4=29.500 MOD9/HVLC_I4=1.051 MOD9/HVLC_V5=29.500
         * MOD9/HVLC_I5=1.054 MOD9/HVLC_V6=29.501 MOD9/HVLC_I6=1.050
         * MOD9/HVLC_V7=29.501 MOD9/HVLC_I7=1.045 MOD9/HVLC_V8=29.501
         * MOD9/HVLC_I8=1.042 MOD9/HVLC_V9=29.502 MOD9/HVLC_I9=1.047
         * MOD9/HVLC_V10=29.501 MOD9/HVLC_I10=1.052 MOD9/HVLC_V11=29.501
         * MOD9/HVLC_I11=1.045 MOD9/HVLC_V12=29.501 MOD9/HVLC_I12=1.047
         * MOD9/HVLC_V13=0.001 MOD9/HVLC_I13=0.000 MOD9/HVLC_V14=0.001
         * MOD9/HVLC_I14=0.000 MOD9/HVLC_V15=0.000 MOD9/HVLC_I15=0.000
         * MOD9/HVLC_V16=0.000 MOD9/HVLC_I16=0.000 MOD9/HVLC_V17=0.000
         * MOD9/HVLC_I17=0.000 MOD9/HVLC_V18=0.000 MOD9/HVLC_I18=0.001
         * MOD9/HVLC_V19=26.001 MOD9/HVLC_I19=0.917 MOD9/HVLC_V20=18.001
         * MOD9/HVLC_I20=0.635 MOD9/HVLC_V21=29.500 MOD9/HVLC_I21=1.044
         * MOD9/HVLC_V22=29.500 MOD9/HVLC_I22=1.053 MOD9/HVLC_V23=29.500
         * MOD9/HVLC_I23=1.045 MOD9/HVLC_V24=29.501 MOD9/HVLC_I24=1.049
         * MOD9/HVHC_V1=0.023 MOD9/HVHC_I1=0.001 MOD9/HVHC_V2=1.537
         * MOD9/HVHC_I2=0.041 MOD9/HVHC_V3=0.001 MOD9/HVHC_I3=0.000
         * MOD9/HVHC_V4=1.284 MOD9/HVHC_I4=0.029 MOD9/HVHC_V5=0.001
         * MOD9/HVHC_I5=0.001 MOD9/HVHC_V6=0.001 MOD9/HVHC_I6=0.005
         */
    }

    public static class DriverStatus extends BoardStatus {

        private static final long serialVersionUID = 4253020001968479029L;

        public DriverStatus(int id, Map<String, String> map) {
            super(id, map);
        }

    }

    public static class ADStatus extends BoardStatus {

        private static final long serialVersionUID = -6862988098182047914L;

        // no extra info
        public ADStatus(int id, Map<String, String> map) {
            super(id, map);
        }
    }

}
