package org.lsst.ccs.drivers.archon;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.Socket;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.bootstrap.resources.BootstrapResourceUtils;
import static org.lsst.ccs.drivers.archon.ArchonStatus.pi;
import static org.lsst.ccs.drivers.archon.ArchonStatus.plx;
import org.lsst.ccs.drivers.archon.RawImageData.Mode;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.logging.Logger;

public class ArchonControllerDriver implements ArchonController {

    private String ip = "10.0.0.2";
    private int port = 4242;

    private static Logger log = Logger.getLogger("org.lsst.ccs.drivers.archon");

    public ArchonControllerDriver() throws DriverException {
        init();
    }

    public ArchonControllerDriver(String ip) throws DriverException {
        this.ip = ip;
        init();
    }

    public ArchonControllerDriver(String ip, int port) throws DriverException {
        this.ip = ip;
        this.port = port;
        init();
    }

    private void init() throws DriverException {
        openComm();
        readConfig();
        readStatus();
    }

    private Socket sock;
    private BufferedOutputStream out;
    private InputStream in;

    // we have to read either ISO-LATIN-1 strings, or binary data from the
    // input.
    // but we alternate and we can flush the buffers between one mode or the
    // other
    // so we can delegate to java.io and create both a BufferedReader and a
    // DataInputStream
    // on top of the same SocketInputStream without them fighting for input.

    private DataInputStream inBin;
    private BufferedReader inString;

    private void openComm() throws DriverException {
        try {
            sock = new Socket(ip, port);
            sock.setSoTimeout(5000); // 5 second timeout on all I/O
            in = sock.getInputStream();
            BufferedInputStream bis = new BufferedInputStream(in); // common
                                                                   // buffer
            inBin = new DataInputStream(bis);
            inString = new BufferedReader(new InputStreamReader(bis, csLatin1));

            out = new BufferedOutputStream(sock.getOutputStream());
        } catch (IOException e) {
            in = null;
            inBin = null;
            inString = null;
            out = null;
            sock = null;
            throw new DriverException(
                    "error opening socket to Archon controller", e);
        }
    }

    private void closeComm() throws DriverException {
        try {
            if (in != null)
                in.close();
            if (out != null)
                out.close();
        } catch (IOException e) {
            throw new DriverException(
                    "error closing streams to Archon controller", e);
        } finally {
            try {
                sock.close();
            } catch (IOException e) {
                throw new DriverException(
                        "error closing socket to Archon controller", e);
            } finally {
                in = null;
                inBin = null;
                inString = null;
                out = null;
                sock = null;
            }
        }
    }

    private int cmdId = 0;
    private final Charset csLatin1 = Charset.forName("ISO-8859-1");

    private String sendCommand(String cmd) throws DriverException {

        cmdId++;
        if (cmdId >= 256)
            cmdId = 1;

        if (sock == null)
            openComm();

        String cmdLine = String.format(">%02X%s\r", cmdId, cmd);
        log.debug(cmdLine);
        DataInputStream din = new DataInputStream(new BufferedInputStream(in));

        try {
            out.write(cmdLine.getBytes(csLatin1));
            out.flush();

            while (true) {
                String rep = inString.readLine();
                if (rep.startsWith("<")) {
                    int seq = Integer.parseInt(rep.substring(1, 3), 16);
                    if (seq < cmdId) {
                        log.warn("out of sequence answer, got " + seq
                                + " expecting " + cmdId);
                        continue;
                    }
                    if (seq > cmdId) {
                        log.warn("out of sequence answer, got " + seq
                                + " expecting " + cmdId);
                        throw new DriverException(
                                "out of sequence answer, got " + seq
                                        + " expecting " + cmdId);
                    }
                    return rep.substring(3);
                } else if (rep.startsWith("?")) {
                    log.error("Archon could not parse " + rep);
                    throw new DriverException("Archon could not parse " + rep);
                } else {
                    log.warn("Archon invalid answer "
                            + (rep.length() > 5 ? rep.substring(0, 5) : rep)
                            + "...");// retry and fail
                    // if timeout
                }

            }
        } catch (IOException e) {
            throw new DriverException(
                    "Exception sending command to Archon controller", e);
        }
    }

    public byte[] sendBinCommand(String cmd, int nBlocks)
            throws DriverException {
        byte[] rep = new byte[nBlocks * 1024];
        sendBinCommand(cmd, nBlocks, rep);
        return rep;
    }

    // blocks are 1024-byte
    public void sendBinCommand(String cmd, int nBlocks, byte[] rep)
            throws DriverException {
        cmdId++;
        if (cmdId >= 256)
            cmdId = 1;

        if (sock == null)
            openComm();

        String cmdLine = String.format(">%02X%s\r", cmdId, cmd);
        log.info(cmdLine);

        int repsz = rep.length;
        byte[] buf = new byte[1028];
        int i = 0;

        log.info("expecting " + rep.length + " in " + nBlocks + " blocks");

        try {
            out.write(cmdLine.getBytes(csLatin1));
            out.flush();

            while (true) {
                // log.info("reading block " + i);
                inBin.readFully(buf);
                if (buf[0] == '<') {
                    StringBuilder sb = new StringBuilder();
                    sb.append((char) buf[1]);
                    sb.append((char) buf[2]);
                    int seq = Integer.parseInt(sb.toString(), 16);

                    if (seq < cmdId) {
                        log.warn("out of sequence answer, got " + seq + "("
                                + (char) buf[1] + ":" + (char) buf[2] + ")"
                                + " expecting " + cmdId);
                        continue;
                    }
                    if (seq > cmdId) {
                        log.warn("out of sequence answer, got " + seq + "("
                                + (char) buf[1] + ":" + (char) buf[2] + ")"
                                + " expecting " + cmdId);
                        throw new DriverException(
                                "out of sequence answer, got " + seq
                                        + " expecting " + cmdId);
                    }

                    if (buf[3] != ':') {
                        log.warn("Archon invalid answer " + (char) buf[0]
                                + (char) buf[1] + (char) buf[2] + (char) buf[3]
                                + (char) buf[4] + "...");
                        continue;
                    }

                    int len = 1024;
                    if (i * 1024 + len > repsz)
                        len = repsz - i * 1024;

                    System.arraycopy(buf, 4, rep, i * 1024, len);
                    i++;
                    if (i == nBlocks)
                        return;

                } else if (buf[0] == '?') {
                    log.error("Archon could not parse command "
                            + new String(buf));
                    throw new DriverException("Archon could not parse command "
                            + new String(buf));
                } else {
                    log.warn("Archon invalid answer " + (char) buf[0]
                            + (char) buf[1] + (char) buf[2] + (char) buf[3]
                            + (char) buf[4] + "...");// retry

                    // log.warn("in reader we have " + (char) inString.read());
                    // and
                    // fail
                    // if timeout
                }

            }
        } catch (IOException e) {
            throw new DriverException(
                    "Exception sending command to Archon controller", e);
        }

    }

    protected Map<String, String> parseKeys(String s) {
        Map<String, String> map = new HashMap<>();
        for (String kv : s.split(" +")) {
            String[] kvv = kv.split("=");
            map.put(kvv[0], kvv[1]);
        }

        return map;
    }

    private ArchonConfig config;

    public void readConfig() throws DriverException {
        Map<String, String> m = parseKeys(sendCommand("SYSTEM"));
        config = new ArchonConfig(m);
    }

    private ArchonStatus status;

    public void readStatus() throws DriverException {
        Map<String, String> m = parseKeys(sendCommand("STATUS"));
        if (config == null)
            readConfig();
        status = new ArchonStatus(config, m);
    }

    @Override
    public ArchonStatus getStatus() throws DriverException {
        readStatus();
        return status;
    }

    public static class FrameStatus {
        long timer;
        int rBuf;
        int wBuf;
        int[] bufSample = new int[3];
        int[] bufComplete = new int[3];
        int[] bufMode = new int[3];
        int[] bufFrame = new int[3];
        int[] bufWidth = new int[3];
        int[] bufHeight = new int[3];
        int[] bufPixels = new int[3];
        int[] bufLines = new int[3];
        int[] bufRawBlocks = new int[3];
        int[] bufRawLines = new int[3];
        int[] bufRawOffset = new int[3];
        long[] butTimeStamp = new long[3];

        public void dump() {
            System.out.printf("timer %8X  rbuf %d   wbuf %d\n", timer, rBuf,
                    wBuf);
            for (int i = 0; i < 3; i++) {
                System.out.printf("Buffer %d\n", i + 1);
                System.out
                        .printf("  Complete %d  Mode %d  Frame %d  Width %d Height %d\n",
                                bufComplete[i], bufMode[i], bufFrame[i],
                                bufWidth[i], bufHeight[i]);
                System.out.printf("  Pixels %d  Lines %d \n", bufPixels[i],
                        bufLines[i]);

            }
        }
    }

    public FrameStatus getFrameStatus() throws DriverException {
        FrameStatus f = new FrameStatus();
        Map<String, String> m = parseKeys(sendCommand("FRAME"));

        f.timer = plx(m, "TIMER");
        f.rBuf = pi(m, "RBUF");
        f.wBuf = pi(m, "WBUF");

        for (int i = 0; i < 3; i++) {
            f.bufSample[i] = pi(m, "BUF" + (i + 1) + "SAMPLE");
            f.bufComplete[i] = pi(m, "BUF" + (i + 1) + "COMPLETE");
            f.bufMode[i] = pi(m, "BUF" + (i + 1) + "MODE");
            f.bufFrame[i] = pi(m, "BUF" + (i + 1) + "FRAME");
            f.bufWidth[i] = pi(m, "BUF" + (i + 1) + "WIDTH");
            f.bufHeight[i] = pi(m, "BUF" + (i + 1) + "HEIGHT");
            f.bufPixels[i] = pi(m, "BUF" + (i + 1) + "PIXELS");
            f.bufLines[i] = pi(m, "BUF" + (i + 1) + "LINES");
            f.bufRawBlocks[i] = pi(m, "BUF" + (i + 1) + "RAWBLOCKS");
            f.bufRawLines[i] = pi(m, "BUF" + (i + 1) + "RAWLINES");
            f.bufRawOffset[i] = pi(m, "BUF" + (i + 1) + "RAWOFFSET");
            f.butTimeStamp[i] = plx(m, "BUF" + (i + 1) + "TIMESTAMP");
        }

        return f;
    }

    public String fetchLog() throws DriverException {
        return sendCommand("FETCHLOG");
    }

    // n : 1-3
    public void lock(int n) throws DriverException {
        sendCommand("LOCK" + n);
    }

    // Firmware methods not implemented
    // VERIFYMODxxyyyyzzzz
    // ERASEMODxx
    // FLASHMODxxyyyyzzz...zzz
    // ERASExxxxxxxxyyyyyyyy
    // FLASHxxxxyyy...yyy
    // VERIFYxxxxyyyy

    public void reboot() throws DriverException {
        sendCommand("REBOOT");
    }

    // 1024-byte blocks : return size is nBlockx * 1024 bytes
    public byte[] fetch(int startAddr, int nBlocks) throws DriverException {
        String cmd = String.format("FETCH%08X%08X", startAddr, nBlocks);
        return sendBinCommand(cmd, nBlocks);
    }

    public void fetch(int startAddr, byte[] rep) throws DriverException {
        int nBlocks = (rep.length + 1023) / 1024;
        String cmd = String.format("FETCH%08X%08X", startAddr, nBlocks);
        sendBinCommand(cmd, nBlocks, rep);
    }

    // i : 0-2047
    public String readConfigLine(int i) throws DriverException {
        if (i < 0 || i > 2047)
            throw new ArrayIndexOutOfBoundsException(
                    "config line out of range " + i);
        return sendCommand(String.format("RCONFIG%04X", i));
    }

    public void writeConfigLine(int i, String line) throws DriverException {
        if (i < 0 || i > 2047)
            throw new ArrayIndexOutOfBoundsException(
                    "config line out of range " + i);
        if (line.length() > 2048)
            throw new IllegalArgumentException("config line too long "
                    + line.length());

        sendCommand(String.format("WCONFIG%04X%s", i, line));

    }

    public void clearConfig() throws DriverException {
        sendCommand("CLEARCONFIG");
    }

    public void pollOn() throws DriverException {
        sendCommand("POLLON");
    }

    // magical undocumented command that speeds up RCONFIG/WCONFIG **a lot**
    // no clue to what happens if you don't pollOn() back after...
    public void pollOff() throws DriverException {
        sendCommand("POLLOFF");
    }

    public String[] readConfigLines() throws DriverException {
        pollOff();
        List<String> l = new ArrayList<>(2048);
        for (int i = 0; i < 2048; i++) {
            String line = readConfigLine(i);
            if (line.equals(""))
                break;
            l.add(line);
            if (i % 100 == 0) {
                log.info("reading config : " + i + " ...");
            }
        }
        log.info("done reading config");
        pollOn();
        return l.toArray(new String[0]);
    }

    @Override
    public void writeConfigLines(String[] lines) throws DriverException {
        pollOff();
        clearConfig();
        for (int i = 0; i < lines.length; i++) {
            writeConfigLine(i, lines[i]);
            if (i % 100 == 0) {
                log.info("writing config : " + i + " ...");
            }

        }
        pollOn();
        log.info("done writing config");
    }

    @Override
    public void applyAll() throws DriverException {
        sendCommand("APPLYALL");
    }

    @Override
    public void powerOn() throws DriverException {
        sendCommand("POWERON");
    }

    @Override
    public void powerOff() throws DriverException {
        sendCommand("POWERFF");
    }

    public void loadTiming() throws DriverException {
        sendCommand("LOADTIMING");
    }

    public void loadParams() throws DriverException {
        sendCommand("LOADPARAMS");
    }

    public void loadParam(String p) throws DriverException {
        sendCommand("LOADPARAM " + p);
    }

    public void resetTiming() throws DriverException {
        sendCommand("RESETTIMING");
    }

    public void applyMod(int module) throws DriverException {
        sendCommand(String.format("APPLYMOD%02X", module));
    }

    int lastFrame = 0;


    public RawImageData fetchLastImage() throws DriverException {
        // find a frame that is complete, the most recent, and is more recent
        // than the last one
        // returns null if no new image since last acq
        FrameStatus fs = getFrameStatus();
        int newestFrame = lastFrame;
        int newestBuf = -1;
        for (int i = 0; i < 3; i++) {
            if (fs.bufFrame[i] > newestFrame && fs.bufComplete[i] != 0) {
                newestFrame = fs.bufFrame[i];
                newestBuf = i;
            }
        }
        if (newestBuf < 0)
            return null;

        lock(newestBuf + 1);

        return fetchCurrentLockedFrame();
    }

    @Override
    public RawImageData fetchNewImage(long timeout) throws DriverException,
            TimeoutException {
        // wait for a new frame that is complete
        long start = System.currentTimeMillis();
        while (System.currentTimeMillis() < start + timeout) {
            RawImageData data = fetchLastImage();
            if (data != null)
                return data;
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {
                // nothing
            }
        }

        throw new TimeoutException("timeout waiting for new image");
    }

    public RawImageData fetchCurrentLockedFrame() throws DriverException {
        FrameStatus fs = getFrameStatus();
        int buf = fs.rBuf - 1;
        if (buf < 0 || buf > 2) {
            log.error("invalid RBUF " + buf);
            return null;
        }

        log.info("reading buffer " + buf + " frame " + fs.bufFrame[buf]);

        int w = fs.bufWidth[buf];
        int h = fs.bufHeight[buf];
        int mode = fs.bufSample[buf];

        int sz = (mode == 1) ? 4 * w * h : 2 * w * h;

        // should be keep buffers or allocate each time?
        byte[] data = new byte[sz];

        // according to the documentation (copy pasted below)
        // The starting address is 0x20000000 for frame buffer 1, 0x40000000 for
        // frame buffer 2, and 0x60000000 for frame buffer 3

        // according to the code, buffers start at (rbuf | 4) << 29, ie
        // 0xA0000000 0xC0000000 0xE0000000
        // and this works better.

        fetch((buf + 5) << 29, data);
        // fetch(0x20000000 * (buf + 1), data);

        lastFrame = fs.bufFrame[buf];

        RawImageData imData = new RawImageData();
        imData.width = w;
        imData.height = h;
        imData.mode = Mode.values()[mode];
        imData.data = data;

        return imData;
    }

    public static String[] getConfigFromACF(BufferedReader rdr)
            throws IOException {
        List<String> l = new ArrayList<>(1024);
        boolean inConfig = false;
        while (true) {
            String line = rdr.readLine();
            if (line == null)
                break;
            if (line.startsWith("[CONFIG]")) {
                inConfig = true;
                continue;
            }
            if (!inConfig)
                continue;
            line = line.replaceAll("\"", "");
            line = line.replaceAll("\\\\", "/");
            System.out.println(line);
            l.add(line);
        }
        log.info("read " + l.size() + " config lines");
        return l.toArray(new String[0]);
    }

    public static void main(String[] args) throws DriverException, IOException {
        Locale.setDefault(Locale.US);
        ArchonControllerDriver ctl;

        try {
            ctl = new ArchonControllerDriver("127.0.0.1"); // tunneling
        } catch (DriverException e) {
            ctl = new ArchonControllerDriver("10.0.0.2"); // direct
        }

        System.out.println("\n\n**** CONFIGURATION\n");
        ctl.config.dump();

        System.out.println("\n\n**** STATUS\n");
        ctl.status.dump();

        FrameStatus f = ctl.getFrameStatus();
        System.out.printf("\n\nTimer: %x\nread buffer: %d\nwrite buffer: %d\n",
                f.timer, f.rBuf, f.wBuf);

        System.out.println("\n\n**** TEST READING TWO 1024-BYTE BLOCKS\n");
        byte[] data = ctl.sendBinCommand("FETCH2000000000000002", 2); // 2
        // blocks
        // from FB
        // 1
        System.out.println(data.length);

        data = ctl.fetch(0x20000000, 2);
        System.out.println(data.length);
        System.out.println();

        InputStream is = BootstrapResourceUtils.getBootstrapResource("archon.acf");
        if (is != null) {
            log.info("reading config file");
            BufferedReader rdr = new BufferedReader(new InputStreamReader(is));
            String[] lines = getConfigFromACF(rdr);
            ctl.writeConfigLines(lines);

            ctl.applyAll();
            ctl.powerOn();
        }

        System.out.println(ctl.readConfigLine(0));

        // String[] lines = ctl.readConfigLines();
        // System.out.println("Config lines : " + lines.length + "\n");
        // for (String l : lines) {
        // System.out.println(l);
        // }

        FrameStatus fs = ctl.getFrameStatus();
        fs.dump();

        try {
            Thread.sleep(3000);
        } catch (InterruptedException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        while (true) {
            fs = ctl.getFrameStatus();
            fs.dump();

            // let's write if back
            // ctl.writeConfigLines(lines);

            RawImageData img = ctl.fetchLastImage();
            if (img == null) {
                System.out.println("no last image");
            } else {
                System.out.println("last image size " + img.data.length);
                DataOutputStream os = new DataOutputStream(
                        new FileOutputStream("archonimg-" + ctl.lastFrame
                                + ".raw"));
                os.write(img.data, 0, img.data.length);
                os.close();
            }
            try {
                Thread.sleep(300);
            } catch (InterruptedException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }

    }

}
