package org.lsst.ccs.drivers.apcpdu;

import java.util.Map;
import org.lsst.ccs.drivers.ascii.Session;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Driver for controlling an APC model 7900 power strip
 *
 *  @author Owen Saxton
 */
public interface APC7900Series {

    /**
     *  Constants.
     */
    public static final int
        DEFAULT_BAUDRATE = 9600,
        LOG_TIMEOUT = 200,
        RUN_TIMEOUT = 50;
    public static final String
        STATUS_SPLIT_RE = " *: ";


    /**
     *  Opens a connection and logs in.
     *
     *  @param  connType  The enumerated connection type: TELNET or SERIAL
     *  @param  ident     The host name (TELNET) or port name (SERIAL)
     *  @param  username  The user name
     *  @param  password  The password
     *  @throws  DriverException
     */
    public void open(Session.ConnType connType, String ident, String username, String password)
         throws DriverException;


    /**
     *  Opens a connection.
     *
     *  @param  connType  The enumerated connection type: TELNET or SERIAL
     *  @param  ident     The host name (TELNET) or port name (SERIAL)
     *  @throws  DriverException
     */
    public void open(Session.ConnType connType, String ident) throws DriverException;


    /**
     *  Closes the connection
     *
     *  @throws  DriverException
     */
    public void close() throws DriverException;


    /**
     *  Tests whether the connection is open.
     *
     *  Implemented by Session
     *
     *  @return  Whether the connection is open
     */
    public boolean isOpen();


    /**
     *  Sets the debugging state.
     *
     *  Implemented by Session
     *
     *  @param  on  Whether debugging is to be enabled
     */
    public void setDebug(boolean on);


    /**
     *  Gets the prompt.
     *
     *  Implemented by Session
     *
     *  @return  The saved prompt
     */
    public String getPrompt();


    /**
     *  Sends a command and receives the response.
     *
     *  This is the raw response from Session
     *
     *  @param  command  The command to send
     *  @return  An array of strings each of which is a line of the response.
     *           The first element is the echoed command.
     *  @throws  DriverException
     */
    public String[] receive(String command) throws DriverException;


    /**
     *  Sends a command and receives a string array.
     *
     *  This is the response after processing by APC7900x
     *
     *  @param  command  The command to send
     *  @return  The string array
     *  @throws  DriverException
     */
    public String[] receiveString(String command) throws DriverException;


    /**
     *  Gets the product name
     *
     *  @return  The product name
     *  @throws  DriverException
     */
    public String getProductName() throws DriverException;


    /**
     *  Gets the firmware version
     *
     *  @return  The firmware version
     *  @throws  DriverException
     */
    public String getFWVersion() throws DriverException;


    /**
     *  Gets the number of phases
     *
     *  @return  The number of phases
     *  @throws  DriverException
     */
    public int getPhaseCount() throws DriverException;


    /**
     *  Gets the number of outlets
     *
     *  @return  The number of outlets
     *  @throws  DriverException
     */
    public int getOutletCount() throws DriverException;


    /**
     *  Gets the name of an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The outlet name
     *  @throws  DriverException
     */
    public String getOutletName(int outlet) throws DriverException;


    /**
     *  Sets the name of an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @param  name    The outlet name
     *  @throws  DriverException
     */
    public void setOutletName(int outlet, String name) throws DriverException;


    /**
     *  Gets an outlet number, given its name
     *
     *  @param  outlet  The outlet name
     *  @return  The outlet number
     *  @throws  DriverException
     */
    public int getOutletNumber(String outlet) throws DriverException;


    /**
     *  Gets the map of outlet names to numbers
     *
     *  @return  The outlet number map
     *  @throws  DriverException
     */
    public Map<String, Integer> getOutletNumberMap() throws DriverException;


    /**
     *  Gets the on delay for an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    public int getOutletOnDelay(int outlet) throws DriverException;


    /**
     *  Gets the off delay for an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    public int getOutletOffDelay(int outlet) throws DriverException;


    /**
     *  Gets the on delay for a named outlet
     *
     *  @param  outlet  The outlet name
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    public int getOutletOnDelay(String outlet) throws DriverException;


    /**
     *  Gets the off delay for a named outlet
     *
     *  @param  outlet  The outlet name
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    public int getOutletOffDelay(String outlet) throws DriverException;


    /**
     *  Gets the on delay for all outlets
     *
     *  @return  An array containing the delays for all outlets
     *  @throws  DriverException
     */
    public int[] getOutletOnDelays() throws DriverException;


    /**
     *  Gets the off delay for all outlets
     *
     *  @return  An array containing the delays for all outlets
     *  @throws  DriverException
     */
    public int[] getOutletOffDelays() throws DriverException;


    /**
     *  Gets the map of outlet names to on delays
     *
     *  @return  The delay map
     *  @throws  DriverException
     */
    public Map<String, Integer> getOutletOnDelayMap() throws DriverException;


    /**
     *  Gets the map of outlet names to off delays
     *
     *  @return  The delay map
     *  @throws  DriverException
     */
    public Map<String, Integer> getOutletOffDelayMap() throws DriverException;


    /**
     *  Sets outlet on delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlet  The outlet number, starting from 1
     *  @throws  DriverException
     */
    public void setOutletOnDelay(int delay, int outlet) throws DriverException;


    /**
     *  Sets outlet off delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlet  The outlet number, starting from 1
     *  @throws  DriverException
     */
    public void setOutletOffDelay(int delay, int outlet) throws DriverException;
    /**
     *  Sets named outlet on delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlet  The outlet name
     *  @throws  DriverException
     */
    public void setOutletOnDelay(int delay, String outlet) throws DriverException;


    /**
     *  Sets named outlet off delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlet  The outlet name
     *  @throws  DriverException
     */
    public void setOutletOffDelay(int delay, String outlet) throws DriverException;


    /**
     *  Sets outlets on delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlets  The array of outlet numbers
     *  @throws  DriverException
     */
    public void setOutletOnDelay(int delay, int[] outlets) throws DriverException;


    /**
     *  Sets outlets off delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlets  The array of outlet numbers
     *  @throws  DriverException
     */
    public void setOutletOffDelay(int delay, int[] outlets) throws DriverException;


    /**
     *  Sets named outlets on delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlets  The array of outlet names
     *  @throws  DriverException
     */
    public void setOutletOnDelay(int delay, String[] outlets) throws DriverException;


    /**
     *  Sets named outlets off delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlets  The array of outlet names
     *  @throws  DriverException
     */
    public void setOutletOffDelay(int delay, String[] outlets) throws DriverException;


    /**
     *  Gets the on (powered) state of an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  Whether the outlet is powered
     *  @throws  DriverException
     */
    public boolean isOutletOn(int outlet) throws DriverException;


    /**
     *  Gets the on (powered) state of an outlet
     *
     *  @param  outlet  The outlet name
     *  @return  Whether the outlet is powered
     *  @throws  DriverException
     */
    public boolean isOutletOn(String outlet) throws DriverException;


    /**
     *  Gets the on (powered) state of all outlets
     *
     *  @return  A boolean array containing the on state of all outlets
     *  @throws  DriverException
     */
    public boolean[] getOutletOnStates() throws DriverException;


    /**
     *  Gets the map of outlet names to on (powered) states
     *
     *  @return  The on state map
     *  @throws  DriverException
     */
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException;


    /**
     *  Turns outlet power on immediately.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned on (1)
     *  @throws  DriverException
     */
    public int setOutletOn(int outlet) throws DriverException;


    /**
     *  Turns outlet power off immediately.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    public int setOutletOff(int outlet) throws DriverException;


    /**
     *  Turns named outlet power on immediately.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned on (1)
     *  @throws  DriverException
     */
    public int setOutletOn(String outlet) throws DriverException;


    /**
     *  Turns named outlet power off immediately.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    public int setOutletOff(String outlet) throws DriverException;


    /**
     *  Turns outlets power on immediately.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    public int setOutletsOn(int[] outlets) throws DriverException;


    /**
     *  Turns outlets power off immediately.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    public int setOutletsOff(int[] outlets) throws DriverException;


    /**
     *  Turns named outlets power on immediately.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    public int setOutletsOn(String[] outlets) throws DriverException;


    /**
     *  Turns named outlets power off immediately.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    public int setOutletsOff(String[] outlets) throws DriverException;


    /**
     *  Turns outlet power on with delay.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned on (1)
     *  @throws  DriverException
     */
    public int delayedOutletOn(int outlet) throws DriverException;


    /**
     *  Turns outlet power off with delay.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    public int delayedOutletOff(int outlet) throws DriverException;


    /**
     *  Turns named outlet power on with delay.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned on (1)
     *  @throws  DriverException
     */
    public int delayedOutletOn(String outlet) throws DriverException;


    /**
     *  Turns named outlet power off with delay.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    public int delayedOutletOff(String outlet) throws DriverException;


    /**
     *  Turns outlets power on with delay.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    public int delayedOutletsOn(int[] outlets) throws DriverException;


    /**
     *  Turns outlets power off with delay.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    public int delayedOutletsOff(int[] outlets) throws DriverException;


    /**
     *  Turns named outlets power on with delay.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    public int delayedOutletsOn(String[] outlets) throws DriverException;


    /**
     *  Turns named outlets power off with delay.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    public int delayedOutletsOff(String[] outlets) throws DriverException;


    /**
     *  Reads input currents, one per phase
     * 
     *  @return  The array of input currents (amps)
     *  @throws  DriverException
     */
    public double[] readCurrent() throws DriverException;


    /**
     *  Reads input VA
     * 
     *  @return  NaN if method not implemented
     *  @throws  DriverException
     */
    public default double readVA() throws DriverException {
        return Double.NaN;
    }


    /**
     *  Reads input power
     * 
     *  @return  NaN if method not implemented
     *  @throws  DriverException
     */
    public default double readPower() throws DriverException {
        return Double.NaN;
    }


    /**
     *  Reads input VA and power
     * 
     *  @return  Two-element array of NaNs if method not implemented
     *  @throws  DriverException
     */
    public default double[] readPowers() throws DriverException {
        return new double[]{Double.NaN, Double.NaN};
    }

}
