package org.lsst.ccs.drivers.apcpdu;

import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.ascii.Session;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Driver for controlling an APC model 7900B power strip
 *
 *  @author Owen Saxton
 */
public class APC7900B extends Session implements APC7900Series {

    /**
     *  Private data.
     */
    private String product, fwVersion;
    private Integer numPhases, numOutlets;


    /**
     *  Constructor
     */
    public APC7900B()
    {
        super(OPTN_KEEP_ALIVE, "apc>", "User Name :", "Password  :", "exit", Ascii.Terminator.CR);
    }


    /**
     *  Opens a connection and logs in.
     *
     *  @param  connType  The enumerated connection type: TELNET or SERIAL
     *  @param  ident     The host name (TELNET) or port name (SERIAL)
     *  @param  username  The user name
     *  @param  password  The password
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident, String username, String password) throws DriverException
    {
        open(connType, ident, DEFAULT_BAUDRATE, username, password, LOG_TIMEOUT, RUN_TIMEOUT);
    }


    /**
     *  Opens a connection.
     *
     *  @param  connType  The enumerated connection type: TELNET or SERIAL
     *  @param  ident     The host name (TELNET) or port name (SERIAL)
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident) throws DriverException
    {
        open(connType, ident, DEFAULT_BAUDRATE, RUN_TIMEOUT);
    }


    /**
     *  Closes the connection
     *
     *  @throws  DriverException
     */
    @Override
    public void close() throws DriverException
    {
        try {
            super.close();
        }
        finally {
            product = null;
            fwVersion = null;
            numPhases = null;
            numOutlets = null;
        }
    }


    /**
     *  Gets the product name
     *
     *  @return  The product name
     *  @throws  DriverException
     */
    @Override
    public String getProductName() throws DriverException
    {
        if (product == null) {
            getVersionInfo();
        }
        return product;
    }


    /**
     *  Gets the firmware version
     *
     *  @return  The firmware version
     *  @throws  DriverException
     */
    @Override
    public String getFWVersion() throws DriverException
    {
        if (fwVersion == null) {
            getVersionInfo();
        }
        return fwVersion;
    }


    /**
     *  Gets the number of phases
     *
     *  @return  The number of phases
     *  @throws  DriverException
     */
    @Override
    public int getPhaseCount() throws DriverException
    {
        if (numPhases == null) {
            getVersionInfo();
        }
        return numPhases;
    }


    /**
     *  Gets the number of outlets
     *
     *  @return  The number of outlets
     *  @throws  DriverException
     */
    @Override
    public int getOutletCount() throws DriverException
    {
        if (numOutlets == null) {
            getVersionInfo();
        }
        return numOutlets;
    }


    /**
     *  Gets the name of an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The outlet name
     *  @throws  DriverException
     */
    @Override
    public String getOutletName(int outlet) throws DriverException
    {
        String[] reply = receiveString("olName " + outlet);
        checkReplyLength(reply, 2);
        String[] words = reply[0].split(STATUS_SPLIT_RE);
        return words[1];
    }


    /**
     *  Sets the name of an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @param  name    The outlet name
     *  @throws  DriverException
     */
    @Override
    public void setOutletName(int outlet, String name) throws DriverException
    {
        String[] reply = receiveString("olName " + outlet + " \"" + name + "\"");
    }


    /**
     *  Gets an outlet number, given its name
     *
     *  @param  outlet  The outlet name
     *  @return  The outlet number
     *  @throws  DriverException
     */
    @Override
    public int getOutletNumber(String outlet) throws DriverException
    {
        String[] reply = receiveString("olStatus \"" + outlet + "\"");
        checkReplyLength(reply, 2);
        String[] words = reply[0].split(STATUS_SPLIT_RE);
        int number = 0;
        try {
            number = getInteger(words[0]);
        }
        catch (NumberFormatException e) {
            responseError(reply[0]);
        }
        return number;
    }


    /**
     *  Gets the map of outlet names to numbers
     *
     *  @return  The outlet number map
     *  @throws  DriverException
     */
    @Override
    public Map<String, Integer> getOutletNumberMap() throws DriverException
    {
        String[] reply = receiveString("olStatus all");
        checkReplyLength(reply, getOutletCount() + 1);
        Map<String, Integer> numberMap = new LinkedHashMap<>();
        for (int j = 0; j < reply.length - 1; j++) {
            String[] words = reply[j].split(STATUS_SPLIT_RE);
            try {
                numberMap.put(words.length >= 2 ? words[1] : "", getInteger(words[0]));
            }
            catch (NumberFormatException e) {
                responseError(reply[j]);
            }
        }
        return numberMap;
    }


    /**
     *  Gets the on delay for an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    @Override
    public int getOutletOnDelay(int outlet) throws DriverException
    {
        return extractDelay(receiveString("olOnDelay " + outlet));
    }


    /**
     *  Gets the off delay for an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    @Override
    public int getOutletOffDelay(int outlet) throws DriverException
    {
        return extractDelay(receiveString("olOffDelay " + outlet));
    }


    /**
     *  Gets the on delay for a named outlet
     *
     *  @param  outlet  The outlet name
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    @Override
    public int getOutletOnDelay(String outlet) throws DriverException
    {
        return extractDelay(receiveString("olOnDelay \"" + outlet + "\""));
    }


    /**
     *  Gets the off delay for a named outlet
     *
     *  @param  outlet  The outlet name
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    @Override
    public int getOutletOffDelay(String outlet) throws DriverException
    {
        return extractDelay(receiveString("olOffDelay \"" + outlet + "\""));
    }


    /**
     *  Extracts the delay value from a PDU reply
     *
     *  @param  reply  The reply
     *  @return  The delay (secs), or -1 for never
     *  @throws  DriverException
     */
    private int extractDelay(String[] reply) throws DriverException
    {
        checkReplyLength(reply, 2);
        String[] words = reply[0].split(STATUS_SPLIT_RE);
        String[] values = words[2].split(" ");   // "n sec"
        int delay = 0;
        try {
            delay = getInteger(values[0]);
        }
        catch (NumberFormatException | IndexOutOfBoundsException e) {
            responseError(reply[0]);
        }
        return delay;
    }


    /**
     *  Gets the on delay for all outlets
     *
     *  @return  An array containing the delays for all outlets
     *  @throws  DriverException
     */
    @Override
    public int[] getOutletOnDelays() throws DriverException
    {
        return getOutletDelays(true);
    }


    /**
     *  Gets the off delay for all outlets
     *
     *  @return  An array containing the delays for all outlets
     *  @throws  DriverException
     */
    @Override
    public int[] getOutletOffDelays() throws DriverException
    {
        return getOutletDelays(false);
    }


    /**
     *  Gets the on or off delays for all outlets
     *
     *  @param  on  True to get on delays, false to get off delays
     *  @return  The array of delays
     *  @throws  DriverException
     */
    private int[] getOutletDelays(boolean on) throws DriverException
    {
        String[] reply = receiveString(on ? "olOnDelay all" : "olOffDelay all");
        checkReplyLength(reply, getOutletCount() + 1);
        int[] delays = new int[reply.length - 1];
        for (int j = 0; j < reply.length - 1; j++) {
            String[] values = reply[j].split(STATUS_SPLIT_RE)[2].split(" ");   // "n sec"
            try {
                delays[j] = getInteger(values[0]);
            }
            catch (NumberFormatException | IndexOutOfBoundsException e) {
                responseError(reply[j]);
            }
        }
        return delays;
    }


    /**
     *  Gets the map of outlet names to on delays
     *
     *  @return  The delay map
     *  @throws  DriverException
     */
    @Override
    public Map<String, Integer> getOutletOnDelayMap() throws DriverException
    {
        return getOutletDelayMap(true);
    }


    /**
     *  Gets the map of outlet names to off delays
     *
     *  @return  The delay map
     *  @throws  DriverException
     */
    @Override
    public Map<String, Integer> getOutletOffDelayMap() throws DriverException
    {
        return getOutletDelayMap(false);
    }


    /**
     *  Gets the map of outlet names to on or off delays
     *
     *  @param  on  True to set on delay, false to set off delay
     *  @return  The delay map
     *  @throws  DriverException
     */
    private Map<String, Integer> getOutletDelayMap(boolean on) throws DriverException
    {
        String[] reply = receiveString(on ? "olOnDelay all" : "olOffDelay all");
        checkReplyLength(reply, getOutletCount() + 1);
        Map<String, Integer> delayMap = new LinkedHashMap<>();
        for (int j = 0; j < reply.length - 1; j++) {
            String[] words = reply[j].split(STATUS_SPLIT_RE);
            String[] values = words[2].split(" ");   // "n sec"
            int delay = 0;
            try {
                delay = getInteger(values[0]);
            }
            catch (NumberFormatException | IndexOutOfBoundsException e) {
                responseError(reply[j]);
            }
            delayMap.put(words[1], delay);
        }
        return delayMap;
    }


    /**
     *  Sets outlet on delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlet  The outlet number, starting from 1
     *  @throws  DriverException
     */
    @Override
    public void setOutletOnDelay(int delay, int outlet) throws DriverException
    {
        receiveString("olOnDelay " + outlet + " " + delay);
    }


    /**
     *  Sets outlet off delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlet  The outlet number, starting from 1
     *  @throws  DriverException
     */
    @Override
    public void setOutletOffDelay(int delay, int outlet) throws DriverException
    {
        receiveString("olOffDelay " + outlet + " " + delay);
    }


    /**
     *  Sets named outlet on delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlet  The outlet name
     *  @throws  DriverException
     */
    @Override
    public void setOutletOnDelay(int delay, String outlet) throws DriverException
    {
        receiveString("olOnDelay \"" + outlet + "\" " + delay);
    }


    /**
     *  Sets named outlet off delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlet  The outlet name
     *  @throws  DriverException
     */
    @Override
    public void setOutletOffDelay(int delay, String outlet) throws DriverException
    {
        receiveString("olOffDelay \"" + outlet + "\" " + delay);
    }


    /**
     *  Sets outlets on delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlets  The array of outlet numbers
     *  @throws  DriverException
     */
    @Override
    public void setOutletOnDelay(int delay, int[] outlets) throws DriverException
    {
        receiveString("olOnDelay" + makeOutletString(makeOutletSet(outlets)) + " " + delay);
    }


    /**
     *  Sets outlets off delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlets  The array of outlet numbers
     *  @throws  DriverException
     */
    @Override
    public void setOutletOffDelay(int delay, int[] outlets) throws DriverException
    {
        receiveString("olOffDelay" + makeOutletString(makeOutletSet(outlets)) + " " + delay);
    }


    /**
     *  Sets named outlets on delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlets  The array of outlet names
     *  @throws  DriverException
     */
    @Override
    public void setOutletOnDelay(int delay, String[] outlets) throws DriverException {
        receiveString("olOnDelay " + makeOutletString(makeOutletSet(outlets)) + " " + delay);
    }



    /**
     *  Sets named outlets off delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlets  The array of outlet names
     *  @throws  DriverException
     */
    @Override
    public void setOutletOffDelay(int delay, String[] outlets) throws DriverException {
        receiveString("olOffDelay " + makeOutletString(makeOutletSet(outlets)) + " " + delay);
    }



    /**
     *  Gets the on (powered) state of an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  Whether the outlet is powered
     *  @throws  DriverException
     */
    @Override
    public boolean isOutletOn(int outlet) throws DriverException
    {
        String[] reply = receiveString("olStatus " + outlet);
        checkReplyLength(reply, 2);
        return reply[0].split(STATUS_SPLIT_RE)[2].trim().equals("On");
    }


    /**
     *  Gets the on (powered) state of an outlet
     *
     *  @param  outlet  The outlet name
     *  @return  Whether the outlet is powered
     *  @throws  DriverException
     */
    @Override
    public boolean isOutletOn(String outlet) throws DriverException
    {
        String[] reply = receiveString("olStatus \"" + outlet + "\"");
        checkReplyLength(reply, 2);
        return reply[0].split(STATUS_SPLIT_RE)[2].trim().equals("On");
    }


    /**
     *  Gets the on (powered) state of all outlets
     *
     *  @return  A boolean array containing the on state of all outlets
     *  @throws  DriverException
     */
    @Override
    public boolean[] getOutletOnStates() throws DriverException
    {
        String[] reply = receiveString("olStatus all");
        boolean[] onStates = new boolean[getOutletCount()];
        checkReplyLength(reply, onStates.length + 1);
        for (int j = 0; j < onStates.length; j++) {
            onStates[j] = reply[j].split(STATUS_SPLIT_RE)[2].trim().equals("On");
        }
        return onStates;
    }


    /**
     *  Gets the map of outlet names to on (powered) states
     *
     *  @return  The on state map
     *  @throws  DriverException
     */
    @Override
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException
    {
        String[] reply = receiveString("olStatus all");
        checkReplyLength(reply, getOutletCount() + 1);
        Map<String, Boolean> stateMap = new LinkedHashMap<>();
        for (int j = 0; j < reply.length - 1; j++) {
            String[] words = reply[j].split(STATUS_SPLIT_RE);
            stateMap.put(words[1], words[2].trim().equals("On"));
        }
        return stateMap;
    }


    /**
     *  Turns outlet power on immediately.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned on (1)
     *  @throws  DriverException
     */
    @Override
    public int setOutletOn(int outlet) throws DriverException
    {
        receiveString("olOn " + outlet);
        return 1;
    }


    /**
     *  Turns outlet power off immediately.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    @Override
    public int setOutletOff(int outlet) throws DriverException
    {
        receiveString("olOff " + outlet);
        return 1;
    }


    /**
     *  Turns named outlet power on immediately.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    @Override
    public int setOutletOn(String outlet) throws DriverException
    {
        receiveString("olOn \"" + outlet + "\"");
        return 1;
    }


    /**
     *  Turns named outlet power off immediately.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    @Override
    public int setOutletOff(String outlet) throws DriverException
    {
        receiveString("olOff \"" + outlet + "\"");
        return 1;
    }


    /**
     *  Turns outlets power on immediately.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    @Override
    public int setOutletsOn(int[] outlets) throws DriverException
    {
        return sendOutlets("olOn", outlets);
    }


    /**
     *  Turns outlets power off immediately.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    @Override
    public int setOutletsOff(int[] outlets) throws DriverException
    {
        return sendOutlets("olOff", outlets);
    }


    /**
     *  Turns named outlets power on immediately.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    @Override
    public int setOutletsOn(String[] outlets) throws DriverException
    {
        return sendOutlets("olOn", outlets);
    }



    /**
     *  Turns named outlets power off immediately.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    @Override
    public int setOutletsOff(String[] outlets) throws DriverException
    {
        return sendOutlets("olOff", outlets);
    }



    /**
     *  Turns outlet power on with delay.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned on (1)
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletOn(int outlet) throws DriverException
    {
        receiveString("olDlyOn " + outlet);
        return 1;
    }


    /**
     *  Turns outlet power off with delay.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletOff(int outlet) throws DriverException
    {
        receiveString("olDlyOff " + outlet);
        return 1;
    }


    /**
     *  Turns named outlet power on with delay.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned on (1)
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletOn(String outlet) throws DriverException
    {
        receiveString("olDlyOn \"" + outlet + "\"");
        return 1;
    }


    /**
     *  Turns named outlet power off with delay.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletOff(String outlet) throws DriverException
    {
        receiveString("olDlyOff \"" + outlet + "\"");
        return 1;
    }


    /**
     *  Turns outlets power on with delay.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletsOn(int[] outlets) throws DriverException
    {
        return sendOutlets("olDlyOn", outlets);
    }


    /**
     *  Turns outlets power off with delay.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletsOff(int[] outlets) throws DriverException
    {
        return sendOutlets("olDlyOff", outlets);
    }


    /**
     *  Turns named outlets power on with delay.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletsOn(String[] outlets) throws DriverException
    {
        return sendOutlets("olDlyOn", outlets);
    }


    /**
     *  Turns named outlets power off with delay.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletsOff(String[] outlets) throws DriverException
    {
        return sendOutlets("olDlyOff", outlets);
    }


    /**
     *  Reads input currents, one per phase
     * 
     *  @return  The array of input currents (amps)
     *  @throws  DriverException
     */
    @Override
    public double[] readCurrent() throws DriverException
    {
        String[] reply = receiveString("phReading all current");
        checkReplyLength(reply, getPhaseCount() + 1);
        double[] current = new double[reply.length - 1];
        for (int j = 0; j < current.length; j++) {
            try {
                String token = reply[j].split(": ")[1];
                current[j] = Double.valueOf(token.substring(0, token.length() - 1));
            }
            catch (IndexOutOfBoundsException | NumberFormatException e) {
                responseError(reply[j]);
            }
        }
        return current;
    }


    /**
     *  Sends a command to outlets.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of unique outlets
     *  @throws  DriverException
     */
    private int sendOutlets(String command, int[] outlets) throws DriverException
    {
        Set outletSet = makeOutletSet(outlets);
        receiveString(command + makeOutletString(outletSet));
        return outletSet.size();
    }


    /**
     *  Sends a command to named outlets.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of unique outlets
     *  @throws  DriverException
     */
    private int sendOutlets(String command, String[] outlets) throws DriverException
    {
        Set outletSet = makeOutletSet(outlets);
        receiveString(command + makeOutletString(outletSet));
        return outletSet.size();
    }


    /**
     *  Sends a command and receives a string array
     *
     *  @param  command  The command to send
     *  @return  The string array
     *  @throws  DriverException
     */
    @Override
    public synchronized String[] receiveString(String command) throws DriverException
    {
        String[] resp = receive(command);
        if (resp.length < 2) {
            throw new DriverException("No response received");
        }
        if (!resp[1].startsWith("E0")) {
            throw new DriverException("Command error " + resp[1]);
        }
        String[] reply = new String[resp.length - 2];
        System.arraycopy(resp, 2, reply, 0, reply.length);
        return reply;
    }


    /**
     *  Gets the version information
     *
     *  @return  The saved prompt
     */
    private void getVersionInfo() throws DriverException
    {
        String[] reply = receiveString("prodinfo");
        checkReplyLength(reply, 14);
        product = getVersionItemString(reply[2]);
        fwVersion = getVersionItemString(reply[0]);
        numOutlets = getVersionItemInt(reply[6]);
        numPhases = getVersionItemInt(reply[10]);
    }


    /**
     *  Generates an outlet number set.
     *
     *  @param  outlets  The array of outlet numbers
     *  @return  The generated outlet set
     *  @throws  DriverException if invalid
     */
    private static Set<Integer> makeOutletSet(int[] outlets)
    {
        Set<Integer> outletSet = new HashSet();
        for (int outlet : outlets) {
            outletSet.add(outlet);
        }
        return outletSet;
    }


    /**
     *  Generates an outlet number set.
     *
     *  @param  outlets  The array of outlet names (case-insensitive)
     *  @return  The generated outlet number set
     *  @throws  DriverException if invalid
     */
    private Set<Integer> makeOutletSet(String[] outlets) throws DriverException
    {
        Map<String, Integer> outletMap = getOutletNumberMap();
        Map<String, Integer> outletMapLC = new LinkedHashMap<>();
        for (String outlet : outletMap.keySet()) {
            outletMapLC.put(outlet.toLowerCase(), outletMap.get(outlet));
        }
        Set<Integer> outletSet = new HashSet();
        for (String outlet : outlets) {
            Integer outletNum = outletMapLC.get(outlet.toLowerCase());
            if (outletNum == null) {
                throw new DriverException("Invalid outlet name: \"" + outlet + "\"");
            }
            outletSet.add(outletNum);
        }
        return outletSet;
    }


    /**
     *  Generates an outlet specification string
     *
     *  @param  outlets  The set of outlet numbers
     *  @return  The generated outlet string
     *  @throws  DriverException if invalid
     */
    private static String makeOutletString(Set<Integer> outlets)
    {
        StringBuilder outString = new StringBuilder();
        char sep = ' ';
        for (int outlet : outlets) {
            outString.append(sep).append(outlet);
            sep = ',';
        }
        if (outString.length() == 0) {
            outString.append(" all");
        }
        return outString.toString();
    }


    /**
     *  Gets the string value from a version line.
     * 
     *  Values all start at offset 18 
     *
     *  @return  The string value
     *  @throws  DriverException
     */
    private static String getVersionItemString(String line) throws DriverException
    {
        try {
            return line.substring(18).trim();
        }
        catch (IndexOutOfBoundsException e) {
            responseError(line);
            return null;  // Will never happen
        }
    }


    /**
     *  Gets the integer value from a version line.
     * 
     *  Values all start at offset 18 
     *
     *  @return  The string value
     *  @throws  DriverException
     */
    private static int getVersionItemInt(String line) throws DriverException
    {
        try {
            return Integer.valueOf(line.substring(18).trim());
        }
        catch (IndexOutOfBoundsException | NumberFormatException e) {
            responseError(line);
            return 0;  // Will never happen
        }
    }


    /**
     *  Gets an integer from a numeric string.
     *
     *  @param  str  The numeric string
     *  @return  The integer value
     *  @throws  NumberFormatExceptiom
     */
    private static int getInteger(String str)
    {
        return Integer.valueOf(str.trim());
    }

}
