package org.lsst.ccs.drivers.apcpdu;

import java.io.IOException;
import java.io.PrintStream;
import java.util.HashMap;
import java.util.Map;
import jline.console.ConsoleReader;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Tests the APC model 9630 UPS driver
 *
 *  @author Owen Saxton
 */
public class TestAP9630UPS {

    private static final Map<Integer, String> statusMap = new HashMap<>();
    static {
        statusMap.put(AP9630UPS.ST_ONLINE, "Online");
        statusMap.put(AP9630UPS.ST_BATTERY, "Battery");
    }
    private static final PrintStream out = System.out;
    private final AP9630UPS ups = new AP9630UPS();
    private ConsoleReader reader;

    @Command(description="Open connection to a UPS")
    public void open(@Argument(name="conntype", description="Connection type")
                     AP9630UPS.ConnType connType,
                     @Argument(name="devcid", description="Device ID")
                     String devcId,
                     @Argument(name="username", description="User name")
                     String userName,
                     @Argument(name="password", description="Password")
                     String password) throws DriverException {
        ups.open(connType, devcId, userName, password);
    }

    @Command(description="Open connection to a UPS")
    public void open(@Argument(name="conntype", description="Connection type")
                     AP9630UPS.ConnType connType,
                     @Argument(name="devcid", description="Device ID")
                     String devcId) throws DriverException {
        ups.open(connType, devcId);
    }

    @Command(description="Set the response overall timeout")
    public void setTimeout(@Argument(description="Timeout (ms)")
                           int timeout) throws DriverException {
        ups.setTimeout(timeout);
    }

    @Command(description="Set the response inter-character timeout")
    public void setCharTimeout(@Argument(description="Timeout (ms)")
                               int timeout) throws DriverException {
        ups.setCharTimeout(timeout);
    }

    @Command(description="Close the connection")
    public void close() throws DriverException {
        ups.close();
    }

    @Command(description="Show the product name")
    public String product() throws DriverException {
        return ups.getProductName();
    }

    @Command(description="Show the firmware version")
    public String version() throws DriverException {
        return ups.getFWVersion();
    }

    @Command(description="Show the UPS status")
    public String status() throws DriverException {
        int stat = ups.getStatus();
        return statusMap.get(stat) + " (" + stat + ")";
    }

    @Command(description="Show the output voltage")
    public double outputVoltage() throws DriverException {
        return ups.getOutputVoltage();
    }

    @Command(description="Show the output current")
    public double outputCurrent() throws DriverException {
        return ups.getOutputCurrent();
    }

    @Command(description="Show the output power")
    public double outputPower() throws DriverException {
        return ups.getOutputPower();
    }

    @Command(description="Show the battery temperature")
    public double temperature() throws DriverException {
        return ups.getTemperature();
    }

    @Command(description="Show the battery voltage")
    public double batteryVoltage() throws DriverException {
        return ups.getBatteryVoltage();
    }

    @Command(description="Show the battery charge")
    public double batteryCharge() throws DriverException {
        return ups.getBatteryCharge();
    }

    @Command(description="Show the remaining run time")
    public int runTime() throws DriverException {
        return ups.getRunTime();
    }

    @Command(name="talk", description="Talk to the UPS interactively")
    public void talk() throws DriverException, IOException {
        out.println("Type ctrl-D to exit");
        reader = (reader == null) ? new ConsoleReader() : reader;
        while (true) {
            String cmnd = reader.readLine(ups.getPrompt());
            if (cmnd == null) break;
            String[] resp = ups.receive(cmnd);
            for (int j = 1; j < resp.length; j++) {
                out.println(resp[j]);
            }
        }
        out.println();
    }

}
