package org.lsst.ccs.drivers.apcpdu;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;

/**
 ***************************************************************************
 **
 ** Program to test the AP7900 device driver * * @author Homer *
 * **************************************************************************
 */
public class TestAP7900 {

    protected AP7900 mydev;
    private final Thread readW = new Thread(new Reader());
    private int last_state = -1; // last channel state seen
    private int nRead = 0, nByte = 0, nTimeout = 0;
    private boolean open;

    /*
     Outlet - Setup - Device:
     1 - S1 - Vacuum Gauge
     2 - S1 - PD & Bias
     3 - S1 - Archon Controller
     4 - S2 - Vacuum Gauge
     5 - S2 - PD & Bias
     6 - S2 - Archon Controller
     7 - S1 - Computer
     8 - S2 - Computer
     */
    /**
     ***************************************************************************
     **
     ** Inner class to implement device reading thread. *
     * **************************************************************************
     */
    private class Reader implements Runnable {

        @Override
        public void run() {
            while (open) {
                try {
                    String reply = mydev.read();
                    if (reply.indexOf("State")>=0) {
                        last_state = 0;
                        if (reply.indexOf("ON")>=0) last_state = 1;
                    }
                    System.out.println(reply);
                    nRead++;
                    nByte += reply.length();
                } catch (DriverTimeoutException e) {
                    nTimeout++;
                } catch (DriverException e) {
                    if (open) {
                        System.out.println(e);
                        System.exit(0);
                    }
                }
            }
        }
    }

    @Command(name = "openserial", description = "Open a serial connection to the PDU using default addr")
    public void openserial() throws DriverException {

        this.openserial("/dev/ttyS1");
    }

    @Command(name = "openserial", description = "Open a serial connection to the PDU")
    public void openserial(
            @Argument(name = "devId", description = "The device ID of the device to open") String MyDevId
    ) throws DriverException {

        mydev = new AP7900();
        mydev.openserial(MyDevId, 9600);
    }

    @Command(name = "opennet", description = "Open a network connection to the PDU using default addr")
    public void opennet() throws DriverException {

        this.opennet("130.199.47.27", 23);
    }

    @Command(name = "opennet", description = "Open a network connection to the PDU")
    public void opennet(
            @Argument(name = "netaddr", description = "The netaddr of the device to open - 130.199.47.172") String MyNetAddr,
            @Argument(name = "port", description = "port - 23") int port
    ) throws DriverException {

        mydev = new AP7900();
        mydev.opennet(MyNetAddr, port,3);
        open = true;
        mydev.setTimeout(60);
        readW.setDaemon(true);
        readW.start();

    }

    @Command(name = "close", description = "Close the device")
    public void close() throws Exception {
        mydev.close();
    }

    @Command(name = "ison", description = "is the outlet on")
    public boolean ison(@Argument(name = "outlet", description = "outlet number") int outlet
    ) throws Exception {
        mydev.isOutletOn(outlet);
        last_state = -1; // reset
        for (int i=0;i<200;i++) {
            System.out.println("Waiting for a response from the PDU");
            Thread.sleep(1000);
            if (last_state>=0) break;            
        }
        if (last_state<0) {
            throw new DriverException("PDU never responded to outlet state request");
        }
        if (last_state==1) return(true);
        else return(false);
    }

    @Command(name = "setoutlet", description = "set state of outlet")
    public void setoutlet(
            @Argument(name = "outlet", description = "outlet number") int outlet,
            @Argument(name = "state", description = "state to set") boolean state
    ) throws Exception {
        mydev.setOutletState(outlet, state);
    }

}
