package org.lsst.ccs.drivers.apcpdu;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.ascii.Session;
import static org.lsst.ccs.drivers.ascii.Session.OPTN_KEEP_ALIVE;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Simulated Driver for controlling an APC model 7900 power strip.
 *
 *  This also works on the newer 7900B model even though its official command
 *  syntax has been changed.  But there's a bug: outlet names containing dashes
 *  cause command errors.
 *
 *  @author Owen Saxton
 */
public class APC7900Sim extends Session implements APC7900Series {

    private final List<String> outlets;
    private final List<Integer> offDelays = new ArrayList();
    private final List<Integer> onDelays = new ArrayList();
    private final List<Boolean> outletState = new ArrayList();
    private final Random r = new Random();

    /**
     *  Constructor
     */
    public APC7900Sim(List<String> outlets)
    {
        super(OPTN_KEEP_ALIVE, "APC>", "User Name :", "Password  :", "exit", Ascii.Terminator.CR);
        this.outlets = outlets;
        for ( String outlet : outlets) {
            offDelays.add(1);
            onDelays.add(1);
            outletState.add(Boolean.TRUE);
        }
    }


    /**
     *  Opens a connection and logs in.
     *
     *  @param  connType  The enumerated connection type: TELNET or SERIAL
     *  @param  ident     The host name (TELNET) or port name (SERIAL)
     *  @param  username  The user name
     *  @param  password  The password
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident, String username, String password) throws DriverException
    {
    }


    /**
     *  Opens a connection.
     *
     *  @param  connType  The enumerated connection type: TELNET or SERIAL
     *  @param  ident     The host name (TELNET) or port name (SERIAL)
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident) throws DriverException
    {
    }


    /**
     *  Closes the connection
     *
     *  @throws  DriverException
     */
    @Override
    public void close() throws DriverException
    {
    }


    /**
     *  Gets the product name
     *
     *  @return  The product name
     *  @throws  DriverException
     */
    @Override
    public String getProductName() throws DriverException
    {
        return "APC7900Sim";
    }


    /**
     *  Gets the firmware version
     *
     *  @return  The firmware version
     *  @throws  DriverException
     */
    @Override
    public String getFWVersion() throws DriverException
    {
        return "1.0.0";
    }


    /**
     *  Gets the number of phases
     *
     *  @return  The number of phases
     *  @throws  DriverException
     */
    @Override
    public int getPhaseCount() throws DriverException
    {
        return 1;
    }


    /**
     *  Gets the number of outlets
     *
     *  @return  The number of outlets
     *  @throws  DriverException
     */
    @Override
    public int getOutletCount() throws DriverException
    {
        return outlets.size();
    }


    /**
     *  Gets the name of an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The outlet name
     *  @throws  DriverException
     */
    @Override
    public String getOutletName(int outlet) throws DriverException
    {
        return outlets.get(outlet-1);
    }


    /**
     *  Sets the name of an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @param  name    The outlet name
     *  @throws  DriverException
     */
    @Override
    public void setOutletName(int outlet, String name) throws DriverException
    {
        outlets.set(outlet-1, name);
    }


    /**
     *  Gets an outlet number, given its name
     *
     *  @param  outlet  The outlet name
     *  @return  The outlet number
     *  @throws  DriverException
     */
    @Override
    public int getOutletNumber(String outlet) throws DriverException
    {
        return outlets.indexOf(outlet)+1;
    }


    /**
     *  Gets the map of outlet names to numbers
     *
     *  @return  The outlet number map
     *  @throws  DriverException
     */
    @Override
    public Map<String, Integer> getOutletNumberMap() throws DriverException
    {
        Map<String, Integer> numberMap = new LinkedHashMap<>();
        int count = 1;
        for (String outlet : outlets) {
            numberMap.put(outlet, count);
            count++;
        }
        return numberMap;
    }


    /**
     *  Gets the on delay for an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    @Override
    public int getOutletOnDelay(int outlet) throws DriverException
    {
        return onDelays.get(outlet-1);
    }


    /**
     *  Gets the off delay for an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    @Override
    public int getOutletOffDelay(int outlet) throws DriverException
    {
        return offDelays.get(outlet-1);
    }


    /**
     *  Gets the on delay for a named outlet
     *
     *  @param  outlet  The outlet name
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    @Override
    public int getOutletOnDelay(String outlet) throws DriverException
    {
        return onDelays.get(outlets.indexOf(outlet)+1);
    }


    /**
     *  Gets the off delay for a named outlet
     *
     *  @param  outlet  The outlet name
     *  @return  The delay (secs) or -1 for never
     *  @throws  DriverException
     */
    @Override
    public int getOutletOffDelay(String outlet) throws DriverException
    {
        return offDelays.get(outlets.indexOf(outlet)+1);
    }



    /**
     *  Gets the on delay for all outlets
     *
     *  @return  An array containing the delays for all outlets
     *  @throws  DriverException
     */
    @Override
    public int[] getOutletOnDelays() throws DriverException
    {
        int[] result = new int[getOutletCount()];
        int count = 0;
        for ( Integer i : onDelays ) {
            result[count] = i;
            count++;            
        }
        return result;
    }


    /**
     *  Gets the off delay for all outlets
     *
     *  @return  An array containing the delays for all outlets
     *  @throws  DriverException
     */
    @Override
    public int[] getOutletOffDelays() throws DriverException
    {
        int[] result = new int[getOutletCount()];
        int count = 0;
        for ( Integer i : offDelays ) {
            result[count] = i;
            count++;            
        }
        return result;
    }



    /**
     *  Gets the map of outlet names to on delays
     *
     *  @return  The delay map
     *  @throws  DriverException
     */
    @Override
    public Map<String, Integer> getOutletOnDelayMap() throws DriverException
    {
        return getOutletDelayMap(true);
    }


    /**
     *  Gets the map of outlet names to off delays
     *
     *  @return  The delay map
     *  @throws  DriverException
     */
    @Override
    public Map<String, Integer> getOutletOffDelayMap() throws DriverException
    {
        return getOutletDelayMap(false);
    }


    /**
     *  Gets the map of outlet names to on or off delays
     *
     *  @param  on  True to set on delay, false to set off delay
     *  @return  The delay map
     *  @throws  DriverException
     */
    private Map<String, Integer> getOutletDelayMap(boolean on) throws DriverException
    {
        Map<String, Integer> delayMap = new LinkedHashMap<>();
        int count = 0;
        for (String line : outlets) {
            delayMap.put(line, on ? onDelays.get(count) : offDelays.get(count));
            count++;
        }
        return delayMap;
    }


    /**
     *  Sets outlet on delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlet  The outlet number, starting from 1
     *  @throws  DriverException
     */
    @Override
    public void setOutletOnDelay(int delay, int outlet) throws DriverException
    {
        onDelays.set(outlet-1, delay);
    }


    /**
     *  Sets outlet off delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlet  The outlet number, starting from 1
     *  @throws  DriverException
     */
    @Override
    public void setOutletOffDelay(int delay, int outlet) throws DriverException
    {
        offDelays.set(outlet-1, delay);
    }


    /**
     *  Sets named outlet on delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlet  The outlet name
     *  @throws  DriverException
     */
    @Override
    public void setOutletOnDelay(int delay, String outlet) throws DriverException
    {
        onDelays.set(outlets.indexOf(outlet), delay);
    }


    /**
     *  Sets named outlet off delay.
     *
     *  @param  delay   The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlet  The outlet name
     *  @throws  DriverException
     */
    @Override
    public void setOutletOffDelay(int delay, String outlet) throws DriverException
    {
        offDelays.set(outlets.indexOf(outlet), delay);
    }


    /**
     *  Sets outlets on delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlets  The array of outlet numbers
     *  @throws  DriverException
     */
    @Override
    public void setOutletOnDelay(int delay, int[] outlets) throws DriverException
    {
        for ( int outlet : outlets ) {
            onDelays.set(outlet-1, delay);
        }
    }


    /**
     *  Sets outlets off delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlets  The array of outlet numbers
     *  @throws  DriverException
     */
    @Override
    public void setOutletOffDelay(int delay, int[] outlets) throws DriverException
    {
        for ( int outlet : outlets ) {
            offDelays.set(outlet-1, delay);
        }
    }


    /**
     *  Sets named outlets on delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered on (-1 = never)
     *  @param  outlets  The array of outlet names
     *  @throws  DriverException
     */
    @Override
    public void setOutletOnDelay(int delay, String[] outlets) throws DriverException
    {
        for ( String outlet : outlets ) {
            onDelays.set(this.outlets.indexOf(outlet), delay);
        }
    }


    /**
     *  Sets named outlets off delay.
     *
     *  @param  delay    The delay (sec) before the outlet is powered off (-1 = never)
     *  @param  outlets  The array of outlet names
     *  @throws  DriverException
     */
    @Override
    public void setOutletOffDelay(int delay, String[] outlets) throws DriverException
    {
        for ( String outlet : outlets ) {
            offDelays.set(this.outlets.indexOf(outlet), delay);
        }
    }


    /**
     *  Gets the on (powered) state of an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  Whether the outlet is powered
     *  @throws  DriverException
     */
    @Override
    public boolean isOutletOn(int outlet) throws DriverException
    {
        return outletState.get(outlet-1);        
    }


    /**
     *  Gets the on (powered) state of an outlet
     *
     *  @param  outlet  The outlet name
     *  @return  Whether the outlet is powered
     *  @throws  DriverException
     */
    @Override
    public boolean isOutletOn(String outlet) throws DriverException
    {
        return outletState.get(outlets.indexOf(outlet));                
    }


    /**
     *  Gets the on (powered) state of all outlets
     *
     *  @return  A boolean array containing the on state of all outlets
     *  @throws  DriverException
     */
    @Override
    public boolean[] getOutletOnStates() throws DriverException
    {
        boolean[] states = new boolean[outletState.size()];
        for ( int i = 0; i < outletState.size(); i++ ) {
            states[i] = outletState.get(i);
        }
        return states;
    }


    /**
     *  Gets the map of outlet names to on (powered) states
     *
     *  @return  The on state map
     *  @throws  DriverException
     */
    @Override
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException
    {
        Map<String, Boolean> stateMap = new LinkedHashMap<>();
        for ( int i = 0; i < outletState.size(); i++ ) {
             stateMap.put(outlets.get(i), outletState.get(i));
        }
        return stateMap;
    }


    /**
     *  Turns outlet power on immediately.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned on (1)
     *  @throws  DriverException
     */
    @Override
    public int setOutletOn(int outlet) throws DriverException
    {
        outletState.set(outlet-1, Boolean.TRUE);
        return 1;
    }


    /**
     *  Turns outlet power off immediately.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    @Override
    public int setOutletOff(int outlet) throws DriverException
    {
        outletState.set(outlet-1, Boolean.FALSE);
        return 1;
    }


    /**
     *  Turns named outlet power on immediately.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned on (1)
     *  @throws  DriverException
     */
    @Override
    public int setOutletOn(String outlet) throws DriverException
    {
        outletState.set(outlets.indexOf(outlet), Boolean.TRUE);
        return 1;
    }


    /**
     *  Turns named outlet power off immediately.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    @Override
    public int setOutletOff(String outlet) throws DriverException
    {
        outletState.set(outlets.indexOf(outlet), Boolean.FALSE);
        return 1;
    }


    /**
     *  Turns outlets power on immediately.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    @Override
    public int setOutletsOn(int[] outlets) throws DriverException
    {
        for ( int i : outlets ) {
            outletState.set(i-1, Boolean.TRUE);
        }
        return outlets.length;
    }


    /**
     *  Turns outlets power off immediately.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    @Override
    public int setOutletsOff(int[] outlets) throws DriverException
    {
        for ( int i : outlets ) {
            outletState.set(i-1, Boolean.FALSE);
        }
        return outlets.length;
    }


    /**
     *  Turns named outlets power on immediately.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    @Override
    public int setOutletsOn(String[] outlets) throws DriverException
    {
        for ( String outlet : outlets ) {
            outletState.set(this.outlets.indexOf(outlet), Boolean.TRUE);
        }
        return outlets.length;
    }


    /**
     *  Turns named outlets power off immediately.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    @Override
    public int setOutletsOff(String[] outlets) throws DriverException
    {
        for ( String outlet : outlets ) {
            outletState.set(this.outlets.indexOf(outlet), Boolean.FALSE);
        }
        return outlets.length;
    }


    /**
     *  Turns outlet power on with delay.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned on (1)
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletOn(int outlet) throws DriverException
    {
        return setOutletOn(outlet);        
    }


    /**
     *  Turns outlet power off with delay.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletOff(int outlet) throws DriverException
    {
        return setOutletOff(outlet);        
    }


    /**
     *  Turns named outlet power on with delay.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned on (1)
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletOn(String outlet) throws DriverException
    {
        return setOutletOn(outlet);        
    }


    /**
     *  Turns named outlet power off with delay.
     *
     *  @param  outlet  The outlet name
     *  @return  The number of outlets turned off (1)
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletOff(String outlet) throws DriverException
    {
        return setOutletOff(outlet);        
    }


    /**
     *  Turns outlets power on with delay.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletsOn(int[] outlets) throws DriverException
    {
        return setOutletsOn(outlets);        
    }


    /**
     *  Turns outlets power off with delay.
     *
     *  @param  outlets  An array of outlet numbers
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletsOff(int[] outlets) throws DriverException
    {
        return setOutletsOff(outlets);        
    }


    /**
     *  Turns named outlets power on with delay.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned on
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletsOn(String[] outlets) throws DriverException
    {
        return setOutletsOn(outlets);        
    }


    /**
     *  Turns named outlets power off with delay.
     *
     *  @param  outlets  An array of outlet names
     *  @return  The number of outlets turned off
     *  @throws  DriverException
     */
    @Override
    public int delayedOutletsOff(String[] outlets) throws DriverException
    {
        return setOutletsOff(outlets);        
    }


    /**
     *  Reads input currents, one per phase
     * 
     *  @return  The array of input currents (amps)
     *  @throws  DriverException
     */
    @Override
    public double[] readCurrent() throws DriverException
    {
        double[] currents = new double[outlets.size()];
        for ( int i = 0; i < outlets.size(); i++ ) {
            currents[i] = r.nextDouble()*2;
        }
        return currents;
    }


    /**
     *  Reads input VA
     * 
     *  @return  The VA (watts)
     *  @throws  DriverException
     */
    @Override
    public double readVA() throws DriverException
    {
        return 2 + r.nextDouble()*2;
    }


    /**
     *  Reads input power
     * 
     *  @return  The power (watts)
     *  @throws  DriverException
     */
    @Override
    public double readPower() throws DriverException
    {
        return 2 + r.nextDouble()*2;
    }


    /**
     *  Reads input VA and power
     * 
     *  @return  The array of input power values, VA & power (watts)
     *  @throws  DriverException
     */
    @Override
    public double[] readPowers() throws DriverException
    {
        double[] values = new double[] {readPower(), readVA()};
        return values;
    }


    /**
     *  Sends a command and receives a string array
     *
     *  @param  command  The command to send
     *  @return  The string array
     *  @throws  DriverException
     */
    @Override
    public synchronized String[] receiveString(String command) throws DriverException
    {
        return null;
    }

}
