package org.lsst.ccs.drivers.apcpdu;

import java.io.IOException;
import java.io.PrintStream;
import java.util.Map;
import jline.console.ConsoleReader;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 ***********************************************
 *
 *  Tests the APC model 7900 power strip driver
 *
 *  @author Owen Saxton
 *
 ***********************************************
 */
public class TestAPC7900 {

    private static final PrintStream out = System.out;
    private final APC7900 pdu = new APC7900();
    private ConsoleReader reader;


    @Command(name="open", description="Open connection to a PDU")
    public void open(@Argument(name="conntype", description="Connection type")
                     APC7900.ConnType connType,
                     @Argument(name="devcid", description="Device ID")
                     String devcId,
                     @Argument(name="username", description="User name")
                     String userName,
                     @Argument(name="password", description="Password")
                     String password) throws DriverException
    {
        pdu.open(connType, devcId, userName, password);
    }


    @Command(name="open", description="Open connection to a PDU")
    public void open(@Argument(name="conntype", description="Connection type")
                     APC7900.ConnType connType,
                     @Argument(name="devcid", description="Device ID")
                     String devcId) throws DriverException
    {
        pdu.open(connType, devcId);
    }


    @Command(name="close", description="Close the connection")
    public void close() throws DriverException
    {
        pdu.close();
    }


    @Command(name="product", description="Show the product name")
    public String product() throws DriverException
    {
        return pdu.getProductName();
    }


    @Command(name="version", description="Show the firmware version")
    public String version() throws DriverException
    {
        return pdu.getFWVersion();
    }


    @Command(name="phasecount", description="Show the phase count")
    public String phaseCount() throws DriverException
    {
        return String.valueOf(pdu.getPhaseCount());
    }


    @Command(name="outletcount", description="Show the outlet count")
    public String outletCount() throws DriverException
    {
        return String.valueOf(pdu.getOutletCount());
    }


    @Command(name="outletstate", description="Show the state of an outlet")
    public String outletState(@Argument(name="outlet", description="Outlet number")
                              int outlet) throws DriverException
    {
        return (pdu.isOutletOn(outlet) ? "on" : "off");
    }


    @Command(name="outletstaten", description="Show the state of a named outlet")
    public String outletState(@Argument(name="outlet", description="Outlet name")
                              String outlet) throws DriverException
    {
        return (pdu.isOutletOn(outlet) ? "on" : "off");
    }


    @Command(name="outletstate", description="Show the state of all outlets")
    public String outletState() throws DriverException
    {
        boolean[] states = pdu.getOutletOnStates();
        StringBuilder text = new StringBuilder();
        for (int j = 0; j < states.length; j++) {
            if (j % 8 == 0) {
                text.append(j == 0 ? "" : "\n");
            }
            text.append(String.format("%4s: %3s", j + 1, states[j] ? "on " : "off"));
        }
        return text.toString();
    }


    @Command(name="outletstaten", description="Show the state of all outlets by name")
    public String outletStateN() throws DriverException
    {
        Map<String, Boolean> stateMap = pdu.getOutletOnStateMap();
        StringBuilder text = new StringBuilder();
        int lineLeng = 0;
        String itemSep = "";
        for (String name : stateMap.keySet()) {
            boolean on = stateMap.get(name);
            String item = String.format("%s: %s", name, on ? "on" : "off");
            if (lineLeng + itemSep.length() + item.length() > 80) {
                text.append("\n");
                lineLeng = 0;
                itemSep = "";
            }
            text.append(itemSep).append(item);
            lineLeng += itemSep.length() + item.length();
            itemSep = "   ";
        }
        return text.toString();
    }


    @Command(name="outletname", description="Show the name of an outlet")
    public String outletName(@Argument(name="outlet", description="Outlet number")
                             int outlet) throws DriverException
    {
        return pdu.getOutletName(outlet);
    }


    @Command(name="outletname", description="Set the name of an outlet")
    public void outletName(@Argument(name="outlet", description="Outlet number")
                             int outlet,
                             @Argument(name="name", description="Outlet name")
                             String name) throws DriverException
    {
        pdu.setOutletName(outlet, name);
    }


    @Command(name="outletnumber", description="Show the number of a named outlet")
    public String outletNumber(@Argument(name="outlet", description="Outlet name")
                               String outlet) throws DriverException
    {
        return String.valueOf(pdu.getOutletNumber(outlet));
    }


    @Command(name="outletnumber", description="Show the numbers of all outlets by name")
    public String outletNumber() throws DriverException
    {
        return formatDelayMap(pdu.getOutletNumberMap());
    }


    @Command(name="ondelay", description="Show outlet on delay")
    public String outletOnDelay(@Argument(name="outlet", description="Outlet number")
                                int outlet) throws DriverException
    {
        return String.valueOf(pdu.getOutletOnDelay(outlet));
    }


    @Command(name="ondelay", description="Show all outlet on delays")
    public String outletOnDelay() throws DriverException
    {
        return formatDelays(pdu.getOutletOnDelays());
    }


    @Command(name="offdelay", description="Show outlet off delay")
    public String outletOffDelay(@Argument(name="outlet", description="Outlet number")
                                 int outlet) throws DriverException
    {
        return String.valueOf(pdu.getOutletOffDelay(outlet));
    }


    @Command(name="offdelay", description="Show all outlet off delays")
    public String outletOffDelay() throws DriverException
    {
        return formatDelays(pdu.getOutletOffDelays());
    }


    private String formatDelays(int[] delays)
    {
        StringBuilder text = new StringBuilder();
        for (int j = 0; j < delays.length; j++) {
            if (j % 8 == 0) {
                text.append(j == 0 ? "" : "\n");
            }
            text.append(String.format("%4s: %-3s", j + 1, delays[j]));
        }
        return text.toString();
    }


    @Command(name="ondelayn", description="Show named outlet on delay")
    public String outletOnDelayN(@Argument(name="outlet", description="Outlet name")
                                 String outlet) throws DriverException
    {
        return String.valueOf(pdu.getOutletOnDelay(outlet));
    }


    @Command(name="ondelayn", description="Show all outlet on delays by name")
    public String outletOnDelayN() throws DriverException
    {
        return formatDelayMap(pdu.getOutletOnDelayMap());
    }


    @Command(name="offdelayn", description="Show named outlet off delay")
    public String outletOffDelayN(@Argument(name="outlet", description="Outlet name")
                                  String outlet) throws DriverException
    {
        return String.valueOf(pdu.getOutletOffDelay(outlet));
    }


    @Command(name="offdelayn", description="Show all outlet off delays by name")
    public String outletOffDelayN() throws DriverException
    {
        return formatDelayMap(pdu.getOutletOffDelayMap());
    }


    private String formatDelayMap(Map<String, Integer> delayMap) throws DriverException
    {
        StringBuilder text = new StringBuilder();
        int lineLeng = 0;
        String itemSep = "";
        for (String name : delayMap.keySet()) {
            int value = delayMap.get(name);
            String item = String.format("%s: %s", name, value);
            if (lineLeng + itemSep.length() + item.length() > 80) {
                text.append("\n");
                lineLeng = 0;
                itemSep = "";
            }
            text.append(itemSep).append(item);
            lineLeng += itemSep.length() + item.length();
            itemSep = "   ";
        }
        return text.toString();
    }


    @Command(name="ondelay", description="Set outlet on delay")
    public void outletOnDelay(@Argument(name="outlet", description="Outlet number")
                              int outlet,
                              @Argument(name="delay", description="Delay (sec)")
                              int delay) throws DriverException
    {
        pdu.setOutletOnDelay(delay, outlet);
    }


    @Command(name="ondelayn", description="Set named outlet on delay")
    public void outletOnDelay(@Argument(name="outlet", description="Outlet name")
                              String outlet,
                              @Argument(name="delay", description="Delay (sec)")
                              int delay) throws DriverException
    {
        pdu.setOutletOnDelay(delay, outlet);
    }


    @Command(name="offdelay", description="Set outlet off delay")
    public void outletOffDelay(@Argument(name="outlet", description="Outlet number")
                               int outlet,
                               @Argument(name="delay", description="Delay (sec)")
                               int delay) throws DriverException
    {
        pdu.setOutletOffDelay(delay, outlet);
    }


    @Command(name="offdelayn", description="Set named outlet off delay")
    public void outletOffDelay(@Argument(name="outlet", description="Outlet name")
                               String outlet,
                               @Argument(name="delay", description="Delay (sec)")
                               int delay) throws DriverException
    {
        pdu.setOutletOffDelay(delay, outlet);
    }


    @Command(name="on", description="Turn outlet(s) power on")
    public void outletOn(@Argument(name="outlet", description="Outlet number(s)")
                         int... outlet) throws DriverException
    {
        if (outlet.length == 1) {
            pdu.setOutletOn(outlet[0]);
        }
        else {
            pdu.setOutletsOn(outlet);
        }
    }


    @Command(name="onn", description="Turn named outlet(s) power on")
    public void outletOn(@Argument(name="outlet", description="Outlet name(s)")
                         String... outlet) throws DriverException
    {
        if (outlet.length == 1) {
            pdu.setOutletOn(outlet[0]);
        }
        else {
            pdu.setOutletsOn(outlet);
        }
    }


    @Command(name="off", description="Turn outlet(s) power off")
    public void outletOff(@Argument(name="outlet", description="Outlet number(s)")
                          int... outlet) throws DriverException
    {
        if (outlet.length == 1) {
            pdu.setOutletOff(outlet[0]);
        }
        else {
            pdu.setOutletsOff(outlet);
        }
    }


    @Command(name="offn", description="Turn named outlet(s) power off")
    public void outletOff(@Argument(name="outlet", description="Outlet name(s)")
                          String... outlet) throws DriverException
    {
        if (outlet.length == 1) {
            pdu.setOutletOff(outlet[0]);
        }
        else {
            pdu.setOutletsOff(outlet);
        }
    }


    @Command(name="delayedon", description="Turn outlet(s) power on with delay")
    public void delayedOn(@Argument(name="outlet", description="Outlet number(s)")
                          int... outlet) throws DriverException
    {
        if (outlet.length == 1) {
            pdu.delayedOutletOn(outlet[0]);
        }
        else {
            pdu.delayedOutletsOn(outlet);
        }
    }


    @Command(name="delayedonn", description="Turn named outlet(s) power on with delay")
    public void delayedOn(@Argument(name="outlet", description="Outlet name(s)")
                          String... outlet) throws DriverException
    {
        if (outlet.length == 1) {
            pdu.delayedOutletOn(outlet[0]);
        }
        else {
            pdu.delayedOutletsOn(outlet);
        }
    }


    @Command(name="delayedoff", description="Turn outlet(s) power off with delay")
    public void delayedOff(@Argument(name="outlet", description="Outlet number(s)")
                           int... outlet) throws DriverException
    {
        if (outlet.length == 1) {
            pdu.delayedOutletOff(outlet[0]);
        }
        else {
            pdu.delayedOutletsOff(outlet);
        }
    }


    @Command(name="delayedoffn", description="Turn named outlet(s) power off with delay")
    public void delayedOff(@Argument(name="outlet", description="Outlet name(s)")
                           String... outlet) throws DriverException
    {
        if (outlet.length == 1) {
            pdu.delayedOutletOff(outlet[0]);
        }
        else {
            pdu.delayedOutletsOff(outlet);
        }
    }


    @Command(name="inputcurrent", description="Show input current")
    public String inputCurrent() throws DriverException
    {
        StringBuilder reply = new StringBuilder();
        for (double value : pdu.readCurrent()) {
            reply.append(String.format("%.3f amps  ", value));
        }
        return reply.toString();
    }


    @Command(name="inputpower", description="Show input power")
    public String inputPower() throws DriverException
    {
        return String.format("%.2f watts", pdu.readPower());
    }


    @Command(name="inputva", description="Show input VA")
    public String inputVA() throws DriverException
    {
        return String.format("%.2f watts", pdu.readVA());
    }


    @Command(name="talk", description="Talk to the PDU interactively")
    public void talk() throws DriverException, IOException
    {
        out.println("Type ctrl-D to exit");
        reader = (reader == null) ? new ConsoleReader() : reader;
        while (true) {
            String cmnd = reader.readLine(pdu.getPrompt());
            if (cmnd == null) break;
            pdu.send(cmnd);
            String[] resp = pdu.receive();
            for (int j = 1; j < resp.length; j++) {
                out.println(resp[j]);
            }
        }
        out.println();
    }

}
