package org.lsst.ccs.drivers.apcpdu;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.SocketTimeoutException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.net.telnet.TelnetClient;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *********************************************
 *
 *  Controls an APC model 7900 power strip
 *
 *  @author Owen Saxton
 *
 *********************************************
 */
public class APC7900 {

   /**
    **  Public enumerations.
    */
    public enum ConnType {

        TELNET(null),
        FTDI(Ascii.ConnType.FTDI),
        SERIAL(Ascii.ConnType.SERIAL);

        private final Ascii.ConnType connType;

        ConnType(Ascii.ConnType connType) {
            this.connType = connType;
        }

        Ascii.ConnType getValue() {
            return connType;
        }

    }

   /**
    *  Private data.
    */
    private static final int
        DEFAULT_BAUDRATE = 9600,
        TELNET_PORT = 23,
        SOCKET_TIMEOUT = 20,
        POLL_PERIOD = 10,
        INITIAL_TIMEOUT = 2000,
        FINAL_TIMEOUT = 50;
    private static final String
        STATUS_SPLIT_RE = " *: ";
    private final Ascii asc = new Ascii(Ascii.Option.NO_NET);
    private final TelnetClient tc = new TelnetClient();
    private boolean isTelnet;
    private InputStream in;
    private OutputStream out;
    private String prompt = "";
    private String product, fwVersion;
    private Integer numPhases, numOutlets;


   /**
    *  Opens a connection and logs in.
    *
    *  @param  connType  The enumerated connection type: TELNET, FTDI or SERIAL
    *
    *  @param  ident     The host name (TELNET), USB ID (FTDI) or port name (SERIAL)
    *
    *  @param  username  The user name
    *
    *  @param  password  The password
    *
    *  @throws  DriverException
    */
    public void open(ConnType connType, String ident, String username, String password) throws DriverException
    {
        open(connType, ident);
        try {
            if (isTelnet) {
                receive();
            }
            else {
                receive("");
                if (prompt.isEmpty()) {  // Try again
                    receive("");
                }
                if (prompt.contains("APC>")) return;  // Already logged in
            }
            if (!prompt.contains("User Name :")) {
                throw new DriverException("Unrecognized login prompt: " + prompt);
            }

            receive(username);
            if (!prompt.contains("Password  :")) {
                throw new DriverException("Unrecognized password prompt: " + prompt);
            }

            String[] resp = receive(password + " -c");
            if (resp.length == 0) {
                throw new DriverException("Login failed");
            }
        }
        catch (DriverException e) {
            close();
            throw e;
        }
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The enumerated connection type: FTDI or SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @throws  DriverException
    */
    public void open(ConnType connType, String ident) throws DriverException
    {
        Ascii.ConnType cType = connType.getValue();
        if (cType != null) {
            isTelnet = false;
            asc.open(cType, ident, DEFAULT_BAUDRATE);
            asc.setTerminator("\r");
        }
        else {
            isTelnet = true;
            if (in != null) {
                throw new DriverException("Connection already open");
            }
            try {
                tc.connect(ident, TELNET_PORT);
                tc.setSoTimeout(SOCKET_TIMEOUT);
                in = tc.getInputStream();
                out = tc.getOutputStream();
            }
            catch (IOException e) {
                throw new DriverException(e);
            }
        }
    }


   /**
    *  Closes the connection
    *
    *  @throws  DriverException
    */
    public void close() throws DriverException
    {
        checkOpen();
        if (!isTelnet) {
            try {
                send("quit");
            }
            catch (DriverException e) {
            }
        }
        try {
            if (isTelnet) {
                tc.disconnect();
            }
            else {
                asc.close();
            }
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
        finally {
            prompt = "";
            in = null;
            out = null;
            product = null;
            fwVersion = null;
            numPhases = null;
            numOutlets = null;
            isTelnet = false;
        }
    }


   /**
    *  Gets the product name
    *
    *  @return  The product name
    *
    *  @throws  DriverException
    */
    public String getProductName() throws DriverException
    {
        if (product == null) {
            getVersionInfo();
        }
        return product;
    }


   /**
    *  Gets the firmware version
    *
    *  @return  The firmware version
    *
    *  @throws  DriverException
    */
    public String getFWVersion() throws DriverException
    {
        if (fwVersion == null) {
            getVersionInfo();
        }
        return fwVersion;
    }


   /**
    *  Gets the number of phases
    *
    *  @return  The number of phases
    *
    *  @throws  DriverException
    */
    public int getPhaseCount() throws DriverException
    {
        if (numPhases == null) {
            getVersionInfo();
        }
        return numPhases;
    }


   /**
    *  Gets the number of outlets
    *
    *  @return  The number of outlets
    *
    *  @throws  DriverException
    */
    public int getOutletCount() throws DriverException
    {
        if (numOutlets == null) {
            getVersionInfo();
        }
        return numOutlets;
    }


   /**
    *  Gets the name of an outlet
    *
    *  @param  outlet  The outlet number, starting from 1
    * 
    *  @return  The outlet name
    *
    *  @throws  DriverException
    */
    public String getOutletName(int outlet) throws DriverException
    {
        String[] reply = receiveString("status " + outlet);
        checkReplyLength(reply, 1);
        String[] words = reply[0].split(STATUS_SPLIT_RE);
        return words[2];
    }


   /**
    *  Sets the name of an outlet
    *
    *  @param  outlet  The outlet number, starting from 1
    * 
    *  @param  name    The outlet name
    *
    *  @throws  DriverException
    */
    public void setOutletName(int outlet, String name) throws DriverException
    {
        String[] reply = receiveString("name " + outlet + " " + name);
    }


   /**
    *  Gets an outlet number, given its name
    *
    *  @param  outlet  The outlet name
    * 
    *  @return  The outlet number
    *
    *  @throws  DriverException
    */
    public int getOutletNumber(String outlet) throws DriverException
    {
        String[] reply = receiveString("status \"" + outlet + "\"");
        checkReplyLength(reply, 1);
        String[] words = reply[0].split(STATUS_SPLIT_RE);
        int number = 0;
        try {
            number = Integer.valueOf(words[0]);
        }
        catch (NumberFormatException e) {
            responseError(reply[0]);
        }
        return number;
    }


   /**
    *  Gets the map of outlet names to numbers
    *
    *  @return  The outlet number map
    *
    *  @throws  DriverException
    */
    public Map<String, Integer> getOutletNumberMap() throws DriverException
    {
        String[] reply = receiveString("status");
        checkReplyLength(reply, getOutletCount());
        Map<String, Integer> numberMap = new LinkedHashMap<>();
        for (String line : reply) {
            String[] words = line.split(STATUS_SPLIT_RE);
            try {
                numberMap.put(words[2], Integer.valueOf(words[0]));
            }
            catch (NumberFormatException | IndexOutOfBoundsException e) {
                responseError(reply[0]);
            }
        }
        return numberMap;
    }


   /**
    *  Gets the on delay for an outlet
    *
    *  @param  outlet  The outlet number, starting from 1
    * 
    *  @return  The delay (secs) or -1 for never
    *
    *  @throws  DriverException
    */
    public int getOutletOnDelay(int outlet) throws DriverException
    {
        return extractDelay(receiveString("powerondelay " + outlet));
    }


   /**
    *  Gets the off delay for an outlet
    *
    *  @param  outlet  The outlet number, starting from 1
    * 
    *  @return  The delay (secs) or -1 for never
    *
    *  @throws  DriverException
    */
    public int getOutletOffDelay(int outlet) throws DriverException
    {
        return extractDelay(receiveString("poweroffdelay " + outlet));
    }


   /**
    *  Gets the on delay for a named outlet
    *
    *  @param  outlet  The outlet name
    * 
    *  @return  The delay (secs) or -1 for never
    *
    *  @throws  DriverException
    */
    public int getOutletOnDelay(String outlet) throws DriverException
    {
        return extractDelay(receiveString("powerondelay \"" + outlet + "\""));
    }


   /**
    *  Gets the off delay for a named outlet
    *
    *  @param  outlet  The outlet name
    * 
    *  @return  The delay (secs) or -1 for never
    *
    *  @throws  DriverException
    */
    public int getOutletOffDelay(String outlet) throws DriverException
    {
        return extractDelay(receiveString("poweroffdelay \"" + outlet + "\""));
    }


   /**
    *  Extracts the delay value from a PDU reply
    *
    *  @param  reply  The reply
    *
    *  @return  The delay (secs), or -1 for never
    *
    *  @throws  DriverException
    */
    private int extractDelay(String[] reply) throws DriverException
    {
        checkReplyLength(reply, 1);
        String[] words = reply[0].split(STATUS_SPLIT_RE);
        String[] values = words[2].split(" ");   // "Power on(off) delay is n seconds."
        int delay = 0;
        try {
            delay = values[4].equals("never.") ? -1 : Integer.valueOf(values[4]);
        }
        catch (NumberFormatException | IndexOutOfBoundsException e) {
            responseError(reply[0]);
        }
        return delay;
    }


   /**
    *  Gets the on delay for all outlets
    *
    *  @return  An array containing the delays for all outlets
    *
    *  @throws  DriverException
    */
    public int[] getOutletOnDelays() throws DriverException
    {
        return getOutletDelays(true);
    }


   /**
    *  Gets the off delay for all outlets
    *
    *  @return  An array containing the delays for all outlets
    *
    *  @throws  DriverException
    */
    public int[] getOutletOffDelays() throws DriverException
    {
        return getOutletDelays(false);
    }


   /**
    *  Gets the on or off delays for all outlets
    *
    *  @param  on  True to get on delays, false to get off delays
    *
    *  @return  The array of delays
    *
    *  @throws  DriverException
    */
    private int[] getOutletDelays(boolean on) throws DriverException
    {
        String[] reply = receiveString(on ? "powerondelay" : "poweroffdelay");
        checkReplyLength(reply, getOutletCount());
        int[] delays = new int[reply.length];
        for (int j = 0; j < reply.length; j++) {
            String[] words = reply[j].split(STATUS_SPLIT_RE);
            String[] values = words[2].split(" ");   // "Power on(off) delay is n seconds."
            try {
                delays[j] = values[4].equals("never.") ? -1 : Integer.valueOf(values[4]);
            }
            catch (NumberFormatException | IndexOutOfBoundsException e) {
                responseError(reply[j]);
            }
        }
        return delays;
    }


   /**
    *  Gets the map of outlet names to on delays
    *
    *  @return  The delay map
    *
    *  @throws  DriverException
    */
    public Map<String, Integer> getOutletOnDelayMap() throws DriverException
    {
        return getOutletDelayMap(true);
    }


   /**
    *  Gets the map of outlet names to off delays
    *
    *  @return  The delay map
    *
    *  @throws  DriverException
    */
    public Map<String, Integer> getOutletOffDelayMap() throws DriverException
    {
        return getOutletDelayMap(false);
    }


   /**
    *  Gets the map of outlet names to on or off delays
    *
    *  @param  on  True to set on delay, false to set off delay
    *
    *  @return  The delay map
    *
    *  @throws  DriverException
    */
    private Map<String, Integer> getOutletDelayMap(boolean on) throws DriverException
    {
        String[] reply = receiveString(on ? "powerondelay" : "poweroffdelay");
        checkReplyLength(reply, getOutletCount());
        Map<String, Integer> delayMap = new LinkedHashMap<>();
        for (String line : reply) {
            String[] words = line.split(STATUS_SPLIT_RE);
            String[] values = words[2].split(" ");   // "Power on delay is n seconds."
            int delay = 0;
            try {
                delay = values[4].equals("never.") ? -1 : Integer.valueOf(values[4]);
            }
            catch (NumberFormatException | IndexOutOfBoundsException e) {
                responseError(line);
            }
            delayMap.put(words[1], delay);
        }
        return delayMap;
    }


   /**
    *  Sets outlet on delay.
    *
    *  @param  delay   The delay (sec) before the outlet is powered on (-1 = never)
    * 
    *  @param  outlet  The outlet number, starting from 1
    *
    *  @throws  DriverException
    */
    public void setOutletOnDelay(int delay, int outlet) throws DriverException
    {
        receiveString("powerondelay " + outlet + ":" + delay);
    }


   /**
    *  Sets outlet off delay.
    *
    *  @param  delay   The delay (sec) before the outlet is powered off (-1 = never)
    * 
    *  @param  outlet  The outlet number, starting from 1
    *
    *  @throws  DriverException
    */
    public void setOutletOffDelay(int delay, int outlet) throws DriverException
    {
        receiveString("poweroffdelay " + outlet + ":" + delay);
    }


   /**
    *  Sets named outlet on delay.
    *
    *  @param  delay   The delay (sec) before the outlet is powered on (-1 = never)
    * 
    *  @param  outlet  The outlet name
    *
    *  @throws  DriverException
    */
    public void setOutletOnDelay(int delay, String outlet) throws DriverException
    {
        receiveString("powerondelay \"" + outlet + "\":" + delay);
    }


   /**
    *  Sets named outlet off delay.
    *
    *  @param  delay   The delay (sec) before the outlet is powered off (-1 = never)
    * 
    *  @param  outlet  The outlet name
    *
    *  @throws  DriverException
    */
    public void setOutletOffDelay(int delay, String outlet) throws DriverException
    {
        receiveString("poweroffdelay \"" + outlet + "\":" + delay);
    }


   /**
    *  Sets outlets on delay.
    *
    *  @param  delay    The delay (sec) before the outlet is powered on (-1 = never)
    * 
    *  @param  outlets  The array of outlet numbers
    *
    *  @throws  DriverException
    */
    public void setOutletOnDelay(int delay, int[] outlets) throws DriverException
    {
        receiveString("powerondelay " + makeOutletString(outlets) + ":" + delay);
    }


   /**
    *  Sets outlets off delay.
    *
    *  @param  delay    The delay (sec) before the outlet is powered off (-1 = never)
    * 
    *  @param  outlets  The array of outlet numbers
    *
    *  @throws  DriverException
    */
    public void setOutletOffDelay(int delay, int[] outlets) throws DriverException
    {
        receiveString("poweroffdelay " + makeOutletString(outlets) + ":" + delay);
    }


   /**
    *  Sets named outlets on delay.
    *
    *  @param  delay    The delay (sec) before the outlet is powered on (-1 = never)
    * 
    *  @param  outlets  The array of outlet names
    *
    *  @throws  DriverException
    */
    public void setOutletOnDelay(int delay, String[] outlets) throws DriverException
    {
        receiveString("powerondelay " + makeOutletString(outlets) + ":" + delay);
    }


   /**
    *  Sets named outlets off delay.
    *
    *  @param  delay    The delay (sec) before the outlet is powered off (-1 = never)
    * 
    *  @param  outlets  The array of outlet names
    *
    *  @throws  DriverException
    */
    public void setOutletOffDelay(int delay, String[] outlets) throws DriverException
    {
        receiveString("poweroffdelay " + makeOutletString(outlets) + ":" + delay);
    }


   /**
    *  Gets the on (powered) state of an outlet
    *
    *  @param  outlet  The outlet number, starting from 1
    * 
    *  @return  Whether the outlet is powered
    *
    *  @throws  DriverException
    */
    public boolean isOutletOn(int outlet) throws DriverException
    {
        String[] reply = receiveString("status " + outlet);
        checkReplyLength(reply, 1);
        String[] words = reply[0].split(STATUS_SPLIT_RE);
        return words[1].equals("ON");
    }


   /**
    *  Gets the on (powered) state of an outlet
    *
    *  @param  outlet  The outlet name
    * 
    *  @return  Whether the outlet is powered
    *
    *  @throws  DriverException
    */
    public boolean isOutletOn(String outlet) throws DriverException
    {
        String[] reply = receiveString("status \"" + outlet + "\"");
        checkReplyLength(reply, 1);
        String[] words = reply[0].split(STATUS_SPLIT_RE);
        return words[1].equals("ON");
    }


   /**
    *  Gets the on (powered) state of all outlets
    *
    *  @return  A boolean array containing the on state of all outlets
    *
    *  @throws  DriverException
    */
    public boolean[] getOutletOnStates() throws DriverException
    {
        String[] reply = receiveString("status");
        boolean[] onStates = new boolean[getOutletCount()];
        checkReplyLength(reply, onStates.length);
        for (int j = 0; j < onStates.length; j++) {
            String[] words = reply[j].split(STATUS_SPLIT_RE);
            onStates[j] = words[1].equals("ON");
        }
        return onStates;
    }


   /**
    *  Gets the map of outlet names to on (powered) states
    *
    *  @return  The on state map
    *
    *  @throws  DriverException
    */
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException
    {
        String[] reply = receiveString("status");
        checkReplyLength(reply, getOutletCount());
        Map<String, Boolean> stateMap = new LinkedHashMap<>();
        for (String line : reply) {
            String[] words = line.split(STATUS_SPLIT_RE);
            stateMap.put(words[2], words[1].equals("ON"));
        }
        return stateMap;
    }


   /**
    *  Turns outlet power on immediately.
    *
    *  @param  outlet  The outlet number, starting from 1
    *
    *  @return  The number of outlets turned on (1)
    *
    *  @throws  DriverException
    */
    public int setOutletOn(int outlet) throws DriverException
    {
        return receiveString("on " + outlet).length;
    }


   /**
    *  Turns outlet power off immediately.
    *
    *  @param  outlet  The outlet number, starting from 1
    *
    *  @return  The number of outlets turned off (1)
    *
    *  @throws  DriverException
    */
    public int setOutletOff(int outlet) throws DriverException
    {
        return receiveString("off " + outlet).length;
    }


   /**
    *  Turns named outlet power on immediately.
    *
    *  @param  outlet  The outlet name
    *
    *  @return  The number of outlets turned on (1)
    *
    *  @throws  DriverException
    */
    public int setOutletOn(String outlet) throws DriverException
    {
        return receiveString("on \"" + outlet + "\"").length;
    }


   /**
    *  Turns named outlet power off immediately.
    *
    *  @param  outlet  The outlet name
    *
    *  @return  The number of outlets turned off (1)
    *
    *  @throws  DriverException
    */
    public int setOutletOff(String outlet) throws DriverException
    {
        return receiveString("off \"" + outlet + "\"").length;
    }


   /**
    *  Turns outlets power on immediately.
    *
    *  @param  outlets  An array of outlet numbers
    *
    *  @return  The number of outlets turned on
    *
    *  @throws  DriverException
    */
    public int setOutletsOn(int[] outlets) throws DriverException
    {
        return receiveString("on" + makeOutletString(outlets)).length;
    }


   /**
    *  Turns outlets power off immediately.
    *
    *  @param  outlets  An array of outlet numbers
    *
    *  @return  The number of outlets turned off
    *
    *  @throws  DriverException
    */
    public int setOutletsOff(int[] outlets) throws DriverException
    {
        return receiveString("off" + makeOutletString(outlets)).length;
    }


   /**
    *  Turns named outlets power on immediately.
    *
    *  @param  outlets  An array of outlet names
    *
    *  @return  The number of outlets turned on
    *
    *  @throws  DriverException
    */
    public int setOutletsOn(String[] outlets) throws DriverException
    {
        return receiveString("on" + makeOutletString(outlets)).length;
    }


   /**
    *  Turns named outlets power off immediately.
    *
    *  @param  outlets  An array of outlet names
    *
    *  @return  The number of outlets turned off
    *
    *  @throws  DriverException
    */
    public int setOutletsOff(String[] outlets) throws DriverException
    {
        return receiveString("off" + makeOutletString(outlets)).length;
    }


   /**
    *  Turns outlet power on with delay.
    *
    *  @param  outlet  The outlet number, starting from 1
    *
    *  @return  The number of outlets turned on (1)
    *
    *  @throws  DriverException
    */
    public int delayedOutletOn(int outlet) throws DriverException
    {
        return receiveString("delayedon " + outlet).length;
    }


   /**
    *  Turns outlet power off with delay.
    *
    *  @param  outlet  The outlet number, starting from 1
    *
    *  @return  The number of outlets turned off (1)
    *
    *  @throws  DriverException
    */
    public int delayedOutletOff(int outlet) throws DriverException
    {
        return receiveString("delayedoff " + outlet).length;
    }


   /**
    *  Turns named outlet power on with delay.
    *
    *  @param  outlet  The outlet name
    *
    *  @return  The number of outlets turned on (1)
    *
    *  @throws  DriverException
    */
    public int delayedOutletOn(String outlet) throws DriverException
    {
        return receiveString("delayedon \"" + outlet + "\"").length;
    }


   /**
    *  Turns named outlet power off with delay.
    *
    *  @param  outlet  The outlet name
    *
    *  @return  The number of outlets turned off (1)
    *
    *  @throws  DriverException
    */
    public int delayedOutletOff(String outlet) throws DriverException
    {
        return receiveString("delayedoff \"" + outlet + "\"").length;
    }


   /**
    *  Turns outlets power on with delay.
    *
    *  @param  outlets  An array of outlet numbers
    *
    *  @return  The number of outlets turned on
    *
    *  @throws  DriverException
    */
    public int delayedOutletsOn(int[] outlets) throws DriverException
    {
        return receiveString("delayedon" + makeOutletString(outlets)).length;
    }


   /**
    *  Turns outlets power off with delay.
    *
    *  @param  outlets  An array of outlet numbers
    *
    *  @return  The number of outlets turned off
    *
    *  @throws  DriverException
    */
    public int delayedOutletsOff(int[] outlets) throws DriverException
    {
        return receiveString("delayedoff" + makeOutletString(outlets)).length;
    }


   /**
    *  Turns named outlets power on with delay.
    *
    *  @param  outlets  An array of outlet names
    *
    *  @return  The number of outlets turned on
    *
    *  @throws  DriverException
    */
    public int delayedOutletsOn(String[] outlets) throws DriverException
    {
        return receiveString("delayedon" + makeOutletString(outlets)).length;
    }


   /**
    *  Turns named outlets power off with delay.
    *
    *  @param  outlets  An array of outlet names
    *
    *  @return  The number of outlets turned off
    *
    *  @throws  DriverException
    */
    public int delayedOutletsOff(String[] outlets) throws DriverException
    {
        return receiveString("delayedoff" + makeOutletString(outlets)).length;
    }


   /**
    *  Reads input currents, one per phase
    * 
    *  @return  The array of input currents (amps)
    *
    *  @throws  DriverException
    */
    public double[] readCurrent() throws DriverException
    {
        String[] reply = receiveString("current");
        checkReplyLength(reply, getPhaseCount());
        double[] current = new double[reply.length];
        for (int j = 0; j < current.length; j++) {
            try {
                String token = reply[j].split(": ")[1];
                current[j] = Double.valueOf(token.substring(0, token.length() - 1));
            }
            catch (IndexOutOfBoundsException | NumberFormatException e) {
                responseError(reply[j]);
            }
        }
        return current;
    }


   /**
    *  Reads input VA
    * 
    *  @return  The VA (watts)
    *
    *  @throws  DriverException
    */
    public double readVA() throws DriverException
    {
        return readPowers()[0];
    }


   /**
    *  Reads input power
    * 
    *  @return  The power (watts)
    *
    *  @throws  DriverException
    */
    public double readPower() throws DriverException
    {
        return readPowers()[1];
    }


   /**
    *  Reads input VA and power
    * 
    *  @return  The array of input power values, VA & power (watts)
    *
    *  @throws  DriverException
    */
    public double[] readPowers() throws DriverException
    {
        String[] reply = receiveString("power");
        checkReplyLength(reply, 2);
        double[] values = new double[2];
        for (int j = 0; j < 2; j++) {
            try {
                values[j] = Double.valueOf(reply[j].split(" ")[0]);
            }
            catch (IndexOutOfBoundsException | NumberFormatException e) {
                responseError(reply[j]);
            }
        }
        return values;
    }


   /**
    *  Sends a command and receives a string array
    *
    *  @param  command  The command to send
    *
    *  @return  The string array
    *
    *  @throws  DriverException
    */
    public synchronized String[] receiveString(String command) throws DriverException
    {
        String[] resp = receive(command);
        if (resp.length < 2) {
            throw new DriverException("No response received");
        }
        if (!resp[1].equals(" OK")) {
            throw new DriverException("Command error" + resp[1]);
        }
        if (resp.length == 3) {
            int posn = resp[2].startsWith(" E1") ? 1 : resp[2].startsWith("E1") ? 0 : -1;
            if (posn >= 0) {
                throw new DriverException("Command error " + resp[2].substring(posn));
            }
        }
        String[] reply = new String[resp.length - 2];
        for (int j = 0; j < reply.length; j++) {
            reply[j] = resp[j + 2].substring(1);
        }
        return reply;
    }


   /**
    *  Sends a command
    *
    *  @param  command  The command to send
    *
    *  @throws  DriverException
    */
    public synchronized void send(String command) throws DriverException
    {
        checkOpen();
        if (isTelnet) {
            try {
                out.write((command + "\r").getBytes());
                out.flush();
            }
            catch (IOException e) {
                throw new DriverException(e);
            }
        }
        else {
            asc.write(command);
        }
    }


   /**
    *  Receives a response
    *
    *  @return  An array of strings each of which is a line of the response.
    *           The first element is the echoed command.
    *
    *  @throws  DriverException
    */
    public synchronized String[] receive() throws DriverException
    {
        checkOpen();
        List<byte[]> buffs = new ArrayList<>();
        int lastLeng = 0;
        int timeout = INITIAL_TIMEOUT;
        while (true) {
            byte[] buff = new byte[1024];
            int offs = 0, leng;
            while (offs < buff.length) {
                try {
                    leng = read(buff, offs, buff.length - offs, timeout);
                    //System.out.println("Read length = " + leng);
                }
                catch (DriverTimeoutException e) {
                    if (offs == 0 && buffs.isEmpty()) {
                        throw e;
                    }
                    else break;
                }
                timeout = FINAL_TIMEOUT;
                offs += leng;
            }
            if (offs > 0) {
                buffs.add(buff);
                lastLeng = offs;
            }
            if (offs < buff.length) break;
        }
        List<String> resp = new ArrayList<>();
        String line = "";
        boolean gotCr = false;
        for (int bufNo = 0; bufNo < buffs.size(); bufNo++) {
            byte[] buff = buffs.get(bufNo);
            int leng = (bufNo == buffs.size() - 1) ? lastLeng : buff.length;
            int start = 0;
            for (int posn = 0; posn < leng; posn++) {
                if (buff[posn] == '\r') {
                    if (!gotCr) {
                        line += new String(buff, start, posn - start);
                        resp.add(line);
                        line = "";
                    }
                    start = posn + 1;
                    gotCr = true;
                }
                else {
                    if (buff[posn] == '\n' && gotCr) {
                        start = posn + 1;
                    }
                    gotCr = false;
                }
            }
            line += new String(buff, start, leng - start);
        }
        prompt = line;

        return resp.toArray(new String[0]);
    }


   /**
    *  Sends a command and receives the response
    *
    *  @param  command  The command to send
    *
    *  @return  An array of strings each of which is a line of the response.
    *           The first element is the echoed command.
    *
    *  @throws  DriverException
    */
    public synchronized String[] receive(String command) throws DriverException
    {
        send(command);
        return receive();
    }


   /**
    *  Gets the prompt
    *
    *  @return  The saved prompt
    */
    public String getPrompt()
    {
        return prompt;
    }


   /**
    *  Gets the version information
    *
    *  @return  The saved prompt
    */
    private void getVersionInfo() throws DriverException
    {
        String[] reply = receiveString("ver");
        checkReplyLength(reply, 6);
        fwVersion = reply[0];
        product = reply[2].split(": ")[1];
        try {
            numOutlets = Integer.decode(reply[3].split(": ")[1]);
        }
        catch (NumberFormatException e) {
            throw new DriverException("Invalid outlets line: " + reply[3]);
        }
        String inpType = reply[5].split(": ")[1];
        numPhases = inpType.equals("Single") ? 1 : 3;
    }


   /**
    *  Reads available data.
    * 
    *  A polling loop is used because the socket timeout mechanism was
    *  found to be unreliable.
    *
    *  @param  buff     A buffer to receive the data
    *
    *  @param  offset   The offset to the first available byte in buff
    *
    *  @param  mleng    The maximum number of bytes to read
    *
    *  @param  timeout  The read timeout (ms)
    *
    *  @return  The number of bytes read
    *
    *  @throws  DriverException
    */
    private int read(byte[] buff, int offset, int mleng, int timeout)
        throws DriverException
    {
        if (isTelnet) {
            long endTime = System.currentTimeMillis() + timeout;
            int count = Math.min(mleng, buff.length - offset);
            int leng = 0;
            while (leng == 0 && System.currentTimeMillis() < endTime) {
                try {
                    leng = in.read(buff, offset, count);
                }
                catch (SocketTimeoutException e) {
                }
                catch (IOException e) {
                    throw new DriverException(e);
                }
                if (leng < 0) {
                    throw new DriverException("Connection closed remotely");
                }
                try {
                    Thread.sleep(POLL_PERIOD);
                }
                catch (InterruptedException e) {
                }
            }
            if (leng == 0) {
                throw new DriverTimeoutException("Read timed out");
            }
            return leng;
        }
        else {
            asc.setTimeout(timeout);
            return asc.readBytes(buff, offset, mleng);
        }
    }


   /**
    *  Checks whether the connection is open
    *
    *  @throws  DriverException if not
    */
    private void checkOpen() throws DriverException
    {
        if (isTelnet && in == null) {
            throw new DriverException("Connection not open");
        }
    }


   /**
    *  Generates an outlet specification string
    *
    *  @param  outlets  The array of outlet numbers
    *
    *  @return  The generated outlet string
    *
    *  @throws  DriverException if invalid
    */
    private String makeOutletString(int[] outlets)
    {
        StringBuilder outString = new StringBuilder();
        char sep = ' ';
        for (int outlet : outlets) {
            outString = outString.append(sep).append(outlet);
            sep = ',';
        }
        return outString.toString();
    }


   /**
    *  Generates an outlet specification string
    *
    *  @param  outlets  The array of outlet names
    *
    *  @return  The generated outlet string
    *
    *  @throws  DriverException if invalid
    */
    private String makeOutletString(String[] outlets)
    {
        StringBuilder outString = new StringBuilder();
        char sep = ' ';
        for (String outlet : outlets) {
            outString = outString.append(sep).append('"').append(outlet).append('"');
            sep = ',';
        }
        return outString.toString();
    }


   /**
    *  Checks whether a reply size is valid
    *
    *  @param  reply  The array of reply lines
    *
    *  @param  count  The expected line count
    *
    *  @throws  DriverException if unequal
    */
    private void checkReplyLength(String[] reply, int count) throws DriverException
    {
        if (reply.length == count) return;
        if (reply.length == 0) {
            throw new DriverException("Missing response");
        }
        else {
            throw new DriverException("Incorrect response length: " + reply.length + "; first line = " + reply[0]); 
        }
    }


   /**
    *  Generates an invalid response exception
    *
    *  @param  resp  The response string
    *
    *  @throws  DriverException if unequal
    */
    private void responseError(String resp) throws DriverException
    {
        throw new DriverException("Incorrect response: " + resp);
    }

}
