package org.lsst.ccs.drivers.ametek;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Driver for controlling an Ametek AVC cryocooler
 *
 *  @author Owen Saxton
 */
public class AvcCooler extends Ascii {

    /**
     *  Public constants
     */
    public enum Mode {OFF, ON, POWER}

    /**
     *  Private data.
     */
    private static final Map<String, Mode> modeMap = new HashMap<>();
    static {
        for (Mode mode : Mode.values()) {
            modeMap.put(mode.name(), mode);
        }
    }
    private static final int
        DEFAULT_BAUDRATE = 9600;


    /**
     *  Constructor
     */
    public AvcCooler()
    {
        setOptions(Option.NO_NET);
        setDefaultParm(DEFAULT_BAUDRATE);
    }


    /**
     *  Opens a connection.
     *
     *  @param  connType  The enumerated connection type: FTDI or SERIAL
     *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
     *  @param  baudRate  The baud rate
     *  @param  dataChar  The data characteristics
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident, int baudRate, int dataChar) throws DriverException
    {
        super.open(connType, ident, baudRate, dataChar);
        setCommandTerm(Terminator.CR);
        setResponseTerm(Terminator.CRLF);
    }


    /**
     *  Gets the product type
     *
     *  @return  The product type
     *  @throws  DriverException
     */
    public String getProductType() throws DriverException
    {
        return readString("MODE");
    }


    /**
     *  Gets the firmware version
     *
     *  @return  The firmware version
     *  @throws  DriverException
     */
    public String getFWVersion() throws DriverException
    {
        return readString("VERSION");
    }


    /**
     *  Gets the sensor type.
     *
     *  @return  The sensor type
     *  @throws  DriverException
     */
    public String getSensor() throws DriverException
    {
        return readString("SENSOR");
    }


    /**
     *  Sets the user lock state.
     *
     *  @param  passwd  The password
     *  @throws  DriverException
     */
    public void lock(String passwd) throws DriverException
    {
        readDouble("LOGOUT=" + passwd);
    }


    /**
     *  Clears the user lock state.
     *
     *  @param  passwd  The password
     *  @throws  DriverException
     */
    public void unlock(String passwd) throws DriverException
    {
        readDouble("LOGIN=" + passwd);
    }


    /**
     *  Gets the user lock state.
     *
     *  @return  Whether the user lock is in effect
     *  @throws  DriverException
     */
    public boolean isLocked() throws DriverException
    {
        return readDouble("LOGIN") == 1.0;
    }


    /**
     *  Sets the user password.
     *
     *  @param  passwd  The new password
     *  @throws  DriverException
     */
    public void setPassword(String passwd) throws DriverException
    {
        readDouble("PASSWD=" + passwd);
    }


    /**
     *  Sets the cooler mode.
     *
     *  @param  mode  The mode to set: OFF, ON, POWER
     *  @throws  DriverException
     */
    public void setMode(Mode mode) throws DriverException
    {
        String resp = readString("COOLER=" + mode);
        Mode retMode = modeMap.get(resp);
        if (retMode == null || retMode != mode) {
            unexpectedResponse(resp);
        }
    }


    /**
     *  Gets the cooler mode.
     *
     *  @return  The mode: OFF, ON, POWER
     *  @throws  DriverException
     */
    public Mode getMode() throws DriverException
    {
        String resp = readString("COOLER");
        Mode mode = modeMap.get(resp);
        if (mode == null) {
            unexpectedResponse(resp);
        }
        return mode;
    }


    /**
     *  Sets the target temperature.
     *
     *  @param  temp  The target temperature
     *  @throws  DriverException
     */
    public void setTemperature(double temp) throws DriverException
    {
        readDouble("TTARGET=" + temp);
    }


    /**
     *  Gets the target temperature.
     *
     *  @return  The target temperature
     *  @throws  DriverException
     */
    public double getTemperature() throws DriverException
    {
        return readDouble("TTARGET");
    }


    /**
     *  Sets the commanded power.
     *
     *  @param  power  The power value
     *  @throws  DriverException
     */
    public void setPower(double power) throws DriverException
    {
        readDouble("PWOUT=" + power);
    }


    /**
     *  Gets the commanded power.
     *
     *  @return  The power value
     *  @throws  DriverException
     */
    public double getPower() throws DriverException
    {
        return readDouble("PWOUT");
    }


    /**
     *  Gets all the commanded power values.
     *
     *  @return  The three-element power values array
     *  @throws  DriverException
     */
    public double[] getAllPower() throws DriverException
    {
        String[] resp = readLines("E", 3);
        double[] values = new double[3];
        for (int j = 0; j < values.length; j++) {
            try {
                values[j] = Double.valueOf(resp[j]);
            }
            catch (NumberFormatException e) {
                throw new DriverException("Invalid numeric value: " + resp[j]);
            }
        }
        return values;
    }


    /**
     *  Reads the temperature
     *
     *  @return  The temperature
     *  @throws  DriverException
     */
    public double readTemperature() throws DriverException
    {
        return readDouble("TC");
    }


    /**
     *  Reads the reject temperature
     *
     *  @return  The reject temperature
     *  @throws  DriverException
     */
    public double readRejectTemp() throws DriverException
    {
        return readDouble("TEMP RJ");
    }


    /**
     *  Reads the power.
     *
     *  @return  The power
     *  @throws  DriverException
     */
    public double readPower() throws DriverException
    {
        return readDouble("P");
    }


    /**
     *  Gets the error word.
     *
     *  @return  The error word
     *  @throws  DriverException
     */
    public int getError() throws DriverException
    {
        String resp = readString("ERROR");
        if (!resp.matches("[01][01][01][01][01][01][01][01]")) {
            unexpectedResponse(resp);
        }
        int value = 0;
        for (int j = 0; j < 8; j++) {
            if (resp.charAt(j) == '1') {
                value |= (1 << (7 - j));
            }
        }
        return value;
    }


    /**
     *  Sets the proportional control constant.
     *
     *  @param  value  The proportional value
     *  @throws  DriverException
     */
    public void setProportional(double value) throws DriverException
    {
        readDouble("KP=" + value);
    }


    /**
     *  Gets the proportional control constant.
     *
     *  @return  The proportional value
     *  @throws  DriverException
     */
    public double getProportional() throws DriverException
    {
        return readDouble("KP");
    }


    /**
     *  Sets the integral control constant.
     *
     *  @param  value  The integral value
     *  @throws  DriverException
     */
    public void setIntegral(double value) throws DriverException
    {
        readDouble("KI=" + value);
    }


    /**
     *  Gets the integral control constant.
     *
     *  @return  The integral value
     *  @throws  DriverException
     */
    public double getIntegral() throws DriverException
    {
        return readDouble("KI");
    }


    /**
     *  Sets the derivative control constant.
     *
     *  @param  value  The derivative value
     *  @throws  DriverException
     */
    public void setDerivative(double value) throws DriverException
    {
        readDouble("KD=" + value);
    }


    /**
     *  Gets the derivative control constant.
     *
     *  @return  The derivative value
     *  @throws  DriverException
     */
    public double getDerivative() throws DriverException
    {
        return readDouble("KD");
    }


    /**
     *  Sends a command and reads the response
     *
     *  @param  command  The command to send
     *  @param  nLines   The expected number of response lines
     *  @return  The array of response lines
     *  @throws  DriverException
     */
    public String[] readLines(String command, int nLines) throws DriverException
    {
        flush();
        String echo = read(command);
        if (!echo.equals(command)) {
            throw new DriverException("Unexpected command echo: " + echo);
        }
        String resp = read();
        if (resp.toLowerCase().startsWith("undefined")) {
            throw new DriverException("Undefined command");
        }
        if (resp.toLowerCase().contains("locked")) {
            throw new DriverException("Command is user locked");
        }
        String[] lines = new String[nLines];
        lines[0] = resp;
        for (int j = 1; j < nLines; j++) {
            lines[j] = read();
        }
        return lines;
    }


    /**
     *  Sends a command and reads a string
     *
     *  @param  command  The command to send
     *  @return  The string
     *  @throws  DriverException
     */
    public String readString(String command) throws DriverException
    {
        String[] lines = readLines(command, 1);
        return lines[0];
    }


    /**
     *  Sends a command and reads a double value
     *
     *  @param  command  The command to send
     *  @return  The double value
     *  @throws  DriverException
     */
    public double readDouble(String command) throws DriverException
    {
        String resp = readString(command);
        try {
            return Double.valueOf(resp);
        }
        catch (NumberFormatException e) {
            throw new DriverException("Invalid numeric value: " + resp);
        }
    }


    /**
     *  Generates an "unexpected response" exception
     *
     *  @param  resp  The response string
     *  @throws  DroverException
     */
    private static void unexpectedResponse(String resp) throws DriverException
    {
        throw new DriverException("Unexpected response: " + resp);
    }

}
