
package org.lsst.ccs.drivers.aiousb;

import java.util.Set;
import java.util.EnumSet;
import AIOUSB.AIOUSB;
import AIOUSB.AIODeviceInfo;
import AIOUSB.DIOBuf;
import AIOUSB.ResultCode;
import static AIOUSB.AIOUSB.AIODeviceInfoGet;
import static AIOUSB.AIOUSB.DIO_ConfigureWithDIOBuf;
import static AIOUSB.AIOUSB.DIO_ReadAllToDIOBuf;
import static AIOUSB.AIOUSB_BOOL_VAL.AIOUSB_FALSE;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.BUFFER_ZERO;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.DEVICE_INDEX;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.FCSLOG;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.LOAD_FAILURE;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.LOAD_SUCCESS;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.OUTPUT_MASK;
import org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.AF0;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.AF0b;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.AF1;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.AF1b;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.AF3;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.AF3b;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.AP2;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.AP2b;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.ENG;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.ENGb;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.LOCKOUT;
import static org.lsst.ccs.drivers.aiousb.USB_DIO_96_Utils.Loader_Signal.LOCKOUTb;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *
 * A class to :
 * <UL>
 * <LI>load library libAIOUSB64.so
 * <LI>check if a device USB_DIO_96 is installed and connected on the HCU.
 * <LI>emulate PLC signals for loader which say there is no filter at HANDOFF.
 * <LI>emulate PLC signals for loader which say that a filter is at HANDOFF and
 * held.
 * </UL>
 *
 * @author virieux
 */
public class USB_DIO_96 implements USB_DIO_96_Interface {

    DIOBuf buffer;

    public void connect() throws DriverException {
        boolean success = false;

        FCSLOG.info("Loading AIOUSB64");
        long load_result = AIOUSB.AIOUSB_Init();
        if (load_result == 0) {
            FCSLOG.info(LOAD_SUCCESS);
        } else {
            throw new DriverException(LOAD_FAILURE);
        }

        FCSLOG.info("Checking AIOUSB devices");
        long check_result = AIOUSB.AIOUSB_ListDevices();
        FCSLOG.info("Result code is " + check_result);

        if (0 == ResultCode.AIOUSB_SUCCESS.swigValue()) {
            FCSLOG.info(ResultCode.AIOUSB_SUCCESS);
            success = true;
        } else if (4 == ResultCode.AIOUSB_ERROR_FILE_NOT_FOUND.swigValue()) {
            throw new DriverException(ResultCode.AIOUSB_ERROR_FILE_NOT_FOUND.name());
        }

        if (success) {
            this.buffer = new DIOBuf(BUFFER_ZERO);
            AIODeviceInfo obj = AIODeviceInfoGet(DEVICE_INDEX);
            FCSLOG.info("device name : " + obj.getName());
        }
    }

    /**
     * Read values from the device and update buffer
     * 
     * @return result code of operation
     */
    public long readAllDIO() {
        return DIO_ReadAllToDIOBuf(DEVICE_INDEX, this.buffer);
    }

    /**
     * Write the buffer content on the device
     */
    public void write2AllDIO() {
        DIO_ConfigureWithDIOBuf(DEVICE_INDEX, (short) AIOUSB_FALSE.swigValue(), OUTPUT_MASK, this.buffer);
    }

    /**
     * Simulate that there is no filter at HANDOFF.
     * 
     * AP2 = 1 and AP2b = 0
     * AF0 = 1 and AF0b = 0
     * AF1 = 0 and AF1b = 1
     * AF3 = 0 and AF3b = 1
     * LOCKOUT = 0 and LOCKOUTb = 1     * ENG = 0 and ENGb = 1
     */
    @Override
    public void simulateAutochangerEmpty() {
        FCSLOG.info("==>simulateAutochangerEmpty");
        Loader_Signal[] signalList0 = new Loader_Signal[]{AP2b, AF0b, LOCKOUT, AF1, AF3, ENG};
        setListOfSignals(signalList0, (short) 0);
        Loader_Signal[] signalList1 = new Loader_Signal[]{AP2, AF0, LOCKOUTb, AF1b, AF3b, ENGb};
        setListOfSignals(signalList1, (short) 1);
    }

    /**
     * Simulate that a filter is at HANDOFF and is held.
     * 
     * AP2 = 1 and AP2b = 0
     * AF3 = 1 and AF3b = 0
     * AF0 = 0 and AF0b = 1
     * AF1 = 0 and AF1b = 1
     * LOCKOUT = 0 and LOCKOUTb = 1 ENG = 0 and ENGb = 1
     **/
    @Override
    public void simulateAutochangerHoldFilter() {
        FCSLOG.info("==>simulateAutochangerHoldFilter");
        Loader_Signal[] signalList0 = new Loader_Signal[]{AP2b, AF0, LOCKOUT, AF1, AF3b, ENG};
        setListOfSignals(signalList0, (short) 0);
        Loader_Signal[] signalList1 = new Loader_Signal[]{AP2, AF0b, LOCKOUTb, AF1b, AF3, ENGb};
        setListOfSignals(signalList1, (short) 1);
    }

    @Override
    public void simulateLockout() {
        FCSLOG.info("==>simulateLockout");
        Loader_Signal[] signalList1 = new Loader_Signal[]{LOCKOUT};
        setListOfSignals(signalList1, (short) 1);
        Loader_Signal[] signalList0 = new Loader_Signal[]{LOCKOUTb};
        setListOfSignals(signalList0, (short) 0);
    }

    @Override
    public void simulateNoLockout() {
        FCSLOG.info("==>simulateLockout");
        Loader_Signal[] signalList1 = new Loader_Signal[]{LOCKOUTb};
        setListOfSignals(signalList1, (short) 1);
        Loader_Signal[] signalList0 = new Loader_Signal[]{LOCKOUT};
        setListOfSignals(signalList0, (short) 0);
    }

    /**
     * Set a same value in buffer for a list of signals.
     *
     * @param listOfSignals list of signals to set
     * @param valueToSet    new value to be set for signals
     */
    private void setListOfSignals(Loader_Signal[] listOfSignals, short valueToSet) {
        for (Loader_Signal signal : listOfSignals) {
            buffer.set(signal.getBuffer_index(), valueToSet);
            write2AllDIO();
        }
    }

    @Override
    public String toString() {
        Set<Loader_Signal> l_sig = EnumSet.allOf(Loader_Signal.class);
        StringBuilder sb = new StringBuilder("Loader Signals read on card USB_DIO_96 :\n");
        int i = 1;
        for (Loader_Signal sig : l_sig) {
            sb.append(sig.name());
            sb.append(" = ");
            sb.append(buffer.get(sig.getBuffer_index()));
            String sep = i % 2 == 0 ? "\n" : " ";
            sb.append(sep);
            i++;
        }
        return sb.toString();
    }

}
