package org.lsst.ccs.drivers.agilent;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 *  Routines for controlling an Agilent N6700 series modular power supply
 *
 *  @author  Owen Saxton
 */
public class N6700 extends Scpi implements PowerSupplyDriver {

    public static final int
        VOLTAGE_MODE     = 0,
        CURRENT_MODE     = 1,
        CHAN_INFO_MODEL  = 0,
        CHAN_INFO_OPTION = 1,
        CHAN_INFO_SERIAL = 2,
        MIN_CHANNEL      = 1,
        MAX_CHANNEL      = 4,
        NUM_CHANNELS     = 4,
        DEFAULT_PORT     = 5025;

    private int mode = VOLTAGE_MODE;


   /**
    *  Constructor.
    */
    public N6700() {
        setOptions(Option.NO_SERIAL);   // Disallow serial connections
        setDefaultPort(DEFAULT_PORT);
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The connection type (must be NET)
    *  @param  hostname  The network host name
    *  @param  port      The network port number, or 0 for the default (5025)
    *  @param  parm2     The second parameter (not used)
    *  @throws  DriverException
    */
    @Override
    public void open(ConnType connType, String hostname, int port, int parm2)
        throws DriverException
    {
        super.open(connType, hostname, port, parm2);
        try {
            checkIdentification("Agilent", CHECK_STARTS_WITH, "N67", CHECK_STARTS_WITH);
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }


   /**
    *  Opens a connection.
    *
    *  @param  hostname  The network host name
    *  @throws  DriverException
    */
    public void open(String hostname) throws DriverException
    {
        open(hostname, 0);
    }


   /**
    *  Opens a connection.
    *
    *  @param  hostname  The network host name
    *  @param  port      The network port number
    *  @throws  DriverException
    */
    public void open(String hostname, int port) throws DriverException
    {
        open(ConnType.NET, hostname, port);
    }


   /**
    *  Turns output on or off.
    *
    *  @param  on    Turn output on if true, off if false
    *  @param  chan  The channel number
    *  @throws  DriverException
    */
    @Override
    public void setOutput(boolean on, int chan) throws DriverException
    {
        writeAg("OUTP " + (on ? "ON" : "OFF"), chan);
    }


   /**
    *  Turns outputs on or off.
    *
    *  @param  on    Turn outputs on if true, off if false
    *  @param  chan  The array of channel numbers
    *  @throws  DriverException
    */
    public void setOutput(boolean on, int[] chan) throws DriverException
    {
        writeAg("OUTP " + (on ? "ON" : "OFF"), chan);
    }


   /**
    *  Gets the output state.
    *
    *  @param  chan  The channel number
    *  @return  The output-on state: true or false
    *  @throws  DriverException
    */
    @Override
    public boolean getOutput(int chan) throws DriverException
    {
        return readIntegerAg("OUTP?", chan) != 0;
    }


   /**
    *  Gets output states.
    *
    *  @param  chan  The array of channel numbers
    *  @return  The array of output-on states
    *  @throws  DriverException
    */
    public boolean[] getOutput(int[] chan) throws DriverException
    {
        int[] iState = readIntegerAg("OUTP?", chan);
        boolean[] state = new boolean[iState.length];
        for (int j = 0; j < state.length; j++) {
            state[j] = iState[j] != 0;
        }

        return state;
    }


   /**
    *  Sets the output-on delay.
    *
    *  @param  time  The delay time to set
    *  @param  chan  The channel number
    *  @throws  DriverException
    */
    @Override
    public void setOnDelay(double time, int chan) throws DriverException
    {
        writeAg("OUTP:DEL:RISE " + time, chan);
    }
    

   /**
    *  Gets the output-on delay.
    *
    *  @param  chan  The channel number
    *  @return  The delay time
    *  @throws  DriverException
    */
    public double getOnDelay(int chan) throws DriverException
    {
        return readDoubleAg("OUTP:DEL:RISE?", chan);
    }
    

   /**
    *  Gets output-on delays.
    *
    *  @param  chan  The array of channel numbers
    *  @return  The array of delay times
    *  @throws  DriverException
    */
    public double[] getOnDelay(int[] chan) throws DriverException
    {
        return readDoubleAg("OUTP:DEL:RISE?", chan);
    }
    

   /**
    *  Sets the output-off delay.
    *
    *  @param  time  The delay time to set
    *  @param  chan  The channel number
    *  @throws  DriverException
    */
    @Override
    public void setOffDelay(double time, int chan) throws DriverException
    {
        writeAg("OUTP:DEL:FALL " + time, chan);
    }


   /**
    *  Gets the output-off delay.
    *
    *  @param  chan  The channel number
    *  @return  The delay time
    *  @throws  DriverException
    */
    public double getOffDelay(int chan) throws DriverException
    {
        return readDoubleAg("OUTP:DEL:FALL?", chan);
    }
    

   /**
    *  Gets output-off delays.
    *
    *  @param  chan  The array of channel numbers
    *  @return  The array of delay times
    *  @throws  DriverException
    */
    public double[] getOffDelay(int[] chan) throws DriverException
    {
        return readDoubleAg("OUTP:DEL:FALL?", chan);
    }
    

   /**
    *  Sets the voltage.
    *
    *  @param  value  The voltage to set
    *  @param  chan   The channel number
    *  @throws  DriverException
    */
    @Override
    public void setVoltage(double value, int chan) throws DriverException
    {
        writeAg("VOLT " + value, chan);
    }


   /**
    *  Gets the set voltage.
    *
    *  @param  chan  The channel number
    *  @return  The set voltage
    *  @throws  DriverException
    */
    @Override
    public double getVoltage(int chan) throws DriverException
    {
        return readDoubleAg("VOLT?", chan);
    }


   /**
    *  Gets set voltages.
    *
    *  @param  chan  The array of channel numbers
    *  @return  The array of set voltages
    *  @throws  DriverException
    */
    public double[] getVoltage(int[] chan) throws DriverException
    {
        return readDoubleAg("VOLT?", chan);
    }


   /**
    *  Reads the voltage.
    *
    *  @param  chan  The channel number
    *  @return  The measured voltage
    *  @throws  DriverException
    */
    @Override
    public double readVoltage(int chan) throws DriverException
    {
        return readDoubleAg("MEAS:VOLT?", chan);
    }


   /**
    *  Reads voltages.
    *
    *  @param  chan  The array of channel numbers
    *  @return  The array of measured voltages
    *  @throws  DriverException
    */
    public double[] readVoltage(int[] chan) throws DriverException
    {
        return readDoubleAg("MEAS:VOLT?", chan);
    }


   /**
    *  Sets the current.
    *
    *  @param  value  The current to set
    *  @param  chan   The channel number
    *  @throws  DriverException
    */
    @Override
    public void setCurrent(double value, int chan) throws DriverException
    {
        writeAg("CURR " + value, chan);
    }


   /**
    *  Gets the set current.
    *
    *  @param  chan  The channel number
    *  @return  The set current
    *  @throws  DriverException
    */
    @Override
    public double getCurrent(int chan) throws DriverException
    {
        return readDoubleAg("CURR?", chan);
    }


   /**
    *  Gets the set currents.
    *
    *  @param  chan  The array of channel numbers
    *  @return  The array of set currents
    *  @throws  DriverException
    */
    public double[] getCurrent(int[] chan) throws DriverException
    {
        return readDoubleAg("CURR?", chan);
    }


   /**
    *  Reads the current.
    *
    *  @param  chan  The channel number
    *  @return  The measured current
    *  @throws  DriverException
    */
    @Override
    public double readCurrent(int chan) throws DriverException
    {
        return readDoubleAg("MEAS:CURR?", chan);
    }


   /**
    *  Reads currents.
    *
    *  @param  chan  The array of channel numbers
    *  @return  The array of measured currents
    *  @throws  DriverException
    */
    public double[] readCurrent(int[] chan) throws DriverException
    {
        return readDoubleAg("MEAS:CURR?", chan);
    }


   /**
    *  Gets channel information.
    *
    *  @param  chan  The channel number
    *  @return  The three-element array of information: model number, options,
    *           serial number
    *  @throws  DriverException
    */
    public String[] getChannelInfo(int chan) throws DriverException
    {
        String[] data = new String[3];
        data[CHAN_INFO_MODEL]  = readStringAg("SYST:CHAN:MODEL?", chan);
        data[CHAN_INFO_OPTION] = readStringAg("SYST:CHAN:OPTION?", chan);
        data[CHAN_INFO_SERIAL] = readStringAg("SYST:CHAN:SERIAL?", chan);

        return data;
    }


   /**
    *  Sets the priority mode (voltage or current).
    *
    *  @param  vctype  The mode to set
    *  @param  chan    The channel number
    *  @throws  DriverException
    */
    public void setPriorityMode(int vctype, int chan) throws DriverException
    {
        writeAg("FUNC " + (vctype == VOLTAGE_MODE ? "VOLT" : "CURR"), chan);
        mode = vctype;
    }


   /**
    *  Sets the slew rate.
    *
    *  @param  rate  The slew rate to set
    *  @param  chan  The channel number
    *  @throws  DriverException
    */
    public void setSlewRate(double rate, int chan) throws DriverException
    {
        writeAg((mode == VOLTAGE_MODE ? "VOLT" : "CURR") + ":SLEW " + rate,
                chan);
    }


   /**
    *  Sets the voltage limit.
    *
    *  @param  limit  The limit to set
    *  @param  chan   The channel number
    *  @throws  DriverException
    */
    public void setVoltageLimit(double limit, int chan) throws DriverException
    {
        writeAg("VOLT:LIM " + limit, chan);
    }


   /**
    *  Gets the voltage limit.
    *
    *  @param  chan  The channel number
    *  @return  The set limit
    *  @throws  DriverException
    */
    public double getVoltageLimit(int chan) throws DriverException
    {
        return readDoubleAg("VOLT:LIM?", chan);
    }


   /**
    *  Sets the current limit.
    *
    *  @param  limit  The limit to set
    *  @param  chan   The channel number
    *  @throws  DriverException
    */
    public void setCurrentLimit(double limit, int chan) throws DriverException
    {
        writeAg("CURR:LIM " + limit, chan);
    }


   /**
    *  Gets the current limit.
    *
    *  @param  chan  The channel number
    *  @return  The set limit
    *  @throws  DriverException
    */
    public double getCurrentLimit(int chan) throws DriverException
    {
        return readDoubleAg("CURR:LIM?", chan);
    }


   /**
    *  Writes a command.
    */
    private void writeAg(String instr, int chan) throws DriverException
    {
        writeCommand(instr + "," + makeChannelList(chan));
    }


   /**
    *  Writes a command.
    */
    private void writeAg(String instr, int[] chan) throws DriverException
    {
        if (chan.length == 0) return;
        writeCommand(instr + "," + makeChannelList(chan));
    }


   /**
    *  Writes a command and reads the (string) result.
    */
    private String readStringAg(String instr, int chan) throws DriverException
    {
        return readString(instr + makeChannelList(chan));
    }


   /**
    *  Writes a command and reads the (string) result.
    */
    private String[] readStringAg(String instr, int[] chan)
        throws DriverException
    {
        if (chan.length == 0) return new String[0];
        return readStringArray(instr + makeChannelList(chan));
    }


   /**
    *  Writes a command and reads the double result.
    */
    private double readDoubleAg(String instr, int chan) throws DriverException
    {
        return readDouble(instr + makeChannelList(chan));
    }


   /**
    *  Writes a command and reads the double result.
    */
    private double[] readDoubleAg(String instr, int[] chan)
        throws DriverException
    {
        if (chan.length == 0) return new double[0];
        return readDoubleArray(instr + makeChannelList(chan));
    }


   /**
    *  Writes a command and reads the integer result.
    */
    private int readIntegerAg(String instr, int chan) throws DriverException
    {
        return readInteger(instr + makeChannelList(chan));
    }


   /**
    *  Writes a command and reads the integer result.
    */
    private int[] readIntegerAg(String instr, int[] chan) throws DriverException
    {
        if (chan.length == 0) return new int[0];
        return readIntegerArray(instr + makeChannelList(chan));
    }
        
}
