package org.lsst.ccs.drivers.agilent;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 *  Basic routines for controlling an Agilent 33250A waveform generator
 *
 *  @author  Owen Saxton
 */
public class WG33250 extends Scpi {

    public static final int
        DEFAULT_BAUDRATE   = 57600;

    public enum Function { SINUSOID, SQUARE, RAMP, PULSE, NOISE, DC, USER; }

    private static final Map<String, Function> funcMap = new HashMap<>();
    static {
        funcMap.put("SIN",  Function.SINUSOID);
        funcMap.put("SQU",  Function.SQUARE);
        funcMap.put("RAMP", Function.RAMP);
        funcMap.put("PULS", Function.PULSE);
        funcMap.put("NOIS", Function.NOISE);
        funcMap.put("DC",   Function.DC);
        funcMap.put("USER", Function.USER);
    }

    public enum Value { DEFAULT, MINIMUM, MAXIMUM; }

    public enum Limit { MINIMUM, MAXIMUM; }

    public enum Autorange { ON, OFF, ONCE; }

    public enum Unit { VPP, VRMS, DBM; }

    private static final Map<String, Unit> unitMap = new HashMap<>();
    static {
        unitMap.put("VPP", Unit.VPP);
        unitMap.put("VRMS", Unit.VRMS);
        unitMap.put("DBM", Unit.DBM);
    }

    public enum Polarity { NORMAL, INVERTED; }

    private static final Map<String, Polarity> polarityMap = new HashMap<>();
    static {
        polarityMap.put("NORM", Polarity.NORMAL);
        polarityMap.put("INV", Polarity.INVERTED);
    }

    public enum Load { MINIMUM, MAXIMUM, INFINITY; }


   /**
    *  Constructor.
    */
    public WG33250() {
        setOptions(Option.NO_NET);   // Disallow network connections
        setDefaultParm(DEFAULT_BAUDRATE);
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The enumerated connection type: FTDI or SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @param  baudRate  The baud rate, or 0 for default (57600)
    *
    *  @param  dataChar  The data characteristics
    *
    *  @throws  DriverException
    */
    @Override
    public void open(ConnType connType, String ident, int baudRate, int dataChar)
        throws DriverException
    {
        super.open(connType, ident, baudRate, dataChar);
        try {
            checkIdentification("Agilent Technologies", CHECK_EQUALS, "33250A", CHECK_EQUALS);
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The enumerated connection type: FTDI or SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @param  baudRate  The baud rate, or 0 for default (57600)
    *
    *  @throws  DriverException
    */
    @Override
    public void open(ConnType connType, String ident, int baudRate)
        throws DriverException
    {
        int dataChar = makeDataCharacteristics(DataBits.EIGHT, StopBits.ONE,
                                               Parity.NONE, FlowCtrl.DTR);
        open(connType, ident, baudRate, dataChar);
    }


   /**
    *  Starts the generator with the given parameters.
    *
    *  @param  func    The function to perform
    *
    *  @param  freq    The frequency
    *
    *  @param  ampl    The amplitude
    *
    *  @param  offset  The offset
    *
    *  @throws  DriverException
    */
    public void start(Function func, double freq, double ampl, double offset)
        throws DriverException
    {
        start(func, String.valueOf(freq), String.valueOf(ampl),
              String.valueOf(offset));
    }


   /**
    *  Starts the generator with the given parameters.
    *
    *  @param  func    The function to perform
    *
    *  @param  freq    The frequency (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @param  ampl    The amplitude
    *
    *  @param  offset  The offset
    *
    *  @throws  DriverException
    */
    public void start(Function func, Value freq, double ampl, double offset)
        throws DriverException
    {
        start(func, freq.name(), String.valueOf(ampl), String.valueOf(offset));
    }


   /**
    *  Starts the generator with the given parameters.
    *
    *  @param  func    The function to perform
    *
    *  @param  freq    The frequency
    *
    *  @param  ampl    The amplitude (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @param  offset  The offset
    *
    *  @throws  DriverException
    */
    public void start(Function func, double freq, Value ampl, double offset)
        throws DriverException
    {
        start(func, String.valueOf(freq), ampl.name(), String.valueOf(offset));
    }


   /**
    *  Starts the generator with the given parameters.
    *
    *  @param  func    The function to perform
    *
    *  @param  freq    The frequency
    *
    *  @param  ampl    The amplitude
    *
    *  @param  offset  The offset (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @throws  DriverException
    */
    public void start(Function func, double freq, double ampl, Value offset)
        throws DriverException
    {
        start(func, String.valueOf(freq), String.valueOf(ampl), offset.name());
    }


   /**
    *  Starts the generator with the given parameters.
    *
    *  @param  func    The function to perform
    *
    *  @param  freq    The frequency (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @param  ampl    The amplitude (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @param  offset  The offset
    *
    *  @throws  DriverException
    */
    public void start(Function func, Value freq, Value ampl, double offset)
        throws DriverException
    {
        start(func, freq.name(), ampl.name(), String.valueOf(offset));
    }


   /**
    *  Starts the generator with the given parameters.
    *
    *  @param  func    The function to perform
    *
    *  @param  freq    The frequency (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @param  ampl    The amplitude
    *
    *  @param  offset  The offset (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @throws  DriverException
    */
    public void start(Function func, Value freq, double ampl, Value offset)
        throws DriverException
    {
        start(func, freq.name(), String.valueOf(ampl), offset.name());
    }


   /**
    *  Starts the generator with the given parameters.
    *
    *  @param  func    The function to perform
    *
    *  @param  freq    The frequency
    *
    *  @param  ampl    The amplitude (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @param  offset  The offset (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @throws  DriverException
    */
    public void start(Function func, double freq, Value ampl, Value offset)
        throws DriverException
    {
        start(func, String.valueOf(freq), ampl.name(), offset.name());
    }


   /**
    *  Starts the generator with the given parameters.
    *
    *  @param  func    The function to perform
    *
    *  @param  freq    The frequency (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @param  ampl    The amplitude (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @param  offset  The offset (DEFAULT, MINIMUM or MAXIMUM)
    *
    *  @throws  DriverException
    */
    public void start(Function func, Value freq, Value ampl, Value offset)
        throws DriverException
    {
        start(func, freq.name(), ampl.name(), offset.name());
    }


   /**
    *  Starts the generator with the given parameters.
    *
    *  @param  func    The function to perform
    *
    *  @param  freq    The frequency
    *
    *  @param  ampl    The amplitude
    *
    *  @param  offset  The offset
    *
    *  @throws  DriverException
    */
    private void start(Function func, String freq, String ampl, String offset)
        throws DriverException
    {
        String freqS = (func == Function.DC || func == Function.NOISE)
                         ? "DEFAULT" : freq;
        String amplS = func == Function.DC ? "DEFAULT" : ampl;
        writeCommand("APPL:" + func.name() + " " + freqS + "," + amplS
                       + "," + offset);
    }


   /**
    *  Sets the generator function.
    *
    *  @param  func  The function to perform
    *
    *  @throws  DriverException
    */
    public void setFunction(Function func) throws DriverException
    {
        writeCommand("FUNC " + func.name());
    }


   /**
    *  Gets the generator function.
    *
    *  @return  The encoded function
    *
    *  @throws  DriverException
    */
    public Function getFunction() throws DriverException
    {
        String funcS = readString("FUNC?");
        Function func = funcMap.get(funcS);
        if (func == null) {
            throw new DriverException("Unrecognized function returned ("
                                        + funcS + ")");
        }
        return func;
    }


   /**
    *  Sets the signal frequency.
    *
    *  @param  freq  The frequency to set
    *
    *  @throws  DriverException
    */
    public void setFrequency(double freq) throws DriverException
    {
        writeCommand("FREQ " + freq);
    }


   /**
    *  Sets the signal frequency to a limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) value to set
    *
    *  @throws  DriverException
    */
    public void setFrequency(Limit limit) throws DriverException
    {
        writeCommand("FREQ " + limit.name());
    }


   /**
    *  Gets the signal frequency.
    *
    *  @return  The frequency
    *
    *  @throws  DriverException
    */
    public double getFrequency() throws DriverException
    {
        return readDouble("FREQ?");
    }


   /**
    *  Gets a signal frequency limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) to get
    *
    *  @return  The frequency
    *
    *  @throws  DriverException
    */
    public double getFrequency(Limit limit) throws DriverException
    {
        return readDouble("FREQ? " + limit.name());
    }


   /**
    *  Sets the signal amplitude.
    *
    *  @param  ampl  The amplitude to set
    *
    *  @throws  DriverException
    */
    public void setAmplitude(double ampl) throws DriverException
    {
        writeCommand("VOLT " + ampl);
    }


   /**
    *  Sets the signal amplitude to a limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) value to set
    *
    *  @throws  DriverException
    */
    public void setAmplitude(Limit limit) throws DriverException
    {
        writeCommand("VOLT " + limit.name());
    }


   /**
    *  Gets the signal amplitude.
    *
    *  @return  The amplitude
    *
    *  @throws  DriverException
    */
    public double getAmplitude() throws DriverException
    {
        return readDouble("VOLT?");
    }


   /**
    *  Gets a signal amplitude limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) to get
    *
    *  @return  The amplitude
    *
    *  @throws  DriverException
    */
    public double getAmplitude(Limit limit) throws DriverException
    {
        return readDouble("VOLT? " + limit.name());
    }


   /**
    *  Sets the signal offset.
    *
    *  @param  offset  The offset to set
    *
    *  @throws  DriverException
    */
    public void setOffset(double offset) throws DriverException
    {
        writeCommand("VOLT:OFFS " + offset);
    }


   /**
    *  Sets the signal offset to a limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) value to set
    *
    *  @throws  DriverException
    */
    public void setOffset(Limit limit) throws DriverException
    {
        writeCommand("VOLT:OFFS " + limit.name());
    }


   /**
    *  Gets the signal offset.
    *
    *  @return  The offset
    *
    *  @throws  DriverException
    */
    public double getOffset() throws DriverException
    {
        return readDouble("VOLT:OFFS?");
    }


   /**
    *  Gets a signal offset limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) to get
    *
    *  @return  The offset
    *
    *  @throws  DriverException
    */
    public double getOffset(Limit limit) throws DriverException
    {
        return readDouble("VOLT:OFFS? " + limit.name());
    }


   /**
    *  Sets the high signal level.
    *
    *  @param  level  The high level to set
    *
    *  @throws  DriverException
    */
    public void setHighLevel(double level) throws DriverException
    {
        writeCommand("VOLT:HIGH " + level);
    }


   /**
    *  Sets the high signal level to a limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) value to set
    *
    *  @throws  DriverException
    */
    public void setHighLevel(Limit limit) throws DriverException
    {
        writeCommand("VOLT:HIGH " + limit.name());
    }


   /**
    *  Gets the high signal level.
    *
    *  @return  The level
    *
    *  @throws  DriverException
    */
    public double getHighLevel() throws DriverException
    {
        return readDouble("VOLT:HIGH?");
    }


   /**
    *  Gets a high signal level limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) to get
    *
    *  @return  The amplitude
    *
    *  @throws  DriverException
    */
    public double getHighLevel(Limit limit) throws DriverException
    {
        return readDouble("VOLT:HIGH? " + limit.name());
    }


   /**
    *  Sets the low signal level.
    *
    *  @param  level  The low level to set
    *
    *  @throws  DriverException
    */
    public void setLowLevel(double level) throws DriverException
    {
        writeCommand("VOLT:LOW " + level);
    }


   /**
    *  Sets the low signal level to a limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) value to set
    *
    *  @throws  DriverException
    */
    public void setLowLevel(Limit limit) throws DriverException
    {
        writeCommand("VOLT:LOW " + limit.name());
    }


   /**
    *  Gets the low signal level.
    *
    *  @return  The level
    *
    *  @throws  DriverException
    */
    public double getLowLevel() throws DriverException
    {
        return readDouble("VOLT:LOW?");
    }


   /**
    *  Gets a low signal level limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) to get
    *
    *  @return  The amplitude
    *
    *  @throws  DriverException
    */
    public double getLowLevel(Limit limit) throws DriverException
    {
        return readDouble("VOLT:LOW? " + limit.name());
    }


   /**
    *  Sets the autorange value.
    *
    *  @param  value  The value to set (ON, off or ONCE)
    *
    *  @throws  DriverException
    */
    public void setAutorange(Autorange value) throws DriverException
    {
        writeCommand("VOLT:RANGE:AUTO " + value.name());
    }


   /**
    *  Gets the autorange on state.
    *
    *  @return  The on state (true or false)
    *
    *  @throws  DriverException
    */
    public boolean isAutorangeOn() throws DriverException
    {
        return readInteger("VOLT:RANGE:AUTO?") != 0;
    }


   /**
    *  Sets the amplitude unit.
    *
    *  @param  unit  The unit to set
    *
    *  @throws  DriverException
    */
    public void setUnit(Unit unit) throws DriverException
    {
        writeCommand("VOLT:UNIT " + unit.name());
    }


   /**
    *  Gets the amplitude unit.
    *
    *  @return  The unit
    *
    *  @throws  DriverException
    */
    public Unit getUnit() throws DriverException
    {
        String unitS = readString("VOLT:UNIT?");
        Unit unit = unitMap.get(unitS);
        if (unit == null) {
            throw new DriverException("Unrecognized unit returned ("
                                        + unitS + ")");
        }
        return unit;
    }


   /**
    *  Sets the square wave duty cycle.
    *
    *  @param  cycle  The duty cycle to set (percent)
    *
    *  @throws  DriverException
    */
    public void setDutyCycle(double cycle) throws DriverException
    {
        writeCommand("FUNC:SQU:DCYC " + cycle);
    }


   /**
    *  Sets the square wave duty cycle to a limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) value to set
    *
    *  @throws  DriverException
    */
    public void setDutyCycle(Limit limit) throws DriverException
    {
        writeCommand("FUNC:SQU:DCYC " + limit.name());
    }


   /**
    *  Gets the square wave duty cycle.
    *
    *  @return  The duty cycle (percent)
    *
    *  @throws  DriverException
    */
    public double getDutyCycle() throws DriverException
    {
        return readDouble("FUNC:SQU:DCYC?");
    }


   /**
    *  Gets a square wave duty cycle limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) to get
    *
    *  @return  The limit
    *
    *  @throws  DriverException
    */
    public double getDutyCycle(Limit limit) throws DriverException
    {
        return readDouble("FUNC:SQU:DCYC? " + limit.name());
    }


   /**
    *  Sets the ramp symmetry.
    *
    *  @param  symm  The symmetry to set (percent)
    *
    *  @throws  DriverException
    */
    public void setSymmetry(double symm) throws DriverException
    {
        writeCommand("FUNC:RAMP:SYMM " + symm);
    }


   /**
    *  Sets the ramp symmetry to a limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) value to set
    *
    *  @throws  DriverException
    */
    public void setSymmetry(Limit limit) throws DriverException
    {
        writeCommand("FUNC:RAMP:SYMM " + limit.name());
    }


   /**
    *  Gets the ramp symmetry.
    *
    *  @return  The symmetry (percent)
    *
    *  @throws  DriverException
    */
    public double getSymmetry() throws DriverException
    {
        return readDouble("FUNC:RAMP:SYMM?");
    }


   /**
    *  Gets a ramp symmetry limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) to get
    *
    *  @return  The limit
    *
    *  @throws  DriverException
    */
    public double getSymmetry(Limit limit) throws DriverException
    {
        return readDouble("FUNC:RAMP:SYMM? " + limit.name());
    }


   /**
    *  Turns front-panel output on or off.
    *
    *  @param  on  Turn on if true, off if false
    *
    *  @throws  DriverException
    */
    public void setOutput(boolean on) throws DriverException
    {
        writeCommand("OUTP " + (on ? "ON" : "OFF"));
    }


   /**
    *  Gets the front-panel output on state.
    *
    *  @return  Whether the output state is on
    *
    *  @throws  DriverException
    */
    public boolean isOutputOn() throws DriverException
    {
        return readInteger("OUTP?") != 0;
    }


   /**
    *  Sets the output load.
    *
    *  @param  load  The output load to set (ohms)
    *
    *  @throws  DriverException
    */
    public void setOutputLoad(double load) throws DriverException
    {
        writeCommand("OUTP:LOAD " + load);
    }


   /**
    *  Sets the output load to a limit or infinity.
    *
    *  @param  load  The value (MINIMUM, MAXIMUM or INFINITY) to set
    *
    *  @throws  DriverException
    */
    public void setOutputLoad(Load load) throws DriverException
    {
        writeCommand("OUTP:LOAD " + load.name());
    }


   /**
    *  Gets the output load.
    *
    *  @return  The output load (ohms)
    *
    *  @throws  DriverException
    */
    public double getOutputLoad() throws DriverException
    {
        return readDouble("OUTP:LOAD?");
    }


   /**
    *  Gets an output load limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) to get
    *
    *  @return  The limit (ohms)
    *
    *  @throws  DriverException
    */
    public double getOutputLoad(Limit limit) throws DriverException
    {
        return readDouble("OUTP:LOAD? " + limit.name());
    }


   /**
    *  Sets the signal polarity.
    *
    *  @param  pol  The polarity to set (NORMAL or INVERTED)
    *
    *  @throws  DriverException
    */
    public void setPolarity(Polarity pol) throws DriverException
    {
        writeCommand("OUTP:POL " + pol.name());
    }


   /**
    *  Gets the signal polarity.
    *
    *  @return  The polarity
    *
    *  @throws  DriverException
    */
    public Polarity getPolarity() throws DriverException
    {
        String polS = readString("OUTP:POL?");
        Polarity pol = polarityMap.get(polS);
        if (pol == null) {
            throw new DriverException("Unrecognized polarity returned ("
                                        + polS + ")");
        }
        return pol;
    }


   /**
    *  Turns front-panel sync output on or off.
    *
    *  @param  on  Turn on if true, off if false
    *
    *  @throws  DriverException
    */
    public void setSyncOutput(boolean on) throws DriverException
    {
        writeCommand("OUTP:SYNC " + (on ? "ON" : "OFF"));
    }


   /**
    *  Gets the front-panel sync output on state.
    *
    *  @return  Whether the sync output state is on
    *
    *  @throws  DriverException
    */
    public boolean isSyncOutputOn() throws DriverException
    {
        return readInteger("OUTP:SYNC?") != 0;
    }


   /**
    *  Sets the pulse period.
    *
    *  @param  period  The pulse period to set (secs)
    *
    *  @throws  DriverException
    */
    public void setPulsePeriod(double period) throws DriverException
    {
        writeCommand("PULS:PER " + period);
    }


   /**
    *  Sets the pulse period to a limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) value to set
    *
    *  @throws  DriverException
    */
    public void setPulsePeriod(Limit limit) throws DriverException
    {
        writeCommand("PULS:PER " + limit.name());
    }


   /**
    *  Gets the pulse period.
    *
    *  @return  The pulse period (secs)
    *
    *  @throws  DriverException
    */
    public double getPulsePeriod() throws DriverException
    {
        return readDouble("PULS:PER?");
    }


   /**
    *  Gets a pulse period limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) to get
    *
    *  @return  The pulse period (secs)
    *
    *  @throws  DriverException
    */
    public double getPulsePeriod(Limit limit) throws DriverException
    {
        return readDouble("PULS:PER? " + limit.name());
    }


   /**
    *  Sets the pulse width.
    *
    *  @param  width  The pulse width to set (secs)
    *
    *  @throws  DriverException
    */
    public void setPulseWidth(double width) throws DriverException
    {
        writeCommand("PULS:WIDT " + width);
    }


   /**
    *  Sets the pulse width to a limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) value to set
    *
    *  @throws  DriverException
    */
    public void setPulseWidth(Limit limit) throws DriverException
    {
        writeCommand("PULS:WIDT " + limit.name());
    }


   /**
    *  Gets the pulse width.
    *
    *  @return  The pulse width (secs)
    *
    *  @throws  DriverException
    */
    public double getPulseWidth() throws DriverException
    {
        return readDouble("PULS:WIDT?");
    }


   /**
    *  Gets a pulse width limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) to get
    *
    *  @return  The pulse width (secs)
    *
    *  @throws  DriverException
    */
    public double getPulseWidth(Limit limit) throws DriverException
    {
        return readDouble("PULS:WIDT? " + limit.name());
    }


   /**
    *  Sets the pulse transition time.
    *
    *  @param  tran  The pulse transition time to set (secs)
    *
    *  @throws  DriverException
    */
    public void setPulseTransition(double tran) throws DriverException
    {
        writeCommand("PULS:TRAN " + tran);
    }


   /**
    *  Sets the pulse transition time to a limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) value to set
    *
    *  @throws  DriverException
    */
    public void setPulseTransition(Limit limit) throws DriverException
    {
        writeCommand("PULS:TRAN " + limit.name());
    }


   /**
    *  Gets the pulse transition time.
    *
    *  @return  The pulse transition time (secs)
    *
    *  @throws  DriverException
    */
    public double getPulseTransition() throws DriverException
    {
        return readDouble("PULS:TRAN?");
    }


   /**
    *  Gets a pulse transition time limit.
    *
    *  @param  limit  The limit (MINIMUM or MAXIMUM) to get
    *
    *  @return  The pulse transition time (secs)
    *
    *  @throws  DriverException
    */
    public double getPulseTransition(Limit limit) throws DriverException
    {
        return readDouble("PULS:TRAN? " + limit.name());
    }

}
