package org.lsst.ccs.drivers.agilent;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 *  Basic routines for controlling an Agilent 34970A data logger.
 *
 *  @author  Owen Saxton
 */
public class DL34970 extends Scpi {

    public enum TcType { B, E, J, K, N, R, S, T; }

    public static final int
        DEFAULT_BAUDRATE = 9600;

    public static final double
        RAW_ERROR_VALUE = 9.9e37,
        ERROR_VALUE     = Double.NaN;

   /**
    *  Constructor.
    */
    public DL34970() {
        setOptions(Option.NO_NET);   // Disallow network connections
        setDefaultParm(DEFAULT_BAUDRATE);
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The enumerated connection type: FTDI or SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @param  baudRate  The baud rate, or 0 for default (9600)
    *
    *  @param  dataChar  The data characteristics
    *
    *  @throws  DriverException
    */
    @Override
    public void open(ConnType connType, String ident, int baudRate, int dataChar)
        throws DriverException
    {
        super.open(connType, ident, baudRate, dataChar);
        try {
            checkIdentification("HEWLETT-PACKARD", CHECK_EQUALS, "34970A", CHECK_EQUALS);
            writeCommand("ABORT");
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The enumerated connection type: FTDI or SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @param  baudRate  The baud rate, or 0 for default (9600)
    *
    *  @throws  DriverException
    */
    @Override
    public void open(ConnType connType, String ident, int baudRate)
        throws DriverException
    {
        int dataChar = makeDataCharacteristics(DataBits.EIGHT, StopBits.ONE,
                                               Parity.NONE, FlowCtrl.DTR);
        open(connType, ident, baudRate, dataChar);
    }


   /**
    *  Configures channels for thermocouple reading.
    *
    *  @param  type   The thermocouple type (B, E, J, K, N, R, S. T)
    *
    *  @param  chans  The array of channel numbers
    *
    *  @throws  DriverException
    */
    public void configTC(TcType type, int[] chans) throws DriverException
    {
        writeCommand("CONF:TEMP TC," + type + ',' + makeChannelList(chans));
        setup();
    }


   /**
    *  Configures channels for DC voltage reading.
    *
    *  @param  range  The desired voltage range
    *
    *  @param  chans  The array of channel numbers
    *
    *  @throws  DriverException
    */
    public void configVolts(double range, int[] chans) throws DriverException
    {
        writeCommand("CONF:VOLT:DC " + range + ',' + makeChannelList(chans));
        setup();
    }


   /**
    *  Configures channels for DC current reading.
    *
    *  @param  range  The desired current range
    *
    *  @param  chans  The array of channel numbers
    *
    *  @throws  DriverException
    */
    public void configCurrent(double range, int[] chans) throws DriverException
    {
        writeCommand("CONF:CURR:DC " + range + ',' + makeChannelList(chans));
        setup();
    }


   /**
    *  Sets up the scan list.
    *
    *  @param  chans  The array of channel numbers
    *
    *  @throws  DriverException
    */
    public void setScanList(int[] chans) throws DriverException
    {
        writeCommand("ROUT:SCAN" + makeChannelList(chans));
    }


   /**
    *  Reads data from channels in scan list.
    *
    *  @return  Double array of read values in ascending channel number order
    *
    *  @throws  DriverException
    */
    public double[] readData() throws DriverException
    {
        writeCommand("FORM:READ:CHAN OFF");
        double[] values = readDoubleArray("READ?");
        for (int j = 0; j < values.length; j++) {
            if (Math.abs(values[j]) == RAW_ERROR_VALUE) {
                values[j] = ERROR_VALUE;
            }
        }
        return values;
    }


   /**
    *  Reads data from one channel.
    *
    *  This also sets the scan list
    *
    *  @param  chan  The channel number
    *
    *  @return  The read value
    *
    *  @throws  DriverException
    */
    public double readData(int chan) throws DriverException
    {
        writeCommand("FORM:READ:CHAN OFF;:ROUT:SCAN" + makeChannelList(new int[]{chan}));
        double value = readDouble(";:READ?");
        return (Math.abs(value) == RAW_ERROR_VALUE) ? ERROR_VALUE : value;
    }


   /**
    *  Reads data from specified channels.
    *
    *  This also sets the scan list
    *
    *  @param  chans  The array of channel numbers
    *
    *  @return  Double array of read values
    *
    *  @throws  DriverException
    */
    public double[] readData(int[] chans) throws DriverException
    {
        writeCommand("FORM:READ:CHAN ON;:ROUT:SCAN" + makeChannelList(chans));
        String reply = readString("READ?");
        String words[] = reply.split(",");
        if (words.length != 2 * chans.length) {
            throw new DriverException("Invalid response: " + reply);
        }
        double[] values = new double[chans.length];
        for (int j = 0; j < values.length; j++) {
            int chan;
            String word = words[2 * j + 1];
            try {
                chan = Integer.parseInt(word);
            }
            catch (NumberFormatException e) {
                throw new DriverException("Invalid integer in response: " + word);
            }
            Double value = 0.0;
            word = words[2 * j];
            try {
                value = Double.parseDouble(word);
            }
            catch (NumberFormatException e) {
                throw new DriverException("Invalid double in response: " + word);
            }
            for (int k = 0; k < chans.length; k++) {
                if (chans[k] == chan) {
                    values[k] = (Math.abs(value) == RAW_ERROR_VALUE) ? ERROR_VALUE : value;
                    value = null;
                    break;
                }
            }
            if (value != null) {
                throw new DriverException("Invalid channel in response: " + chan);
            }
        }
        return values;
    }


   /**
    *  Performs essential setup.
    *
    *  @throws  DriverException
    */
    private void setup() throws DriverException
    {
    }

}
