package org.lsst.ccs.drivers.agilent;

import java.util.concurrent.TimeoutException;
import java.util.Arrays;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 **************************************************************************
 **
 ** General access routines for the Agilent E3631 Power Supply
 ** @author Peter Doherty 
 ** The E3631 manual released April 21, 2014 states:
 ** baud can be set manually, but the factory default is 9600
 ** factory defaults to 8 bits no parity which is also configurable at the front panel
 ** 1 start bit and 2 stop bits 
 **
 ** There are 3 output settings: P6V (+6V), P25V (+25V), N25V (-25V)
 **
 ** The manual does recommend making sure the power supply is set to REMOTE mode
 ** via the SYST:REM command
 **
 * **************************************************************************
 */

public class E3631 extends Ascii {

    /**
     **************************************************************************
     **
     ** Public constants *
     * **************************************************************************
     */

    public enum onOff {  /* not used presently */

        OFF, ON;
    }

    public enum OutputName {
        P6V("P6V"),
        P25V("P25V"),
        N25V("N25V");

        String value;
        OutputName(String value) {
            this.value = value;
        }
        public String getValue() {
            return value;
        }
    }

    /**
     **************************************************************************
     **
     ** Private constants *
     * **************************************************************************
     */
    private final static byte CR = 0x0d, LF = 0x0a;
    private final String terminator = "\r\n";
    private int timeout = 1000;
    private Boolean verbose = true;
    private Boolean fakeout = false;
    
    /**
     **************************************************************************
     **
     ** Constructor * *
     * **************************************************************************
     */
    public E3631() throws DriverException {
    }

    /**
     **************************************************************************
     **
     ** Opens a connection. * * @param device The device name of the serial
     * port
     * @param serialname
     * @param port
     */
    public void open(String serialname, int port) throws DriverException {
        /* Set 8 data bits, 2 stop bits, No Parity, No Flow Control*/

        int asciiParam = makeDataCharacteristics(Ascii.DataBits.EIGHT, 
            Ascii.StopBits.TWO, Ascii.Parity.NONE, Ascii.FlowCtrl.NONE);
        open(Ascii.CONN_TYPE_SERIAL, serialname, port, asciiParam);
        init();   
        /* Make sure we're in remote operation */
        writeE3631("SYST:REM");
    }

    /**
    ***************************************************************************
    **
    **  Opens an FTDI connection.
    **
    ***************************************************************************
     * @param serialname
     * @param baud 
    */
    public void openftdi(String serialname, int baud) throws DriverException {
        System.out.println("opening connection to the Agilent E3631 Power Supply");
        /* Set 8 data bits, 2 stop bits, No Parity, No Flow Control*/
        int asciiParam = makeDataCharacteristics(Ascii.DataBits.EIGHT, 
            Ascii.StopBits.TWO, Ascii.Parity.NONE, Ascii.FlowCtrl.NONE);
        open(Ascii.CONN_TYPE_FTDI, serialname, baud, asciiParam);
        /* Make sure we're in remote operation */
        writeE3631("SYST:REM");
    }
    
    /**
    ***************************************************************************
    **
    **  Initialize the instrument
    ***************************************************************************
    */
    public void init() throws DriverException {
    }
    
    /**
     **************************************************************************
     **
     ** Closes the connection. * * @throws DriverException *
     * *************************************************************************
     */
    public void close() throws DriverException {
    }

   /**
  ***********************************************************************
  **  Set to true if in simulation mode
  */
    public void setFakeout(Boolean f) throws DriverException {
        fakeout = f;
    }

   /**
    * Set verbosity to either true or false
    */
    public void setVerbose(Boolean v) throws DriverException {
        verbose = v;
    }

   /**
     **************************************************************************
     ** Select one of the outputs for control
     * *************************************************************************
     * @param outname
     */
    public void selectOutput(String outname) throws DriverException {
        /* make sure outname is legit */
        if ((Arrays.asList("P6V","P25V","N25V").contains(outname)) == false) {
            String errstr = "E3631 selectOutput: Invalid output specified:" + outname;
            System.out.println(errstr);
            throw new DriverException (errstr);
        }
        /* output name must be OK, so select it */
        try {
            sendCommand("INST:SEL " + outname);
        } catch (DriverException e) {
            String errstr = "E3631 selectOutput: " + e;
            System.out.println(errstr);     
            throw new DriverException(errstr);
        }
    }
    
    /**
     **************************************************************************
     ** Ask which output is currently selected 
     * *************************************************************************
     * @param outname
     * @return 
     */
    public String getOutput() throws DriverException {
        /* get the currently selected channel's name */
        try {
            String outstr = getReply("INST:SEL?");
            return outstr;
        } catch (DriverException e) {
            String errstr = "E3631 gettOutput:" + e;
            System.out.println(errstr); 
            throw new DriverException (errstr);
        } 
    }
    
    /**
     **************************************************************************
     ** Turn an output on 
     * *************************************************************************
     * @param outname
     */
    public void outputOn(String outname) throws DriverException {
        try {
            selectOutput(outname);
            sendCommand("OUTPUT:STATE ON ");
        } catch (DriverException e) {
            String errstr = "E3631 outputOn: " + e;
            System.out.println(errstr); 
            throw new DriverException(errstr);
         }
    }
    
    /**************************************************************************
     * outputOff: Turn an output off
     * @param outname
     */
    public void outputOff(String outname) throws DriverException {
         try {
            selectOutput(outname);
            sendCommand("OUTPUT:STATE OFF ");
        } catch (DriverException e) {
            String errstr = "E3631 outputOn: " + e;
            System.out.println(errstr); 
            throw new DriverException(errstr);
         }
    }

   
    /**************************************************************************
     * getVoltage: Measure voltage for current output 
     * 
     * @return 
     */
    public double getVoltage() throws DriverException {
        try {
            String voltstr = readE3631("MEAS:VOLT:DC?");
            return Double.valueOf(voltstr);
        } catch (DriverException e) {
            String errstr = "E3631 getVoltage: " + e;
            System.out.println(errstr); 
            throw new DriverException(errstr);
         }
    }
    
    
    /**************************************************************************
     * getVoltage: Measure voltage for a channel
     * 
     * @param outname  the name of the output to measure: P5V, P25V, or N25V
     * @return 
     */
    public double getVoltage(String outname) throws DriverException {
        try {
            selectOutput(outname);
            // Sleep before trying to read to avoid reading timeout
            Thread.sleep(800);
            String voltstr = getReply("MEAS:VOLT:DC? " + outname);
            return Double.valueOf(voltstr);
        } catch (DriverException e) {
            String errstr = "E3631 getVoltage: " + e;
            System.out.println(errstr); 
            throw new DriverException(errstr);
         } catch(InterruptedException e) {
            throw new DriverException();
         }
    }
    
    /**
     **************************************************************************
     ** Measure current for currently selected output
     * *************************************************************************
     * @return 
     */
    public double getCurrent() throws DriverException {
        try {
            String ampstr = readE3631("MEAS:CURR:DC?");
            return Double.valueOf(ampstr);
        } catch (DriverException e) {
            String errstr = "E3631 getCurrent: " + e;
            System.out.println(errstr); 
            throw new DriverException(errstr);
         }
    }

    /**
     **************************************************************************
     ** Measure current for a channel
     * *************************************************************************
     * @param outname
     * @return 
     */
    public double getCurrent(String outname) throws DriverException {
        try {
            selectOutput(outname);
            // Sleep before trying to read to avoid reading timeout
            Thread.sleep(800);
            String ampstr = getReply("MEAS:CURR:DC? " + outname);
            return Double.valueOf(ampstr);
        } catch (DriverException e) {
            String errstr = "E3631 getCurrent: " + e;
            System.out.println(errstr); 
            throw new DriverException(errstr);
         } catch(InterruptedException e) {
            throw new DriverException();
         }
    }
    
    /**
     **************************************************************************
     ** Set voltage for a channel
     * *************************************************************************
     * @param outname
     * @param volts
     */
    public void setVoltage(String outname, double volts) throws DriverException {
        if (this.fakeout == true) {
            System.out.println("E3631 setVoltage: outname = " + outname);
            System.out.println("E3631 setVoltage: volts   = " + volts);
            return;
        }
        /* check if requested voltage (in volts) is OK */
        Boolean voltageError = false;
        switch (outname) {
            case "P6V":
                if ((volts < 0.0) || (volts > 6.0)) 
                    voltageError = true;
                break;
            case "P25V":
                if ((volts < 0.0) || (volts > 25.0))
                    voltageError = true;
                break;
            case "N25V":
                if ((volts > 0.0) || (volts < -25.0))
                    voltageError = true;
                break;
            default:
                String errstr = "E3631 setVoltage: output name error";
                System.out.println(errstr); 
                throw new DriverException(errstr);
        }
        if (voltageError != false) {
            String errstr = "E3631 setVoltage: voltage error";
            System.out.println(errstr); 
            throw new DriverException(errstr);
        }
        /* parameters must be OK, so select output and set voltage */
        try {
            selectOutput(outname);
            // Sleep before trying to read to avoid reading timeout
            Thread.sleep(800);
            sendCommand("VOLT:DC " + volts);
        } catch (DriverException e) {
            String errstr = "E3631 setVoltage: " + e;
            System.out.println(errstr); 
            throw new DriverException(errstr);
         } catch(InterruptedException e) {
            throw new DriverException();
         }
    }
    
    /**
     **************************************************************************
     ** Set current for a channel
     * *************************************************************************
     * @param outname
     * @param amps
     */
    public void setCurrent(String outname, double amps) throws DriverException {
        if (this.fakeout == true) {
            System.out.println("E3631 setCurrent: outname = "+outname+"amps ="+ amps);
            return;
        }
        /* check if requested current (in amps) is OK */
        Boolean currentError = false;
        switch (outname) {
            case "P6V":
                if ((amps < 0.0) || (amps > 6.0))
                    currentError = true;
                break;
            case "P25V":
                if ((amps < 0.0) || (amps > 1.0)) 
                    currentError = true;
                break;
            case "N25V":
                if ((amps < 0.0) || (amps > 1.0))
                    currentError = true;
                break;
            default:
                String errstr = "E3631 setCurrent: output name error";
                System.out.println(errstr); 
                throw new DriverException(errstr);
        }
        if (currentError != true) {
            String errstr = "E3631 setCurrent: current error";
            System.out.println(errstr); 
            throw new DriverException(errstr);
        }
        /* parameters must be OK, so select output and then set current */
        try {
            selectOutput(outname);
            // Sleep before trying to read to avoid reading timeout
            Thread.sleep(800);
            sendCommand("CURR:DC " + amps);
        } catch (DriverException e) {
            String errstr = "E3631 setCurrent: " + e;
            System.out.println(errstr);
            throw new DriverException(errstr);
         } catch(InterruptedException e) {
            throw new DriverException();
         }
       
    }
    
    /**
     ***************************************************************************
     **
     ** Get the E3631 ID
     * @return 
    */
    public String getID() throws DriverException {
        /* try and get the ID string from the device */
        try {
            String idString = readE3631("*IDN?");
            if (this.verbose == true) 
                System.out.println("E3631 ID: " + idString);
            return idString;
        } catch (DriverException e) {
            String errstr = "E3631 getID: " + e;
            System.out.println(errstr);
            throw new DriverException(errstr);              
        }
    }
    
    /**
     ***************************************************************************
     **
     ** Get the E36311 error status
     * @return 
    */
    public String getError() throws DriverException {
        /* try and get the Error status from the device */
        try {
            //String errString = getReply("ERR");
            String errString = readE3631("ERR");
            if (this.verbose == true) 
                System.out.println("E3631 ID: " + errString);
            return errString;
        } catch (DriverException e) {
            String errstr = "E3631 getError: " + e;
            System.out.println(errstr);
            throw new DriverException(errstr);      
        }
    }

    /**
     ***************************************************************************
     **
     ** reset the E36311 
     * @return 
    */
    public void reset() throws DriverException {
        /* reset the device */
        try {
            writeE3631("*RST");
        } catch (DriverException e) {
            String errstr = "E3631 reset: " + e;
            System.out.println(errstr);
            throw new DriverException(errstr);      
        }
    }
    
   /**
     ***************************************************************************
     **
     ** sendCommand(String command)
     * 
     * Send a command, but do not wait for a reply other than to check 
     * for the acknowledgment of that command
     * @param command
     */
    public synchronized void sendCommand(String command) throws DriverException {
        /* if being verbose, print out the command */
        if (this.verbose == true) 
          System.out.println("E3631Command: " + command);
        writeE3631(command); /* write the command to the E3631 */
    }   

   /**
     *************************************************************************
     **
     ** getReply(String command)
     * 
     * Send a command, and wait for a reply. Return reply String
     * @param command
     * @return 
     */
    public synchronized String getReply(String command) throws DriverException {
        /* if being verbose, print out the command */
        if (this.verbose == true)
            System.out.println("E3631 getReply: " + command);
  
        writeE3631(command);          
       
        /*send query and get the response, return reply string*/
        String reply = readE3631();
        return reply;
    }   
    
    /**
     **************************************************************************
     **
     ** Sets the receive timeout. * * @param timeout The receive timeout (ms).
     * 0 means no timeout. * * @throws DriverException *
     * *************************************************************************
     * @param timeout
     */
    public void setTimeout(int timeout) throws DriverException {
        this.timeout = timeout;
    }

    /**
     **************************************************************************
     **
     ** Writes a command. * * @param command The command to write, excluding
     * terminator * * @throws DriverException *
     * *************************************************************************
     * @param command
     */
    public synchronized void writeE3631(String command) throws DriverException {
        write(command);
    }

    /**
     **************************************************************************
     **
     ** Reads a response. * * @return The command response string * * @throws
     * DriverException
     * @return
     */
    public synchronized String readE3631() throws DriverException {
        return(read());
    }

    /**
     **************************************************************************
     **
     ** Reads a response after writing a command. * * @param command The
     * command to write, excluding terminator * * @return The command response
     * string * * @throws DriverException
     * @param command
     * @return
     */
    public synchronized String readE3631(String command) throws DriverException {
        writeE3631(command);
        return readE3631();
    }

}
