package org.lsst.ccs.drivers.agilent;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 *************************************************************************** 
 ** 
 **  Program to test the Agilent N6700 device driver
 ** 
 **  @author Owen Saxton
 ** 
 *************************************************************************** 
 */
public class TestN6700 extends TestScpi {

    private N6700 n67 = new N6700();
    private int chanNum = 1;

    public enum onOff {
        OFF, ON;
    }


   /**
    *************************************************************************** 
    ** 
    **  Constructor
    ** 
    *************************************************************************** 
    */
    public TestN6700()
    {
        scpi = n67;
    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    ** 
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="host", description="Host name")
                     String host
                     ) throws DriverException
    {
        n67.open(host);
    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    ** 
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="host", description="Host name")
                     String host,
                     @Argument(name="port", description="Port number")
                     int port
                     ) throws DriverException
    {
        n67.open(host, port);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the channel.
    ** 
    *************************************************************************** 
    */
    @Command(name="setchannel", description="Set the current channel")
    public void setChannel(@Argument(name="channel", description="Channel number")
                           int chan)
    {
        chanNum = chan;
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set channel.
    ** 
    *************************************************************************** 
    */
    @Command(name="showchannel", description="Show the set channel")
    public String getChannel()
    {
        return "Channel = " + chanNum;
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows channel information.
    ** 
    *************************************************************************** 
    */
    @Command(name="showchaninfo", description="Show channel information")
    public String showChanInfo() throws DriverException
    {
        String[] info = n67.getChannelInfo(chanNum);
        return   "Channel number: " + chanNum +
               "\nModel name:     " + info[N6700.CHAN_INFO_MODEL] +
               "\nOptions:        " + info[N6700.CHAN_INFO_OPTION] +
               "\nSerial number:  " + info[N6700.CHAN_INFO_SERIAL];
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows channel information.
    ** 
    *************************************************************************** 
    */
    @Command(name="showchaninfo", description="Show channel information")
    public String showChanInfo(@Argument(name="channel", description="Channel number")
                               int chan) throws DriverException
    {
        String[] info = n67.getChannelInfo(chan);
        return   "Channel number: " + chan +
               "\nModel name:     " + info[N6700.CHAN_INFO_MODEL] +
               "\nOptions:        " + info[N6700.CHAN_INFO_OPTION] +
               "\nSerial number:  " + info[N6700.CHAN_INFO_SERIAL];
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the voltage.
    ** 
    *************************************************************************** 
    */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value) throws DriverException
    {
        n67.setVoltage(value, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the voltage.
    ** 
    *************************************************************************** 
    */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value,
                           @Argument(name="channel", description="Channel number")
                           int chan) throws DriverException
    {
        n67.setVoltage(value, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set voltage.
    ** 
    *************************************************************************** 
    */
    @Command(name="showvoltage", description="Show the set voltage")
    public String getVoltage() throws DriverException
    {
        return "Voltage = " + n67.getVoltage(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set voltage(s).
    ** 
    *************************************************************************** 
    */
    @Command(name="showvoltage", description="Show the set voltage")
    public String getVoltage(@Argument(name="channel", description="Channel number")
                             String chanList) throws DriverException
    {
        int[] chans = getChannels(chanList);
        if (chans.length == 1) {
            return "Voltage = " + n67.getVoltage(chans[0]);
        }
        else {
            return "Voltages = " + getString(n67.getVoltage(chans));
        }
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual voltage.
    ** 
    *************************************************************************** 
    */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage() throws DriverException
    {
        return "Voltage = " + n67.readVoltage(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual voltage(s).
    ** 
    *************************************************************************** 
    */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage(@Argument(name="channel", description="Channel number")
                              String chanList) throws DriverException
    {
        int[] chans = getChannels(chanList);
        if (chans.length == 1) {
            return "Voltage = " + n67.readVoltage(chans[0]);
        }
        else {
            return "Voltages = " + getString(n67.readVoltage(chans));
        }
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the soft voltage limit.
    ** 
    *************************************************************************** 
    */
    @Command(name="setvoltlim", description="Set the soft voltage limit")
    public void setVoltageLimit(@Argument(name="value",
                                          description="Voltage limit to set")
                                double value) throws DriverException
    {
        n67.setVoltageLimit(value, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the soft voltage limit.
    ** 
    *************************************************************************** 
    */
    @Command(name="setvoltlim", description="Set the soft voltage limit")
    public void setVoltageLimit(@Argument(name="value",
                                          description="Voltage limit to set")
                                double value,
                                @Argument(name="channel", description="Channel number")
                                int chan) throws DriverException
    {
        n67.setVoltageLimit(value, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the soft voltage limit.
    ** 
    *************************************************************************** 
    */
    @Command(name="showvoltlim", description="Show the soft voltage limit")
    public String getVoltageLimit() throws DriverException
    {
        return "Voltage limit = " + n67.getVoltageLimit(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the soft voltage limit.
    ** 
    *************************************************************************** 
    */
    @Command(name="showvoltlim", description="Show the soft voltage limit")
    public String getVoltageLimit(@Argument(name="channel", description="Channel number")
                                  int chan) throws DriverException
    {
        return "Voltage limit = " + n67.getVoltageLimit(chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the current.
    ** 
    *************************************************************************** 
    */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set")
                           double value) throws DriverException
    {
        n67.setCurrent(value, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the current.
    ** 
    *************************************************************************** 
    */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set")
                           double value,
                           @Argument(name="channel", description="Channel number")
                           int chan) throws DriverException
    {
        n67.setCurrent(value, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set current.
    ** 
    *************************************************************************** 
    */
    @Command(name="showcurrent", description="Show the set current")
    public String getCurrent() throws DriverException
    {
        return "Current = " + n67.getCurrent(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set current(s).
    ** 
    *************************************************************************** 
    */
    @Command(name="showcurrent", description="Show the set current")
    public String getCurrent(@Argument(name="channel", description="Channel number")
                             String chanList) throws DriverException
    {
        int[] chans = getChannels(chanList);
        if (chans.length == 1) {
            return "Current = " + n67.getCurrent(chans[0]);
        }
        else {
            return "Currents = " + getString(n67.getCurrent(chans));
        }
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual current.
    ** 
    *************************************************************************** 
    */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent() throws DriverException
    {
        return "Current = " + n67.readCurrent(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual current(s).
    ** 
    *************************************************************************** 
    */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent(@Argument(name="channel", description="Channel number")
                              String chanList) throws DriverException
    {
        int[] chans = getChannels(chanList);
        if (chans.length == 1) {
            return "Current = " + n67.readCurrent(chans[0]);
        }
        else {
            return "Currents = " + getString(n67.readCurrent(chans));
        }
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the soft current limit.
    ** 
    *************************************************************************** 
    */
    @Command(name="setcurrlim", description="Set the soft current limit")
    public void setCurrentLimit(@Argument(name="value",
                                          description="Current limit to set")
                                double value) throws DriverException
    {
        n67.setCurrentLimit(value, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the soft current limit.
    ** 
    *************************************************************************** 
    */
    @Command(name="setcurrlim", description="Set the soft current limit")
    public void setCurrentLimit(@Argument(name="value",
                                          description="Current limit to set")
                                double value,
                                @Argument(name="channel", description="Channel number")
                                int chan) throws DriverException
    {
        n67.setCurrentLimit(value, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the soft current limit.
    ** 
    *************************************************************************** 
    */
    @Command(name="showcurrlim", description="Show the soft current limit")
    public String getCurrentLimit() throws DriverException
    {
        return "Current limit = " + n67.getCurrentLimit(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the soft current limit(s).
    ** 
    *************************************************************************** 
    */
    @Command(name="showcurrlim", description="Show the soft current limit")
    public String getCurrentLimit(@Argument(name="channel", description="Channel number")
                                  int chan) throws DriverException
    {
        return "Current limit = " + n67.getCurrentLimit(chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Turns the output on or off.
    ** 
    *************************************************************************** 
    */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state",
                                    description="Output state: on or off")
                          onOff state) throws DriverException
    {
        n67.setOutput(state == onOff.ON, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Turns the output(s) on or off.
    ** 
    *************************************************************************** 
    */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state",
                                    description="Output state: on or off")
                          onOff state,
                          @Argument(name="channel", description="Channel number")
                          String chanList) throws DriverException
    {
        int[] chans = getChannels(chanList);
        if (chans.length == 1) {
            n67.setOutput(state == onOff.ON, chans[0]);
        }
        else {
            n67.setOutput(state == onOff.ON, chans);
        }
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the output state.
    ** 
    *************************************************************************** 
    */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput() throws DriverException
    {
        return "Output state = " + getOnOff(n67.getOutput(chanNum));
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the output state(s).
    ** 
    *************************************************************************** 
    */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput(@Argument(name="channel", description="Channel number")
                             String chanList) throws DriverException
    {
        int[] chans = getChannels(chanList);
        if (chans.length == 1) {
            return "Output state = " + getOnOff(n67.getOutput(chans[0]));
        }
        else {
            boolean[] states = n67.getOutput(chans);
            StringBuilder text = new StringBuilder(getOnOff(states[0]));
            for (int j = 1; j < states.length; j++) {
                text.append(", ").append(getOnOff(states[j]));
            }
            return "Output states = " + text;
        }
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the output-on delay.
    ** 
    *************************************************************************** 
    */
    @Command(name="setondelay", description="Set the output-on delay")
    public void setOnDelay(@Argument(name="time",
                                     description="Output-on delay (sec)")
                           double time) throws DriverException
    {
        n67.setOnDelay(time, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the output-on delay.
    ** 
    *************************************************************************** 
    */
    @Command(name="setondelay", description="Set the output-on delay")
    public void setOnDelay(@Argument(name="time",
                                     description="Output-on delay (sec)")
                           double time,
                           @Argument(name="channel", description="Channel number")
                           int chan) throws DriverException
    {
        n67.setOnDelay(time, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the output-on delay.
    ** 
    *************************************************************************** 
    */
    @Command(name="showondelay", description="Show the output-on delay")
    public String getOnDelay() throws DriverException
    {
        return "Delay = " + n67.getOnDelay(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the output-on delay(s).
    ** 
    *************************************************************************** 
    */
    @Command(name="showondelay", description="Show the output-on delay")
    public String getOnDelay(@Argument(name="channel", description="Channel number")
                             String chanList) throws DriverException
    {
        int[] chans = getChannels(chanList);
        if (chans.length == 1) {
            return "Delay = " + n67.getOnDelay(chans[0]);
        }
        else {
            return "Delays = " + getString(n67.getOnDelay(chans));
        }
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the output-off delay.
    ** 
    *************************************************************************** 
    */
    @Command(name="setoffdelay", description="Set the output-off delay")
    public void setOffDelay(@Argument(name="time",
                                      description="Output-off delay (sec)")
                            double time) throws DriverException
    {
        n67.setOffDelay(time, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the output-off delay.
    ** 
    *************************************************************************** 
    */
    @Command(name="setoffdelay", description="Set the output-off delay")
    public void setOffDelay(@Argument(name="time",
                                      description="Output-off delay (sec)")
                            double time,
                            @Argument(name="channel", description="Channel number")
                            int chan) throws DriverException
    {
        n67.setOffDelay(time, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the output-off delay.
    ** 
    *************************************************************************** 
    */
    @Command(name="showoffdelay", description="Show the output-off delay")
    public String getOffDelay() throws DriverException
    {
        return "Delay = " + n67.getOffDelay(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the output-off delay.
    ** 
    *************************************************************************** 
    */
    @Command(name="showoffdelay", description="Show the output-off delay")
    public String getOffDelay(@Argument(name="channel", description="Channel number")
                              String chanList) throws DriverException
    {
        int[] chans = getChannels(chanList);
        if (chans.length == 1) {
            return "Delay = " + n67.getOffDelay(chans[0]);
        }
        else {
            return "Delays = " + getString(n67.getOffDelay(chans));
        }
    }


   /**
    *************************************************************************** 
    ** 
    **  Converts a channel list string to an array of integers.
    ** 
    *************************************************************************** 
    */
    private int[] getChannels(String chanList) throws DriverException
    {
        String[] parts = chanList.split(",");
        int[] chans = new int[parts.length];
        try {
            for (int j = 0; j < chans.length; j++) {
                chans[j] = Integer.decode(parts[j]);
            }
            return chans;
        }
        catch (NumberFormatException e) {
            throw new DriverException(e.toString());
        }
    }


   /**
    *************************************************************************** 
    ** 
    **  Converts an array of numbers to a string.
    ** 
    *************************************************************************** 
    */
    private StringBuilder getString(double[] values)
    {
        StringBuilder text = new StringBuilder();
        text.append(values[0]);
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j]);
        }

        return text;
    }


   /**
    *************************************************************************** 
    ** 
    **  Converts a boolean to on/off.
    ** 
    *************************************************************************** 
    */
    private String getOnOff(boolean on)
    {
        return on ? "on" : "off";
    }

}
