package org.lsst.ccs.drivers.aerotech;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 *  General access routines for the AerotechPro165 device
 *
 *  @author Homer Neal
 */
public class AerotechPro165 extends Ascii {

    /**
     * Public constants
     */
    public static double timeout = 30000.;                // 30s timeout
    public static final int DEFAULT_PORT = 8000;          // Aerotech default port
    public static final int DEFAULT_BAUD = 9600;          // Aerotech default baud
    public static final String DEFAULT_DEV = "/dev/ttyS1"; // default serial connection

    /**
     * Private data
     */
    private static final char
        RESP_SUCCESS = '%',
        RESP_INVALID = '!',
        RESP_FAULT   = '#',
        RESP_TIMEOUT = '$';

    private double speed = 10.0;

    /**
     *  Constructor.
     */
    public AerotechPro165() {
        setOptions(Option.KEEP_ALIVE);
    }

    /**
     *  Opens a connection to the device.
     *
     *  This overrides the ASCII driver method and so is ultimately called
     *  by all the other open methods.
     *
     *  @param  type   The enumerated type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   device name for serial
     *  @param  parm1  The first device parameter:
     *                   port number for network;
     *                   baud rate for or serial
     *  @param  parm2  The second device parameter: always set to 0
     *
     *  @throws  DriverException
     */
    @Override
    public synchronized void open(ConnType type, String ident, int parm1, int parm2) throws DriverException {
        super.open(type, ident, parm1 == 0 ? type == ConnType.NET ? DEFAULT_PORT : DEFAULT_BAUD : parm1, 0);
        setTimeout(timeout); // set read timeout
        setTerminator(Terminator.LF); // set read and write terminators
    }

    /**
     * Gets the AerotechPro165 controller version
     *
     * @return
     * @throws DriverException
     */
    public String getVersion() throws DriverException {
        return readAP("VERSION");
    }

    /**
     * Gets the AerotechPro165 errors
     *
     * @return
     * @throws DriverException
     */
    public String getError() throws DriverException {
        return "x:" + readAP("AXISFAULT X") + "\n y:" + readAP("AXISFAULT Y") + "\n z:" + readAP("AXISFAULT Z");
    }

    /**
     * resets/clears the AerotechPro165 errors
     *
     * @throws DriverException
     */
    public void reset() throws DriverException {
        writeAP("ACKNOWLEDGEALL");
    }

    /**
     * resets the positioner - communication will be lost
     *
     * @throws DriverException
     */
    public void fullreset() throws DriverException {
        write("reset");
    }

    /**
     * abort any AerotechPro165 motion
     *
     * @throws DriverException
     */
    public void abort() throws DriverException {
        writeAP("ABORT X Y Z");
    }

    /**
     * Gets the AerotechPro165 x position
     *
     * @return
     * @throws DriverException
     */
    public double getPosX() throws DriverException {
        return readDoubleAP("PFBK X");
    }

    /**
     * Gets the AerotechPro165 x position
     *
     * @return
     * @throws DriverException
     */
    @Deprecated
    public double getPos_x() throws DriverException {
        return getPosX();
    }

    /**
     * Gets the AerotechPro165 y position
     *
     * @return
     * @throws DriverException
     */
    public double getPosY() throws DriverException {
        return readDoubleAP("PFBK Y");
    }

    /**
     * Gets the AerotechPro165 y position
     *
     * @return
     * @throws DriverException
     */
    @Deprecated
    public double getPos_y() throws DriverException {
        return getPosY();
    }

    /**
     * Gets the AerotechPro165 z position
     *
     * @return
     * @throws DriverException
     */
    public double getPosZ() throws DriverException {
        return readDoubleAP("PFBK Z");
    }

    /**
     * Gets the AerotechPro165 z position
     *
     * @return
     * @throws DriverException
     */
    @Deprecated
    public double getPos_z() throws DriverException {
        return getPosZ();
    }

    /**
     * Gets the AerotechPro165 x,y and z positions
     *
     * @return
     * @throws DriverException
     */
    public double[] getPosXYZ() throws DriverException {
        double p[] = new double[]{Double.NaN, Double.NaN, Double.NaN};

        p[0] = getPosX();
        p[1] = getPosY();
        p[2] = getPosZ();
        return (p);
    }

    /**
     * Gets the AerotechPro165 x, y & z positions
     *
     * @return
     * @throws DriverException
     */
    @Deprecated
    public double[] getPos_xyz() throws DriverException {
        return getPosXYZ();
    }

    /**
     * enable AerotechPro165 axes
     *
     * @param axis
     * @throws DriverException
     */
    public void enableAxis(char axis) throws DriverException {
        writeAP("ENABLE " + axis);
    }

    /**
     * disable AerotechPro165 axes
     *
     * @param axis
     * @throws DriverException
     */
    public void disableAxis(char axis) throws DriverException {
        writeAP("DISABLE " + axis);
    }

    /**
     * move AerotechPro165 to home position
     *
     * @throws DriverException
     */
    public void goHome() throws DriverException {
//        writeAP("MOVEABS X " + 0 + " Y " + 0 + " Z " + 0 + " F " + speed);
//        writeAP("HOME " + " F " + speed);
        writeAP("HOME X Y Z");
    }

    /**
     * Set up AerotechPro165 home parameters
     *
     * @param mode The mode to set
     * @throws DriverException
     */
    public synchronized void setupHome(int mode) throws DriverException {
        writeAP("setparm x, hometype, "+mode);
        writeAP("setparm y, hometype, "+mode);
        writeAP("setparm z, hometype, "+mode);
        writeAP("setparm z, homesetup, 0x00000001");
        writeAP("commitparameters");
        fullreset();
    }

    /**
     * ramp rate AerotechPro165 (units/s2)
     *
     * @param speed The ranp speed
     * @throws DriverException
     */
    public void setSpeed(double speed) throws DriverException {
        this.speed = speed;
    }

    /**
     * ramp rate AerotechPro165 (units/s2)
     *
     * @param rate
     * @throws DriverException
     */
    public void rampRate(double rate) throws DriverException {
        writeAP("RAMP RATE " + rate);
    }

    /**
     * ramp mode AerotechPro165 (units/s2)
     *
     * @param mode 
     * @throws DriverException
     */
    public void rampMode(String mode) throws DriverException {
        writeAP("RAMP MODE " + mode);
    }

    /**
     * set software low limit on axis motion
     *
     * @param axis
     * @param param
     * @throws DriverException
     */
    public void softLowLim(String axis, int param) throws DriverException {
        writeAP(axis + ", SoftwareLimitLow, " + param);
    }

    /**
     * set software high limit on axis motion
     *
     * @param axis
     * @param param
     * @throws DriverException
     */
    public void softHighLim(String axis, int param) throws DriverException {
        writeAP(axis + ", SoftwareLimitHigh, " + param);
    }


    /**
     * Move absolutely AerotechPro165 dx
     *
     * @param dx
     * @throws DriverException
     */
    public void moveAbsX(double dx) throws DriverException {
        writeAP("MOVEABS X " + dx + " F " + speed);
    }

    /**
     * Move absolutely AerotechPro165 dx
     *
     * @param dx
     * @throws DriverException
     */
    @Deprecated
    public void moveAbs_x(double dx) throws DriverException {
        moveAbsX(dx);
    }

    /**
     * move absolutely AerotechPro165 dy
     *
     * @param dy
     * @throws DriverException
     */
    public void moveAbsY(double dy) throws DriverException {
        writeAP("MOVEABS Y " + dy + " F " + speed);
    }

    /**
     * Move absolutely AerotechPro165 dy
     *
     * @param dy
     * @throws DriverException
     */
    @Deprecated
    public void moveAbs_y(double dy) throws DriverException {
        moveAbsY(dy);
    }

    /**
     * move absolutely AerotechPro165 dz
     *
     * @param dz
     * @throws DriverException
     */
    public void moveAbsZ(double dz) throws DriverException {
            writeAP("MOVEABS Z " + dz + " F " + speed);
    }

    /**
     * Move absolutely AerotechPro165 dz
     *
     * @param dz
     * @throws DriverException
     */
    @Deprecated
    public void moveAbs_z(double dz) throws DriverException {
        moveAbsZ(dz);
    }

    /**
     * move absolutely AerotechPro165 dx, dy
     *
     * @param dx
     * @param dy
     * @throws DriverException
     */
    public void moveAbs(double dx, double dy) throws DriverException {
        writeAP("MOVEABS X " + dx + " Y " + dy + " F " + speed);
    }

    /**
     * move absolutely AerotechPro165 dx, dy, dz
     *
     * @param dx
     * @param dy
     * @param dz
     * @throws DriverException
     */
    public void moveAbs(double dx, double dy, double dz) throws DriverException {
        writeAP("MOVEABS X " + dx + " Y " + dy + " Z " + dz);
    }

    /**
     * move incrementally AerotechPro165 dx
     *
     * @param dx
     * @throws DriverException
     */
    public void moveIncX(double dx) throws DriverException {
        writeAP("MOVEINC X " + dx + " F " + speed);
    }

    /**
     * move incrementally AerotechPro165 dx
     *
     * @param dx
     * @throws DriverException
     */
    @Deprecated
    public void moveInc_x(double dx) throws DriverException {
        moveIncX(dx);
    }

    /**
     * move incrementally AerotechPro165 dy
     *
     * @param dy
     * @throws DriverException
     */
    public void moveIncY(double dy) throws DriverException {
        writeAP("MOVEINC Y " + dy + " F " + speed);
    }

    /**
     * move incrementally AerotechPro165 dy
     *
     * @param dy
     * @throws DriverException
     */
    @Deprecated
    public void moveInc_y(double dy) throws DriverException {
        moveIncY(dy);
    }

    /**
     * move incrementally AerotechPro165 dz
     *
     * @param dz
     * @throws DriverException
     */
    public void moveIncZ(double dz) throws DriverException {
        writeAP("MOVEINC Z " + dz + " F " + speed);
    }

    /**
     * move incrementally AerotechPro165 dz
     *
     * @param dz
     * @throws DriverException
     */
    @Deprecated
    public void moveInc_z(double dz) throws DriverException {
        moveIncZ(dz);
    }

    /**
     * move incrementally AerotechPro165 dx, dy
     *
     * @param dx
     * @param dy
     * @throws DriverException
     */
    public void moveInc(double dx, double dy) throws DriverException {
        writeAP("MOVEINC X " + dx + " Y " + dy + " F " + speed);
    }

    /**
     * move incrementally AerotechPro165 dx, dy, dz
     *
     * @param dx
     * @param dy
     * @param dz
     * @throws DriverException
     */
    public void moveInc(double dx, double dy, double dz) throws DriverException {
        writeAP("MOVEINC X " + dx + " Y " + dy + " Z " + dz + " F " + speed);
    }

    /**
     * Writes a command with error checking.
     *
     ** @param command The command to write
     ** @throws DriverException
     */
    public void writeAP(String command) throws DriverException {
        checkResponse(read(command));
    }

    /**
     * Reads a string after writing a command with error checking.
     *
     * @param command The command to write, excluding terminator
     * @return The command response string
     * @throws DriverException
     * @throws DriverTimeoutException
     */
    public String readAP(String command) throws DriverException {
        String resp = read(command);
        checkResponse(resp);
        return resp.substring(1);
    }

    /**
     * Reads a double after writing a command with error checking.
     *
     * @param command The command to write, excluding terminator
     * @return The double number
     * @throws DriverException
     */
    public double readDoubleAP(String command) throws DriverException {
        String resp = readAP(command);
        try {
            return Double.parseDouble(resp);
        }
        catch (NumberFormatException e) {
            throw new DriverException("Invalid floating point response: " + resp);
        }
    }

    /**
     * Checks a response.
     *
     * @param resp The response to check
     * @throws DriverException
     */
    private void checkResponse(String resp) throws DriverException {
        String errMsg;
        if (resp.length() > 0) {
            char respChar = resp.charAt(0);
            if (respChar == RESP_SUCCESS) return;
            if (respChar == RESP_INVALID) {
                errMsg = "Invalid command";
            }
            else if (respChar == RESP_FAULT) {
                errMsg = "Command fault";
            }
            else if (respChar == RESP_TIMEOUT) {
                errMsg = "Command timeout";
            }
            else {
                errMsg = "Invalid response: " + resp;
            }
        }
        else {
            errMsg = "Empty response";
        }
        throw new DriverException(errMsg);
    }

    /**
     * Writes a command.
     *
     ** @param command The command to write
     ** @throws DriverException
     */
    @Deprecated
    public synchronized void writeAerotechPro165(String command) throws DriverException {
        write(command);
    }

    /**
     * Reads a response.
     *
     ** @return The command response string
     ** @throws DriverException
     */
    @Deprecated
    public synchronized String readAerotechPro165() throws DriverException {
        return read();
    }

    /**
     * Reads a response after writing a command.
     *
     * @param command The command to write, excluding terminator
     * @return The command response string
     * @throws DriverException
     * @throws DriverTimeoutException
     */
    @Deprecated
    public synchronized String readAerotechPro165(String command) throws DriverException {
        return read(command);
    }

}
