package org.lsst.ccs.drivers.aerotech;

import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverLockedException;

/**
 **************************************************************************
 **
 ** General access routines for the AerotechPro165 device
 *
 ** @author Homer Neal *
 * **************************************************************************
 */
public class AerotechPro165 extends Ascii {

    public static double timeout = 30000.;                // 30s timeout
    public static final int DEFAULT_BAUD = 9600;          // Aerotech default baud
    public static final String DEFAULT_DEV = "/dev/ttyS1"; // default serial connection

    /**
     **************************************************************************
     **
     ** Public constants *
     * **************************************************************************
     */
    /**
     **************************************************************************
     **
     ** Private constants *
     * **************************************************************************
     */
    private final String terminator = "\n";
    private byte[] buff = new byte[1024];
    private int buffIn, buffOut;

    // use locks to avoid interrupting communications
    private static final ReentrantLock KeyLock = new ReentrantLock();
    private double speed = 200.0;

    /**
     **************************************************************************
     **
     ** Constructor * *
     * **************************************************************************
     */
    public AerotechPro165() throws DriverException {
    }

    public void open(String devname, int port) throws DriverException {
        open(Ascii.CONN_TYPE_SERIAL, devname, port);
        setTimeout(timeout); // set SCPI timeout
//        reset();
    }

    /**
     ***************************************************************************
     **
     ** Opens a serial connection. *
     * **************************************************************************
     */
    public void openserial(String serialname, int port) throws DriverException {
        System.out.println("openning connection to the Aerotech");
        open(Ascii.CONN_TYPE_SERIAL, serialname, port);
        setTimeout(timeout); // set SCPI timeout
        setTerminator(terminator);

        // get logged in
        System.out.println("Sending returns to get its attention.");
        writeAerotechPro165("");
    }

    /**
     ***************************************************************************
     **
     ** Opens an FTDI connection. *
     * **************************************************************************
     */
    public void openftdi(String serialname, int port) throws DriverException {
        System.out.println("openning connection to the Aerotech");
        setTerminator(terminator);
        open(Ascii.CONN_TYPE_FTDI, serialname, port);
        setTimeout(timeout); // set SCPI timeout

        // get logged in
        System.out.println("Sending esc to get its attention.");
        writeAerotechPro165("");
    }

    /**
     ***************************************************************************
     **
     ** Opens a Network connection. (this is the primary communications used
     * for this device) Example: ./talkAscii.sh -t lf 130.199.47.180 8000
     * Connected. Type CTRL-D to exit, CR for read stats. VERSION %5.01.002
     * AXISSTATUS X %-1993863168 ENABLE X % ENABLE Y % MOVEINC X 2 Y 2 %
     * **************************************************************************
     * @param netaddress
     * @param port
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public void opennet(String netaddress, int port) throws DriverException {
        System.out.println("opening connection to the Aerotech");
        setTerminator(terminator);
        setTimeout(timeout); // set timeout
        open(Ascii.CONN_TYPE_NETWORK, netaddress, port);
    }

    /**
     **************************************************************************
     **
     ** Closes the connection. * * @throws DriverException
     *
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * *************************************************************************
     */
    @Override
    public void close() throws DriverException {
        super.close();
    }

    /**
     ***************************************************************************
     **
     ** Gets the AerotechPro165 controller version
     *
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public String getVersion() throws DriverException {
        return (readAerotechPro165("VERSION"));
    }

    /**
     ***************************************************************************
     **
     ** Gets the AerotechPro165 errors
     *
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public String getError() throws DriverException {
        return ("x:" + readAerotechPro165("AXISFAULT X") + "\n y:" + readAerotechPro165("AXISFAULT Y") + "\n z:" + readAerotechPro165("AXISFAULT Z"));
    }

    /**
     ***************************************************************************
     **
     ** resets/clears the AerotechPro165 errors
     *
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public void reset() throws DriverException {
        writeAerotechPro165("ACKNOWLEDGEALL");
    }

    /**
     ***************************************************************************
     **
     ** abort any AerotechPro165 motion
     *
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public void abort() throws DriverException {
        writeAerotechPro165("ABORT");
    }

    /**
     ***************************************************************************
     **
     ** Gets the AerotechPro165 x position
     *
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public double getPos_x() throws DriverException {
        double px = Double.NaN;
        try {
            KeyLock.lock(); // Waits for lock
            while (Double.isNaN(px)) {
                String rply = readAerotechPro165("PFBK X");
                System.out.println("getPos_x response: " + rply);
                try {
                    px = Double.parseDouble(rply.substring(1));
                } catch (NumberFormatException ex) {
                    System.out.println("Unable to decode " + rply.substring(1));
                }
            }
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
        return (px);
    }

    /**
     ***************************************************************************
     **
     ** Gets the AerotechPro165 x position
     *
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public double getPos_y() throws DriverException {
        double py = Double.NaN;
        try {
            KeyLock.lock(); // Waits for lock

            while (Double.isNaN(py)) {
                String rply = readAerotechPro165("PFBK Y");
                System.out.println("getPos_y response: " + rply);
                try {
                    py = Double.parseDouble(rply.substring(1));
                } catch (NumberFormatException ex) {
                    System.out.println("Unable to decode " + rply.substring(1));
                }
            }
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
        return (py);
    }

    /**
     ***************************************************************************
     **
     ** Gets the AerotechPro165 z position
     *
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public double getPos_z() throws DriverException {
        double pz = Double.NaN;
        try {
            KeyLock.lock(); // Waits for lock

            while (Double.isNaN(pz)) {
                String rply = readAerotechPro165("PFBK Z");
                System.out.println("getPos_z response: " + rply);
                try {
                    pz = Double.parseDouble(rply.substring(1));
                } catch (NumberFormatException ex) {
                    System.out.println("Unable to decode " + rply.substring(1));
                }
            }
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
        return (pz);
    }

    /**
     ***************************************************************************
     **
     ** Gets the AerotechPro165 x,y and z positions
     *
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public double[] getPos_xyz() throws DriverException {
        double p[] = new double[]{Double.NaN, Double.NaN, Double.NaN};

        p[0] = getPos_x();
        p[1] = getPos_y();
        p[2] = getPos_z();
        return (p);
    }

    /**
     ***************************************************************************
     **
     ** enable AerotechPro165 axes *
     * ********************************************************************
     * @param axis ***** @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public void enableAxis(char axis) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock

            System.out.println("Enabling axis " + axis);
            readAerotechPro165("ENABLE " + axis);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** disable AerotechPro165 axes *
     * ********************************************************************
     * @param axis ***** @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public void disableAxis(char axis) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock

            System.out.println("Disabling axis " + axis);
            readAerotechPro165("DISABLE " + axis);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** move absolutely AerotechPro165 dx, dy, dz *
     * **************************************************************************
     * @param dx
     * @param dy
     * @param dz
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public void moveAbs(double dx, double dy, double dz) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("moving absolutely dx, dy, dz = " + dx + " " + dy + " " + dz);
            readAerotechPro165("MOVEABS X " + dx + " Y " + dy + " Z " + dz);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** move AerotechPro165 to home position*
     * **************************************************************************
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public void goHome() throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("moving absolutely 0, 0, 0");
            readAerotechPro165("MOVEABS X " + 0 + " Y " + 0 + " Z " + 0);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** ramp rate AerotechPro165 (units/s2) *
     *
     * **************************************************************************
     */
    public void setSpeed(double speed) throws DriverException {

        this.speed = speed;
        System.out.println("setting speed = " + speed);
    }

    /**
     ***************************************************************************
     **
     ** ramp rate AerotechPro165 (units/s2) *
     *
     * **************************************************************************
     */
    public void rampRate(double rate) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("setting ramp rate = " + rate);
            readAerotechPro165("RAMP RATE " + rate);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** ramp mode AerotechPro165 (units/s2) *
     *
     * **************************************************************************
     */
    public void rampMode(String mode) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("setting ramp mode = " + mode);
            readAerotechPro165("RAMP MODE " + mode);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** move absolutely AerotechPro165 dx *
     * **************************************************************************
     */
    public void moveAbs_x(double dx) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("moving absolutely dx= " + dx);
            readAerotechPro165("MOVEABS X " + dx + " F " + speed);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** move absolutely AerotechPro165 dz *
     * **************************************************************************
     */
    public void moveAbs_z(double dz) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("moving absolutely dz= " + dz + " F " + speed);
            readAerotechPro165("MOVEABS Z " + dz);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** move absolutely AerotechPro165 dy *
     * **************************************************************************
     */
    public void moveAbs_y(double dy) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("moving absolutely dy= " + dy + " F " + speed);
            readAerotechPro165("MOVEABS Y " + dy);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** move absolutely AerotechPro165 dx, dy *
     * **************************************************************************
     */
    public void moveAbs(double dx, double dy) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("moving absolutely dx, dy = " + dx + " " + dy);
            readAerotechPro165("MOVEABS X " + dx + " Y " + dy + " F " + speed);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** move incrementally AerotechPro165 dx, dy, dz *
     * **************************************************************************
     */
    public void moveInc(double dx, double dy, double dz) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("moving incrementally dx, dy, dz = " + dx + " " + dy + " " + dz);
            readAerotechPro165("MOVEINC X " + dx + " Y " + dy + " Z " + dz + " F " + speed);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** move incrementally AerotechPro165 dx *
     * **************************************************************************
     */
    public void moveInc_x(double dx) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("moving incrementally dx= " + dx);
            readAerotechPro165("MOVEINC X " + dx + " F " + speed);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** move incrementally AerotechPro165 dz *
     * **************************************************************************
     */
    public void moveInc_z(double dz) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("moving incrementally dz= " + dz);
            readAerotechPro165("MOVEINC Z " + dz + " F " + speed);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** move incrementally AerotechPro165 dy *
     * **************************************************************************
     */
    public void moveInc_y(double dy) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("moving incrementally dy= " + dy);
            readAerotechPro165("MOVEINC Y " + dy + " F " + speed);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** move incrementally AerotechPro165 dx, dy *
     * **************************************************************************
     */
    public void moveInc(double dx, double dy) throws DriverException {

        try {
            KeyLock.lock(); // Waits for lock
            System.out.println("moving incrementally dx, dy = " + dx + " " + dy);
            readAerotechPro165("MOVEINC X " + dx + " Y " + dy + " F " + speed);
        } finally {
            if (KeyLock.isHeldByCurrentThread()) {
                KeyLock.unlock();
            }
        }
    }

    /**
     **************************************************************************
     **
     ** Writes a command.
     *
     ** @param The command to write
     ** @throws DriverException *
     * *************************************************************************
     */
    public synchronized void writeAerotechPro165(String command) throws DriverException {
        write(command + terminator);
    }

    /**
     **************************************************************************
     **
     ** Reads a response.
     *
     ** @return The command response string
     ** @throws DriverException
     *
     * *************************************************************************
     */
    public synchronized String readAerotechPro165() throws DriverException {
        return read();
    }

    /**
     **************************************************************************
     **
     ** Reads a response after writing a command. * * @param command The
     * command to write, excluding terminator * * @return The command response
     * string * * @throws DriverException
     *
     ** @throws DriverTimeoutException *
     * *************************************************************************
     */
    public synchronized String readAerotechPro165(String command) throws DriverException {
        writeAerotechPro165(command);
        return readAerotechPro165();
    }
}
