package org.lsst.ccs.drivers.ads.wrapper;

import com.sun.jna.Native;
import com.sun.jna.Pointer;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.logging.Level;
import org.lsst.ccs.drivers.ads.wrapper.ADSLibrary.AmsPort;
import org.lsst.ccs.drivers.ads.wrapper.ADSLibrary.ErrorCode;
import org.lsst.ccs.drivers.ads.wrapper.ADSLibrary.IndexGroup;
import org.lsst.ccs.drivers.ads.wrapper.ADSLibrary.TransMode;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Does the same things, with an added write test, as the C++ example program in
 * Beckhoff/ADS/examples on GitHub. The example PLC program
 * in that project should be running when you execute this program.
 * <p>
 * This program assumes that the remote AMS NetId is 5.56.86.54.1.1 
 * and that the corresponding IP address is 192.168.10.40. You may have to edit
 * the assignment to remoteNetId and remoteIpV4. The local AMS NetId will
 * be set to 134.79.86.183.1.1 so you'll need to have a route set in the remote
 * PLC controller for that NetId and the IP number of the machine you're
 * running this program on.
 * <p>
 * This program can be run using Run File under NetBeans.
 * @author tether
 */
public class Example1 {
    
    private Example1() {}

    private static final Logger LOG = Logger.getLogger(Example1.class.getPackage().getName());

    private final static ADSLibrary ADS = (ADSLibrary)
            Native.load("AdsJNA", ADSLibrary.class);

    public static void main(String[] args)  {
        final AmsNetId remoteNetId = new AmsNetId( 5, 56, 86, 54, 1, 1 );
        final String remoteIpV4 = "192.168.10.40";
        final AmsNetId localId = new AmsNetId(134, 79, 86, 183, 1, 1);
        final short remoteADSport = AmsPort.R0_PLC_TC3;

        LOG.log(Level.INFO, "The local AMS NetId will be set to {0}.", localId);
        LOG.log(Level.INFO, "The remote AMS NetId is {0}.", remoteNetId);
        LOG.log(Level.INFO, "That remote NetId will be routed to IP address {0}.", remoteIpV4);
        LOG.log(Level.INFO, "The remote ADS port is {0}.", remoteADSport);
        
        ADS.AdsSetLocalAddress(localId);

        // add local route to your EtherCAT Master
        if (ErrorCode.NOERR != ADS.AdsAddRoute(remoteNetId, remoteIpV4)) {
            LOG.severe("Adding the new ADS route failed, did you specify a valid IP address?");
            return;
        }
        LOG.info("The new ADS route has been added.");

        // Open a new ADS port.
        final long port = ADS.AdsPortOpenEx();
        if (port == 0) {
            LOG.severe("Failed to create a new local ADS port.");
            return;
        }
        LOG.info("The local ADS port has been created.");

        // Form the ADS address, AMS NetId and AMS port, of the server.
        final AmsAddr remote = new AmsAddr(remoteNetId, remoteADSport);
        
        readState(port, remote);
        getSymbolInfo(port, remote);
        writeByAddress(port, remote);
        readByAddress(port, remote);
        readByHandle(port, remote);
        try {
            notifyByAddress(port, remote);
            notifyByName(port, remote);
        }
        catch (IOException exc) {
            LOG.log(Level.SEVERE, "I/O problem!", exc);
        }
    }
    
    private static void readState(final long port, final AmsAddr server) {
        // Read the device state of the remote PLC.
        LOG.info("---------- Read state example ----------");
        short[] adsState = new short[1];
        short[] devState = new short[1];

        final long status = ADS.AdsSyncReadStateReqEx(port, server, adsState, devState);
        if (status != ErrorCode.NOERR) {
            LOG.severe("Reading the state failed with code " + status);
            return;
        }
        LOG.log(Level.INFO, "The ADS state is {0}, the device state is {1}.", new Object[]{adsState[0], devState[0]});
    }
    
    private static void getSymbolInfo(final long port, final AmsAddr server) {
        LOG.info("---------- Symbol info test ----------");
        final ByteBuffer rbuf = makeByteBuffer(3 * Integer.BYTES);
        final ByteBuffer wbuf = makeByteBuffer("MAIN.byByte");
        final long status =
            ADS.AdsSyncReadWriteReqEx2(
                port, server,
                IndexGroup.SYM_INFOBYNAME, 0,
                rbuf.capacity(), rbuf.array(),
                wbuf.capacity(), wbuf.array(),
                null);
        if (status != ErrorCode.NOERR) {
            LOG.log(Level.SEVERE, "Reading of symbol info failed with code {0}.", status);
        }
        final int igroup = rbuf.getInt();
        final int ioff = rbuf.getInt();
        final int size = rbuf.getInt();
        LOG.info(String.format("Symbol MAIN.byByte: group 0x%04x, offset %d, size %d.%n", igroup, ioff, size));
    }
    
    private static void readByAddress(final long port, final AmsAddr server) {
        LOG.info("---------- Read by address ----------");
        final ByteBuffer rbuf = makeByteBuffer(1);
        final int[] bytesRead = new int[]{0};
        for (int i = 0; i < 8; ++i) {
            final long status = ADS.AdsSyncReadReqEx2(
                    port, server,                   // ADS target.
                    IndexGroup.PLC_MEMORY_AREA, i,  // Index group and offset.
                    rbuf.capacity(), rbuf.array(),  //  Buffer spec.
                    bytesRead);
            if (status != ErrorCode.NOERR) {
                LOG.severe("The ADS read failed with code "  + status);
            }
            else {
                LOG.log(Level.INFO, "ADS read %d bytes at address {0}, value {1}.", new Object[]{bytesRead[0], i, rbuf.get(0)});
            }
        }

    }
    
    private static void readByHandle(final long port, final AmsAddr server) {
        LOG.info("---------- Read by handle ----------");
        final String varName = "MAIN.byByte[4]";
        LOG.info("The name of the variable to be read is " + varName);
        final int[] bytesRead = new int[]{0};
        final ByteBuffer rbuf = makeByteBuffer(1);
        final int handle = getNameHandle(port, server, varName);

        for (int i = 0; i < 8; ++i) {
            final long status = ADS.AdsSyncReadReqEx2(port,
                                                  server,
                                                  IndexGroup.SYM_VALBYHND,
                                                  handle,
                                                  rbuf.capacity(),
                                                  rbuf.array(),
                                                  bytesRead);
            if (status != ErrorCode.NOERR) {
                LOG.severe("The ADS read failed with code " + status);
            }
            else {
                LOG.log(Level.INFO, "ADS read {0} bytes, value = {1}.", new Object[]{bytesRead[0], rbuf.get(0)});
            }
        }
        releaseNameHandle(port, server, handle);
    }
    
    private static int getNameHandle(final long port, final AmsAddr server, final String varName) {
        final ByteBuffer rbuf = makeByteBuffer(Integer.BYTES);
        final ByteBuffer wbuf = makeByteBuffer(varName);
        final long handleStatus = ADS.AdsSyncReadWriteReqEx2(port,
                                                         server,
                                                         IndexGroup.SYM_HNDBYNAME,
                                                         0,
                                                         rbuf.capacity(),
                                                         rbuf.array(),
                                                         wbuf.capacity(),
                                                         wbuf.array(),
                                                         null);
        if (handleStatus != ErrorCode.NOERR) {
            LOG.severe("Creating the name handle failed with code " + handleStatus); 
        }
        // As far as I know all Beckhoff PLC controllers are little-endian.
        return rbuf.getInt();
    }

    private static void releaseNameHandle(final long port, final AmsAddr server, int handle)
    {
        final ByteBuffer wbuf = makeByteBuffer(Integer.BYTES);
        wbuf.putInt(handle);
        final long status = ADS.AdsSyncWriteReqEx(port, server, IndexGroup.SYM_RELEASEHND, 0, wbuf.capacity(), wbuf.array());
        if (status != ErrorCode.NOERR) {
            LOG.severe("Releasing the name handle failed with code " + status);
        }
    }

    private static void writeByAddress(final long port, final AmsAddr server) {
        LOG.info("---------- Write by address ----------");
        LOG.info("Writing to address 3 (MAIN.byByte[3]).");
        final ByteBuffer wbuf = makeByteBuffer(1);
        wbuf.put((byte)(System.currentTimeMillis() % 128 + 1));
        final long status = ADS.AdsSyncWriteReqEx(
        port, server,                   // ADS target.
        IndexGroup.PLC_MEMORY_AREA, 3,  // Index group and offset.
        wbuf.capacity(), wbuf.array()); //  Buffer spec.
        if (status != ErrorCode.NOERR) {
            LOG.severe("The ADS write failed with code "  + status);
        }
        else {
            LOG.log(Level.INFO, "ADS wrote {0} to address 3.", wbuf.get(0));
        }

    }

    private static void notifyByAddress(final long port, final AmsAddr server)
        throws IOException
    {
        LOG.info("---------- Notification by address ----------");
        LOG.info("Will ask for a notice when MAIN.byBytes[4] changes.");
        LOG.info("The remote PLC will check for changes every four seconds.");
        final AdsNotificationAttrib attrib = 
                new AdsNotificationAttrib(
                    1, // Callback length = no. of data bytes sent in notices.
                    TransMode.SERVERONCHA, // Notify when target value changes.
                    0, // Don't let a notice sit in the queue but send it immediately.
                    40000000 // The interval between checks in units of 100 ns.
                );
        final int[] hNotify = new int[]{0};
        final int hUser = 0;
        // We have to keep a ref to the callback object so that it doesn't
        // get garbage collected while we're using it.
        final PAdsNotificationFuncEx callback = Example1::notificationHandler;
        final long addStatus =
            ADS.AdsSyncAddDeviceNotificationReqEx(
                 port,
                 server,
                 IndexGroup.PLC_MEMORY_AREA,
                 4, // Address.
                 attrib,
                 callback,
                 hUser,
                 hNotify);
        if (addStatus != ErrorCode.NOERR) {
            LOG.log(Level.SEVERE, "The request for notification failed with code {0}.", addStatus);
            return;
        }
        LOG.info("Type any chanracter to stop notifications.");
        System.in.read();

        final long delStatus = ADS.AdsSyncDelDeviceNotificationReqEx(port, server, hNotify[0]);
        if (delStatus != ErrorCode.NOERR) {
            LOG.log(Level.SEVERE, "Deleting the device notification failed with code {0}.", delStatus);
        }
    }
     
    private static void notifyByName(final long port, final AmsAddr server)
        throws IOException
    {
        LOG.info("---------- Notification by name ----------");
        LOG.info("Will ask for a notice when MAIN.byBytes[4] changes.");
        LOG.info("The remote PLC will check for changes every four seconds.");
        final AdsNotificationAttrib attrib = 
                new AdsNotificationAttrib(
                    1, // Callback length = no. of data bytes sent in notices.
                    TransMode.SERVERONCHA, // Notify when target value changes.
                    0, // Don't let a notice sit in the queue but send it immediately.
                    40000000 // The interval between checks in units of 100 ns.
                );
        final int[] hNotify = new int[]{0};
        final int hUser = 0;
        final PAdsNotificationFuncEx callback = Example1::notificationHandler;
        final int handle = getNameHandle(port, server, "MAIN.byByte[4]");
        final long addStatus =
            ADS.AdsSyncAddDeviceNotificationReqEx(
                 port,
                 server,
                 IndexGroup.SYM_VALBYHND,
                 handle,
                 attrib,
                 callback,
                 hUser,
                 hNotify);
        if (addStatus != ErrorCode.NOERR) {
            LOG.log(Level.SEVERE, "The request for notification failed with code {0}.", addStatus);
            return;
        }
        LOG.info("Type any chanracter to stop notifications.");
        System.in.read();

        final long delStatus = ADS.AdsSyncDelDeviceNotificationReqEx(port, server, hNotify[0]);
        if (delStatus != ErrorCode.NOERR) {
            LOG.log(Level.SEVERE, "Deleting the device notification failed with code {0}.", delStatus);
        }
    }
   
    private static void notificationHandler(final AmsAddr addr, final AdsNotificationHeader header, int hUser)
    {
        final Pointer hdrptr = header.getPointer();
        LOG.log(Level.INFO,
                "Notice with handle {0} from {1} at time {2} with callback data {3} and {4} data bytes. Data =",
                new Object[]{header.hNotification, addr, header.nTimeStamp, hUser, header.cbSampleSize});
        final byte[] data = hdrptr.getByteArray(header.headerSize(), header.cbSampleSize);
        final StringBuilder buf = new StringBuilder();
        for (int i = 0; i < data.length; ++i) {
            buf.append(String.format(" %d", data[i]));
        }
        LOG.info(buf.toString());
    }

    private static ByteBuffer makeByteBuffer(final int size) {
        return makeByteBuffer(new byte[size]);
    }

    private static ByteBuffer makeByteBuffer(final byte[] array) {
        return ByteBuffer.wrap(array).order(ByteOrder.LITTLE_ENDIAN);
    }

    private static ByteBuffer makeByteBuffer(final String name) {
        return makeByteBuffer(name.getBytes(StandardCharsets.US_ASCII));
    }

}
