package org.lsst.ccs.drivers.ads;

import com.sun.jna.Native;
import com.sun.jna.Pointer;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Collection;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.drivers.ads.wrapper.ADSLibrary;
import org.lsst.ccs.drivers.ads.wrapper.ADSLibrary.AmsPort;
import org.lsst.ccs.drivers.ads.wrapper.ADSLibrary.ErrorCode;
import org.lsst.ccs.drivers.ads.wrapper.ADSLibrary.IndexGroup;
import org.lsst.ccs.drivers.ads.wrapper.ADSLibrary.TransMode;
import org.lsst.ccs.drivers.ads.wrapper.AdsNotificationAttrib;
import org.lsst.ccs.drivers.ads.wrapper.AdsNotificationHeader;
import org.lsst.ccs.drivers.ads.wrapper.AmsAddr;
import org.lsst.ccs.drivers.ads.wrapper.AmsNetId;
import org.lsst.ccs.drivers.ads.wrapper.PAdsNotificationFuncEx;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * Uses the ADS wrapper classes to provide a more limited but easier to use
 * interface to a remote PLC server via ADS. Designed with the Beckhoff
 * shutter controller in mind.
 * See <a href="https://confluence.slac.stanford.edu/x/mEI4Dg">this Confluence page</a>.
 * <p>
 * The exceptions defined in this class all inherit from the standard CCS exception
 * {@link org.lsst.ccs.drivers.commons.DriverException}
 * and so bear messages with more details about errors.
 * @author tether
 */
public class ADSDriver {
    
    public static final int NOTIFICATION_QUEUE_SIZE = 1000;
    
    // The constructors of these exception classes must be public so that
    // construction by reflection in checkStatus() is possible.
    /**
     * Thrown when the Beckhoff ADS library can't be initialized properly.
     */
    public static class InitializationException extends DriverException {
        public InitializationException(final String message) {super(message);}
    }
    
    /**
     * Thrown during an attempt to use the driver when it isn't open.
     */
    public static class NotOpenException extends DriverException {
        public NotOpenException(final String message) {super(message);}
    }
    
    /**
     * Thrown when an attempt to open the driver fails.
     */
    public static class OpenException extends DriverException {
        public OpenException(final String message) {super(message);}
    }
    
    /**
     * Thrown when the driver tries to do something and discovers that it
     * has lost contact with the remote ADS server.
     */
    public static class LostContactException extends DriverException {
        public LostContactException(final String message) {super(message);}
    }
    
    /**
     * Thrown when a variable name isn't in the remote PLC symbol table.
     */
    public static class LookupException extends DriverException {
        public LookupException(final String message) {super(message);}
    }
    
    /**
     * Thrown when the driver is given a VariableHandle that's been released or
     * which has been rejected as invalid by the remote ADS server.
     */
    public static class BadVarHandleException extends DriverException {
        public BadVarHandleException(final String message) {super(message);}
    }
    
    /**
     * Thrown when an attempt to establish notification fails.
     */
    public static class NotificationException extends DriverException {
        public NotificationException(final String message) {super(message);}
    }
    
    /**
     * Thrown when a problem arises while trying to read from or write to
     * a PLC variable.
     */
    public static class ReadWriteException extends DriverException {
        public ReadWriteException(final String message) {super(message);}
    }
    
    // Load the Beckhoff native library.
    private final ADSLibrary ADS;

    // The NetId used by the driver.
    private final AmsNetId localNetId; 
    
    // The AMS port useed by the driver.
    private final long localPort;
    
    // The callback object used to enqueue Notifications.
    private final PAdsNotificationFuncEx notificationCallback;
    
    private final BlockingQueue<Notification> notificationQueue;
    
    private final VarHandleTracker vhTracker;
    
    // The AMS address of the remote ADS server.
    private AmsAddr remoteAddr;

    /**
     * Sets the local AMS address and obtains an ADS port.
     * @param localAMSNetId The local AMS address as a dotted string, for example
     * "1.2.3.4.1.1". 
     * @throws InitializationException if an ADS port can't be obtained. 
     * @throws DriverException if the AMS Net Id is ill-formed.
     */
    public ADSDriver(final String localAMSNetId) throws DriverException {
        this(localAMSNetId, (ADSLibrary)Native.load("AdsJNA", ADSLibrary.class));
    }
 
    // Unit tests will call this with a mock ADSLibrary.
    ADSDriver(final String localAMSNetId, final ADSLibrary ADS) throws DriverException {
        this.ADS = ADS;
        localNetId = decodeNetId(localAMSNetId);
        ADS.AdsSetLocalAddress(localNetId);
        // Open a new ADS port.
        localPort = ADS.AdsPortOpenEx();
        if (localPort == 0) {
            throw new InitializationException("Failed to create a new local ADS port.");
        }
        notificationCallback = this::notificationHandler;
        notificationQueue = new ArrayBlockingQueue<>(NOTIFICATION_QUEUE_SIZE);
        remoteAddr = null;
        vhTracker = new VarHandleTracker();
    }
    
    
    
    /**
     * Establishes a connection to the PLC task of the remote ADS server.
     * Closes any existing connection.
     * @param remoteAMSNetId The remote AMS address as a dotted string, for example
     * "5.6.7.8.1.1".
     * @param remoteIPv4 A string containing the host name or the dotted IPv4
     * address of the remote server.
     * @throws OpenException if the remote server can't be reached.
     * @throws DriverException if the AMS netId is ill-formed.
     */
    public synchronized void open(final String remoteAMSNetId, final String remoteIPv4) throws DriverException {
        if (remoteAddr != null) {close();}
        remoteAddr = new AmsAddr(decodeNetId(remoteAMSNetId), AmsPort.R0_PLC_TC3);      
        // Add a local route to the PLC controller.
        checkStatus(ADS.AdsAddRoute(remoteAddr.netId, remoteIPv4),
                "Couldn't connect to the controller.", OpenException.class);
    }
    
   
    /** Closes any existing connection, releasing all {@link VariableHandle}s, 
     * canceling all notifications in effect and clearing the notification queue.
     * 
     */
    public synchronized void close() {
        if (remoteAddr != null) {
            ADS.AdsDelRoute(remoteAddr.netId);
        }
        remoteAddr = null;
        vhTracker.clear();
        notificationQueue.clear();
    }
    
    /**
     * Creates a description of a PLC variable in the MAIN POU.
     * @param varName The name of the variable, for example
     * "foo", "foo[5]", "foo.bar", etc. 
     * @return An instance of VariableHandle.
     * @throws NotOpenException if the driver hasn't been opened.
     * @throws LostContactException if the remote server can't be reached any more.
     * @throws LookupException if the variable's name can't be found in
     * the PLC symbol table.
     * @throws DriverException for any other error.
     * @see #releaseVariableHandle(org.lsst.ccs.drivers.ads.VariableHandle) 
     */
    public synchronized VariableHandle getVariableHandle(final String varName) throws DriverException {
        checkOpen();
        VariableHandle vh = vhTracker.getByName(varName);
        if (vh != null) {return vh;}
        // First get the handle itself.
        final ByteBuffer wbuf = makeByteBuffer("MAIN." + varName);
        ByteBuffer rbuf = makeByteBuffer(Integer.BYTES);
        final long handleStatus = ADS.AdsSyncReadWriteReqEx2(localPort,
            remoteAddr,
            IndexGroup.SYM_HNDBYNAME,
            0,
            rbuf.capacity(),
            rbuf.array(),
            wbuf.capacity(),
            wbuf.array(),
            null);
        checkStatus(handleStatus, "Creation of a variable handle failed.", LookupException.class);
        final int handleValue = rbuf.getInt();
        // Next get the size of the variable in bytes from the PLC symbol table.
        // We get index group, index offset and size, all 4-byte integers.
        rbuf = makeByteBuffer(3 * Integer.BYTES);
        final long status =
            ADS.AdsSyncReadWriteReqEx2(localPort, remoteAddr,
                IndexGroup.SYM_INFOBYNAME, 0,
                rbuf.capacity(), rbuf.array(),
                wbuf.capacity(), wbuf.array(),
                null);
        checkStatus(status, "Reading of symbol info failed.", LookupException.class);
        final int varSize = rbuf.getInt(2 * Integer.BYTES); // Skip group and offset.
        vh = new VariableHandle(handleValue, varName, varSize);
        vhTracker.add(vh);
        return vh;
    }
    
    /**
     * Takes a valid VariableHandle instance and makes it invalid. 
     * @param varHandle The VariableHandle to make invalid.
     * @throws NotOpenException if the driver isn't open.
     * @throws LostContactException if the driver can no longer get through
     * to the remote server.
     * @throws BadVarHandleException if the handle has already been released or
     * if the remote server rejects the attempt to release.
     * @throws DriverException for any other error.
     * @see #getVariableHandle(java.lang.String) 
     */
    public synchronized void releaseVariableHandle(final VariableHandle varHandle) throws DriverException {
        checkOpen();
        checkVarHandle(varHandle);
        final ByteBuffer wbuf = makeByteBuffer(Integer.BYTES);
        wbuf.putInt(varHandle.getHandleValue());
        final long status = ADS.AdsSyncWriteReqEx(localPort,
                remoteAddr,
                IndexGroup.SYM_RELEASEHND,
                0,
                wbuf.capacity(), wbuf.array());
        // If we couldn't release the var handle then it's likely that
        // the handle is invalid and the normal notification cancellation
        // won't work either.
        checkStatus(status, "Releasing the variable handle failed.", BadVarHandleException.class);
        cancelNotifications(varHandle);
        vhTracker.remove(varHandle);
    }
    
    /**
     * Sets up notifications for the value of a PLC variable. Notifications
     * are represented as instances of Notification placed in the
     * notification queue.
     * @param varHandle A valid VariableHandle.
     * @param maxDelay How long the notification messages are allowed to
     * remain queued in the remote server before they are sent.
     * @param checkInterval How often the remote server should decide
     * whether or not to send a new message.
     * @param onlyIfChanged If this is true then the remote server will
     * suppress a new message if the value of the variable hasn't changed since
     * the sending of the previous message. Otherwise it will send the message
     * every time the check interval elapses.
     * @return The 32-bit handle assigned to the notification by the remote server.
     * @throws NotOpenException if the driver isn't open.
     * @throws LostContactException if the driver can no longer get through
     * to the remote server.
     * @throws BadVarHandleException if the handle has already been released or
     * was rejected by the remote server.
     * @throws NotificationException if the request fails for some other reason.
     * @throws DriverException for any other error.
     * @see Notification
     * @see #getVariableHandle(java.lang.String) 
     */
    public synchronized int requestNotifications(
            final VariableHandle varHandle,
            final Duration maxDelay,
            final Duration checkInterval,
            final boolean onlyIfChanged)
        throws DriverException
    {
        checkOpen();
        checkVarHandle(varHandle);
        final Integer oldHandle = vhTracker.getNotification(varHandle);
        if (oldHandle != null) {return oldHandle;}
        final AdsNotificationAttrib attrib = 
                new AdsNotificationAttrib(
                    varHandle.getVarSize(),
                    onlyIfChanged ? TransMode.SERVERONCHA : TransMode.SERVERCYCLE,
                    (int)maxDelay.toNanos() / 100,
                    (int)checkInterval.toNanos() / 100);
        final int[] hNotify = new int[]{0};
        final int hUser = 0;
        final long addStatus =
            ADS.AdsSyncAddDeviceNotificationReqEx(
                 localPort,
                 remoteAddr,
                 IndexGroup.SYM_VALBYHND,
                 varHandle.getHandleValue(),
                 attrib,
                 notificationCallback,
                 hUser,
                 hNotify);
        checkStatus(addStatus, "The request for notifications failed.", NotificationException.class);
        vhTracker.addNotification(varHandle, hNotify[0]);
        return hNotify[0];
    }
    
    /**
     * Cancels notifications for a given PLC variable.
     * @param varHandle A valid instance of VariableHandle.
     * @throws NotOpenException if the driver isn't open.
     * @throws BadVarHandleException if the variable handle isn't valid.
     * @throws LostContactException if communications with the server have been lost.
     * @throws NotificationException if the canceling failed on the remote server.
     * @throws DriverException for any other error.
     * @see #requestNotifications(org.lsst.ccs.drivers.ads.VariableHandle, java.time.Duration, java.time.Duration, boolean) 
     */
    public synchronized void cancelNotifications(final VariableHandle varHandle)
            throws DriverException
    {
        checkOpen();
        checkVarHandle(varHandle);
        // Find the notification handle (if any).
        final Integer nh = vhTracker.removeNotification(varHandle);
        // Tell the PLC controller to stop notifications.
        if (nh != null) {
            final long status = ADS.AdsSyncDelDeviceNotificationReqEx(localPort, remoteAddr, nh);
            checkStatus(status, "Canceling notifications failed.", NotificationException.class);
        }
    }
    
    /**
     * Reads the value of a given PLC variable.
     * @param varHandle A valid VariableHandle instance.
     * @return A read-write byte buffer with the correct byte order, with position = 0 and
     * limit and capacity both equal to the variable size.
     * @throws NotOpenException if the driver isn't open.
     * @throws BadVarHandleException if the variable handle isn't valid.
     * @throws LostContactException if communications with the server have been lost.
     * @throws ReadWriteException if the reading failed on the remote server.
     * @throws DriverException for any other error.
     */
    public synchronized ByteBuffer readVariable(final VariableHandle varHandle) throws DriverException {
        checkOpen();
        checkVarHandle(varHandle);
        final ByteBuffer rbuf = makeByteBuffer(varHandle.getVarSize());
        final long status = 
             ADS.AdsSyncReadReqEx2(
                localPort, remoteAddr,
                IndexGroup.SYM_VALBYHND, varHandle.getHandleValue(),
                rbuf.capacity(), rbuf.array(),
                null);
        checkStatus(status, "Reading a variable value failed.", ReadWriteException.class);
        return rbuf;
    }
    
    /**
     * Writes the value of a given PLC value.
     * @param varHandle A valid instance of VariableHandle.
     * @param wbuf A byte buffer whose capacity is exactly the same as the
     * variable size. All the bytes will be written. I recommend that
     * you obtain a buffer of the right size and byte order using
     * {@link VariableHandle#createBuffer()} or
     * {@link #readVariable(org.lsst.ccs.drivers.ads.VariableHandle)}.
     * @throws NotOpenException if the driver isn't open.
     * @throws BadVarHandleException if the variable handle isn't valid.
     * @throws LostContactException if communications with the server have been lost.
     * @throws ReadWriteException If the writing failed on the remote server.
     * @throws DriverException for any other error.
     */
    public synchronized void writeVariable(final VariableHandle varHandle, final ByteBuffer wbuf) throws DriverException {
        checkOpen();
        checkVarHandle(varHandle);
        if (wbuf.capacity() != varHandle.getVarSize()) {
            throw new ReadWriteException("wbuf.capacity() != variable size.");
        }
        final long status = 
             ADS.AdsSyncWriteReqEx(
                localPort, remoteAddr,
                IndexGroup.SYM_VALBYHND, varHandle.getHandleValue(),
                wbuf.capacity(), wbuf.array());
        checkStatus(status, "Writing a variable value failed.", ReadWriteException.class);
    }

    /**
     * @see BlockingQueue#take() 
     */
    public Notification takeNotification() throws InterruptedException {
        return notificationQueue.take();
    }

    /**
     * @see BlockingQueue#poll()
     */
    public Notification pollNotification() {
        return notificationQueue.poll();
    }
    
    /**
     * @see BlockingQueue#poll(long, java.util.concurrent.TimeUnit) 
     */
    public Notification pollNotification(long timeout, TimeUnit unit) throws InterruptedException {
        return notificationQueue.poll(timeout, unit);
    }

    /**
     * @see BlockingQueue#drainTo(java.util.Collection) 
     */
    public int drainNotifications(Collection<Notification> collectn) {
        return notificationQueue.drainTo(collectn);
    }

    /**
     * @see BlockingQueue#drainTo(java.util.Collection, int) 
     */
    public int drainNotifications(Collection<Notification> collectn, int maxElements) {
        return notificationQueue.drainTo(collectn, maxElements);
    }
    
    private static ByteBuffer makeByteBuffer(final int size) {
        return makeByteBuffer(new byte[size]);
    }

    private static ByteBuffer makeByteBuffer(final byte[] array) {
        return ByteBuffer.wrap(array).order(ByteOrder.LITTLE_ENDIAN);
    }

    private static ByteBuffer makeByteBuffer(final String name) {
        return makeByteBuffer(name.getBytes(StandardCharsets.US_ASCII));
    }
    
    private static final Pattern NETID_PATTERN = Pattern.compile("(\\d+)\\.(\\d+)\\.(\\d+)\\.(\\d+)\\.1\\.1");
    
    private static AmsNetId decodeNetId(final String id) throws DriverException {
        final Matcher mat = NETID_PATTERN.matcher(id);
        if (mat.matches()) {
            return new AmsNetId(
                (byte)Integer.parseInt(mat.group(1)),
                (byte)Integer.parseInt(mat.group(2)),
                (byte)Integer.parseInt(mat.group(3)),
                (byte)Integer.parseInt(mat.group(4)),
                1,
                1
            );
        }
        else {
            throw new DriverException("Ill-formed AMS NetId.");
        }
    }
       
    private void notificationHandler(final AmsAddr addr, final AdsNotificationHeader header, int hUser)
    {
        // For each notification handle the Beckhoff ADS library maintains
        // a single buffer that's overwritten each time a new notification
        // message comes in for that handle. What we need from the message
        // we therefore have to copy before exiting this callback. The
        // notification header is at the beginning of the buffer followed
        // immediately by the data bytes from PLC memory.
        final Pointer bufptr = header.getPointer();
        // Skip past the header to extract the data bytes.
        final ByteBuffer data =
            ByteBuffer.wrap(
                bufptr.getByteArray(AdsNotificationHeader.headerSize(), header.cbSampleSize))
            .asReadOnlyBuffer()  // Returned buffer has default byte order.
            .order(ByteOrder.LITTLE_ENDIAN);
        final VariableHandle varHandle = vhTracker.getByNotification(header.hNotification);
        final Notification notice = new Notification(header.hNotification, varHandle, data);
        try {
            notificationQueue.put(notice);
        }
        catch (InterruptedException exc) {
            Thread.currentThread().interrupt(); // Handle this at a higher level.
        }
    }

    private void checkOpen() throws DriverException {
        if (remoteAddr == null) {
            throw new NotOpenException("open() hasn't been called.");
        }
    }
    
    private void checkVarHandle(final VariableHandle varHandle) throws DriverException {
        if (!vhTracker.isValid(varHandle)) {
            throw new BadVarHandleException("That variable handle was released and is no longer valid.");
        }
    }
    
    private static final Set<Long> LOST_COMM_STATUS;
    static {
        // See the page on "ADS return codes" in the Beckhoff infosystem. 
        LOST_COMM_STATUS = new TreeSet<>();
        LOST_COMM_STATUS.add(0x6L);  // Target port not found.
        LOST_COMM_STATUS.add(0xdL); // Port not connected.
        LOST_COMM_STATUS.add(0x12L);  // Port disabled.
        LOST_COMM_STATUS.add(0X1AL); // TCP send error.
        LOST_COMM_STATUS.add(0x1bL); // Host unreachable.
        LOST_COMM_STATUS.add(0x274cL); // WINSOCK - host unreachable.
        LOST_COMM_STATUS.add(0x274dL); // WINSOCK - host failed to respond.
        LOST_COMM_STATUS.add(0x748L); // ADS port not opened.
    }

    private void checkStatus(final long status, final String msg, final Class<?> klass) throws DriverException {
        if (status != ErrorCode.NOERR) {
            final String codeMsg = String.format("%s Error code 0x%04x.", msg, status);
            if (LOST_COMM_STATUS.contains(status)) {
                throw new LostContactException(codeMsg);
            }
            try {
                throw (DriverException)klass.getConstructor(String.class).newInstance(codeMsg);
            }
            catch (ReflectiveOperationException exc) {
                throw new DriverException("BUG - can't create the right kind of exception.", exc);
            }
        }
    }
}

