package org.lsst.ccs.drivers.ads;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;

/**
 * Contains information about a PLC variable in the remote server.
 * Instances are generated by the ADS driver.
 * @see ADSDriver#getVariableHandle(java.lang.String) 
 * @see ADSDriver#releaseVariableHandle(org.lsst.ccs.drivers.ads.VariableHandle) 
 * @author tether
 */
public class VariableHandle {
    
    private final int handleValue;
    private final String name;
    private final int varSize;

    // Only the driver may create instances.
    VariableHandle(final int handle, final String name, final int varSize) {
        this.handleValue = handle;
        this.name = name;
        this.varSize = varSize;
    }

    /**
     * Gets the 32-bit handle assigned to the variable by the remote server.
     * @return The handle.
     */
    public int getHandleValue() {
        return handleValue;
    }

    /**
     * Gets the variable name passed to
     * {@link ADSDriver#getVariableHandle(java.lang.String)}.
     * @return The name string.
     */
    public String getName() {
        return name;
    }

    /**
     * Gets the size of the PLC variable in bytes.
     * @return The size.
     */
    public int getVarSize() {
        return varSize;
    }
    
    /**
     * Creates a read-write ByteBuffer that has the correct byte order, position=0 with
     * capacity and limit both set to the variable size.
     * @return The buffer.
     */
    public ByteBuffer createBuffer() {
        return ByteBuffer.wrap(new byte[getVarSize()]).order(ByteOrder.LITTLE_ENDIAN);
    }
    
}
