package org.lsst.ccs.drivers.ads.wrapper;

import com.sun.jna.Library;
import com.sun.jna.Structure;

/**
 *
 * @author tether
 */
public interface ADSLibrary extends Library {

    /**
     * Add a new AMS route to a target system.
     * @param ams (In) AMS address of the target system.
     * @param ip (In) IP address of the target system.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsAddRoute(AmsNetId ams, String ip);

    /**
     * Delete an AMS route that had previously been added with AdsAddRoute().
     * @param ams (In) address of the target system
     */
    void AdsDelRoute(AmsNetId ams);

    /**
     * The connection (communication port) to the message router is
     * closed. The port to be closed must previously have been opened via
     * an AdsPortOpenEx() call.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsPortCloseEx(long port);

    /**
     * Establishes a connection (communication port) to the message
     * router. The port number returned by AdsPortOpenEx() is required as
     * parameter for further AdsLib function calls.
     * @return port number of a new Ads port or 0 if no more ports available
     */
    long AdsPortOpenEx();

    /**
     * Returns the local NetId and port number.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (Out) Reference to the structure of type AmsAddr.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsGetLocalAddressEx(long port, AmsAddr pAddr);

    /**
     * Change local NetId.
     * @param ams (In) local AmsNetId
     */
    void AdsSetLocalAddress(AmsNetId ams);

    /**
     * Reads data synchronously from an ADS server.
     * @param port (In)  port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param indexGroup (In) Index Group.
     * @param indexOffset (In) Index Offset.
     * @param bufferLength (In) Length of the data in bytes.
     * @param buffer (Out) Pointer to a data buffer that will receive the data.
     * @param bytesRead (Out) Pointer to a variable. If successful, this variable will return the number of actually read data bytes.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncReadReqEx2(long      port,
                           AmsAddr   pAddr,
                           int       indexGroup,
                           int       indexOffset,
                           int       bufferLength,
                           byte[]    buffer,
                           int[]     bytesRead);

    /**
     * Reads the identification and version number of an ADS server.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param devName (Out) Pointer to a character string of at least 16 bytes, that will receive the name of the ADS device.
     * @param version (Out) Address of a variable of type AdsVersion, which will receive the version number, revision number and the build number.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncReadDeviceInfoReqEx(
            long port,
            AmsAddr pAddr,
            String devName,
            AdsVersion version);

    /**
     * Reads the ADS status and the device status from an ADS server.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param adsState (Out) Address of a variable that will receive the ADS status (see data type [ADSSTATE](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/tcadsdll_enumadsstate.htm?id=17630)).
     * @param devState (Out) Address of a variable that will receive the device status.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncReadStateReqEx(long port, AmsAddr pAddr, short[] adsState, short[] devState);

    /**
     * Writes data synchronously into an ADS server and receives data back from the ADS server.
     * @param port  (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param indexGroup (In) Index Group.
     * @param indexOffset (In) Index Offset.
     * @param readLength (In) Length, in bytes, of the read buffer readData.
     * @param readData (Out) Buffer for data read from the ADS server.
     * @param writeLength (In) Length of the data, in bytes, send to the ADS server.
     * @param writeData (In) Buffer with data send to the ADS server.
     * @param bytesRead (Out) pointer to a variable. If successful, this variable will return the number of actually read data bytes.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncReadWriteReqEx2(long           port,
                                AmsAddr        pAddr,
                                int            indexGroup,
                                int            indexOffset,
                                int            readLength,
                                byte[]         readData,
                                int            writeLength,
                                byte[]         writeData,
                                int[]          bytesRead);

    /**
     * Writes data synchronously to an ADS server.
     * @param port  (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param indexGroup (In) Index Group.
     * @param indexOffset (In) Index Offset.
     * @param bufferLength (In) Length, in bytes, of the read buffer readData.
     * @param buffer (Out) Buffer for data read from the ADS server.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncWriteReqEx(long           port,
                           AmsAddr        pAddr,
                           int            indexGroup,
                           int            indexOffset,
                           int            bufferLength,
                           byte[]         buffer);

    /**
     * Changes the ADS status and the device status of an ADS server.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param adsState (In) New ADS status.
     * @param devState (In) New device status.
     * @param bufferLength (In) Length of the additional data, in bytes, send to the ADS server.
     * @param buffer (In) Buffer with additional data send to the ADS server.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncWriteControlReqEx(long           port,
                                  AmsAddr        pAddr,
                                  short          adsState,
                                  short          devState,
                                  int            bufferLength,
                                  byte[]         buffer);

    /**
     * A notification is defined within an ADS server (e.g. PLC). When a
     * certain event occurs a function (the callback function) is invoked in
     * the ADS client (C program).
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param indexGroup (In) Index Group.
     * @param indexOffset (In) Index Offset.
     * @param pAttrib (In) Pointer to the structure that contains further information.
     * @param pFunc (In) Pointer to the structure describing the callback function.
     * @param hUser (In) 32-bit value that is passed to the callback function.
     * @param pNotification (Out) Address of the variable that will receive the handle of the notification.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncAddDeviceNotificationReqEx(long                         port,
                                           AmsAddr                      pAddr,
                                           int                          indexGroup,
                                           int                          indexOffset,
                                           AdsNotificationAttrib        pAttrib,
                                           PAdsNotificationFuncEx       pFunc,
                                           int                          hUser,
                                           int[]                        pNotification);

    /**
     * A notification defined previously is deleted from an ADS server.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param hNotification (In) The handle of the notification.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncDelDeviceNotificationReqEx(long port, AmsAddr pAddr, int hNotification);

    /**
     * Read the configured timeout for the ADS functions. The standard value is 5000 ms.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param timeout (Out) Buffer to store timeout value in ms.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncGetTimeoutEx(long port, int[] timeout);

    /**
     * Alters the timeout for the ADS functions. The standard value is 5000 ms.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param timeout (In) Timeout in ms.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncSetTimeoutEx(long port, int timeout);

    /**
     * Transmission modes for ADS notifications.
     */
    public static class TransMode {
        
        private TransMode() {}
        
        /**
         * Send the notification whenever the cycle time specified
         * in the notification request has elapsed, regardless of whether the
         * variable being monitored has changed its value.
         */
        public static final int    SERVERCYCLE = 3;
        
        /**
         * Send the notification whenever the cycle time specified
         * in the notification request has elapsed, but only if the
         * variable being monitored has changed its value.
         */
        public static final int    SERVERONCHA = 4;
    }

    /**
     * Possible states for an ADS device.
     */
    public static class DeviceState {
        private DeviceState() {}
        public static final int    INVALID = 0;
        public static final int    IDLE = 1;
        public static final int    RESET = 2;
        public static final int    INIT = 3;
        public static final int    START = 4;
        public static final int    RUN = 5;
        public static final int    STOP = 6;
        public static final int    SAVECFG = 7;
        public static final int    LOADCFG = 8;
        public static final int    POWERFAILURE = 9;
        public static final int    POWERGOOD = 10;
        public static final int    ERROR = 11;
        public static final int    SHUTDOWN = 12;
        public static final int    SUSPEND = 13;
        public static final int    RESUME = 14;
        public static final int    CONFIG = 15;
        public static final int    RECONFIG = 16;
        public static final int    STOPPING = 17;
        public static final int    INCOMPATIBLE = 18;
        public static final int    EXCEPTION = 19;
    }

    /**
     * ADS error codes.
     */
    public static class ErrorCode {
        private ErrorCode() {}
        public static final int NOERR = 0;
        public static final int ADS_ERROR_BASE = 1792;
        public static final int DEVICE_LICENSEINVALID = 39 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDOFFSET = 3 + ADS_ERROR_BASE;
        public static final int DEVICE_SYNTAX = 13 + ADS_ERROR_BASE;
        public static final int DEVICE_SYMBOLVERSIONINVALID = 17 + ADS_ERROR_BASE;
        public static final int DEVICE_LICENSEDUPLICATED = 45 + ADS_ERROR_BASE;
        public static final int DEVICE_SYMBOLNOTFOUND = 16 + ADS_ERROR_BASE;
        public static final int DEVICE_LICENSETIMETOLONG = 43 + ADS_ERROR_BASE;
        public static final int CLIENT_VARUSED = 67 + ADS_ERROR_BASE;
        public static final int DEVICE_NOTINIT = 24 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDACCESS = 4 + ADS_ERROR_BASE;
        public static final int DEVICE_BUSY = 8 + ADS_ERROR_BASE;
        public static final int CLIENT_NOMORESYM = 83 + ADS_ERROR_BASE;
        public static final int CLIENT_REMOVEHASH = 82 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDCONTEXT = 9 + ADS_ERROR_BASE;
        public static final int CLIENT_SYNCTIMEOUT = 69 + ADS_ERROR_BASE;
        public static final int DEVICE_NOTIFYHNDINVALID = 20 + ADS_ERROR_BASE;
        public static final int DEVICE_LICENSEEXCEEDED = 38 + ADS_ERROR_BASE;
        public static final int CLIENT_SYNCRESINVALID = 84 + ADS_ERROR_BASE;
        public static final int DEVICE_CERTIFICATEINVALID = 47 + ADS_ERROR_BASE;
        public static final int DEVICE_PENDING = 30 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDOBJID = 29 + ADS_ERROR_BASE;
        public static final int DEVICE_NOMEMORY = 10 + ADS_ERROR_BASE;
        public static final int DEVICE_NOTREADY = 7 + ADS_ERROR_BASE;
        public static final int DEVICE_WARNING = 32 + ADS_ERROR_BASE;
        public static final int DEVICE_LICENSESYSTEMID = 40 + ADS_ERROR_BASE;
        public static final int DEVICE_ABORTED = 31 + ADS_ERROR_BASE;
        public static final int CLIENT_LISTEMPTY = 66 + ADS_ERROR_BASE;
        public static final int DEVICE_CLIENTUNKNOWN = 21 + ADS_ERROR_BASE;
        public static final int CLIENT_TIMEOUTINVALID = 71 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDWATCHSIZE = 23 + ADS_ERROR_BASE;
        public static final int DEVICE_ERROR = 0 + ADS_ERROR_BASE;
        public static final int DEVICE_LICENSENOTIMELIMIT = 41 + ADS_ERROR_BASE;
        public static final int DEVICE_SIGNATUREINVALID = 46 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDSIZE = 5 + ADS_ERROR_BASE;
        public static final int CLIENT_PORTNOTOPEN = 72 + ADS_ERROR_BASE;
        public static final int DEVICE_LICENSEFUTUREISSUE = 42 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDPARM = 11 + ADS_ERROR_BASE;
        public static final int CLIENT_SYNCINTERNAL = 80 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDCLSID = 28 + ADS_ERROR_BASE;
        public static final int DEVICE_NOINTERFACE = 26 + ADS_ERROR_BASE;
        public static final int DEVICE_LICENSEEXPIRED = 37 + ADS_ERROR_BASE;
        public static final int CLIENT_W32ERROR = 70 + ADS_ERROR_BASE;
        public static final int DEVICE_SYMBOLNOTACTIVE = 34 + ADS_ERROR_BASE;
        public static final int CLIENT_ADDHASH = 81 + ADS_ERROR_BASE;
        public static final int DEVICE_INCOMPATIBLE = 14 + ADS_ERROR_BASE;
        public static final int DEVICE_ACCESSDENIED = 35 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDDATA = 6 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDINTERFACE = 27 + ADS_ERROR_BASE;
        public static final int DEVICE_EXISTS = 15 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDSTATE = 18 + ADS_ERROR_BASE;
        public static final int DEVICE_LICENSENOTFOUND = 36 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDARRAYIDX = 33 + ADS_ERROR_BASE;
        public static final int DEVICE_NOTFOUND = 12 + ADS_ERROR_BASE;
        public static final int DEVICE_INVALIDGRP = 2 + ADS_ERROR_BASE;
        public static final int DEVICE_TRANSMODENOTSUPP = 19 + ADS_ERROR_BASE;
        public static final int CLIENT_INVALIDPARM = 65 + ADS_ERROR_BASE;
        public static final int CLIENT_NOAMSADDR = 73 + ADS_ERROR_BASE;
        public static final int DEVICE_NOMOREHDLS = 22 + ADS_ERROR_BASE;
        public static final int DEVICE_SRVNOTSUPP = 1 + ADS_ERROR_BASE;
        public static final int CLIENT_DUPLINVOKEID = 68 + ADS_ERROR_BASE;
        public static final int DEVICE_TIMEOUT = 25 + ADS_ERROR_BASE;
        public static final int DEVICE_EXCEPTION = 44 + ADS_ERROR_BASE;
        public static final int CLIENT_SYNCPORTLOCKED = 85 + ADS_ERROR_BASE;
        public static final int CLIENT_ERROR = 64 + ADS_ERROR_BASE;
    }

    /**
     * The index group numbers meaningful to ADS.
     */
    public static class IndexGroup {
        private IndexGroup() {}
        /** Area of variables mapped with %M in PLC code. */
        public static final int PLC_MEMORY_AREA = 0x4020;
        public static final int SYM_INFOBYNAME = 61447;
        public static final int SYM_DOWNLOAD = 61450;
        public static final int SYMTAB = 61440;
        public static final int DEVICE_DATA = 61696;
        public static final int SYM_RELEASEHND = 61446;
        public static final int SYM_UPLOADINFO = 61452;
        public static final int IOIMAGE_ROSIZE = 61493;
        public static final int SYMNAME = 61441;
        public static final int IOIMAGE_RWOB = 61488;
        public static final int SYM_UPLOAD = 61451;
        public static final int IOIMAGE_RWIOB = 61536;
        public static final int SYM_UPLOADINFO2 = 61455;
        public static final int SYM_DOWNLOAD2 = 61453;
        public static final int IOIMAGE_CLEARI = 61504;
        public static final int IOIMAGE_RWIB = 61472;
        public static final int IOIMAGE_CLEARO = 61520;
        public static final int SUMUP_READEX2 = 61572;
        public static final int SUMUP_DELDEVNOTE = 61574;
        public static final int SUMUP_READ = 61568;
        public static final int SYMVAL = 61442;
        public static final int SYM_VALBYHND = 61445;
        public static final int SUMUP_WRITE = 61569;
        public static final int IOIMAGE_RWOX = 61489;
        public static final int SUMUP_READEX = 61571;
        public static final int SYM_VALBYNAME = 61444;
        public static final int SYM_INFOBYNAMEEX = 61449;
        public static final int SYM_VERSION = 61448;
        public static final int SYMNOTE = 61456;
        public static final int SYM_DT_UPLOAD = 61454;
        public static final int SUMUP_ADDDEVNOTE = 61573;
        public static final int SYM_HNDBYNAME = 61443;
        public static final int SUMUP_READWRITE = 61570;
        public static final int IOIMAGE_RISIZE = 61477;
        public static final int IOIMAGE_RWIX = 61473;
    }

    public static class AmsPort {
        private AmsPort() {}
        public static final short LOGGER                  = 100;
        public static final short R0_RTIME                = 200;
        public static final short R0_TRACE                = (short)(R0_RTIME + 90);
        public static final short R0_IO                   = 300;
        public static final short R0_SPS                  = 400;
        public static final short R0_NC                   = 500;
        public static final short R0_ISG                  = 550;
        public static final short R0_PCS                  = 600;
        public static final short R0_PLC                  = 801;
        public static final short R0_PLC_RTS1             = 801; // TwinCAT2 PLC task.
        public static final short R0_PLC_RTS2             = 811;
        public static final short R0_PLC_RTS3             = 821;
        public static final short R0_PLC_RTS4             = 831;
        public static final short R0_PLC_TC3              = 851; // TwinCAT3 PLC task.
    }
}
