package org.lsst.ccs.drivers.ads.wrapper;

import com.sun.jna.Library;
import com.sun.jna.Structure;

/**
 *
 * @author tether
 */
public interface ADSLibrary extends Library {
    /** All the structures and unions in the original AdsDef.h header
     are compiled with #pragma pack(1).
     */
    int STRUCTURE_ALIGNMENT = Structure.ALIGN_NONE;

    /**
     * Add a new AMS route to a target system.
     * @param ams (In) AMS address of the target system.
     * @param ip (In) IP address of the target system.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsAddRoute(AmsNetId ams, String ip);

    /**
     * Delete an AMS route that had previously been added with AdsAddRoute().
     * @param ams (In) address of the target system
     */
    void AdsDelRoute(AmsNetId ams);

    /**
     * The connection (communication port) to the message router is
     * closed. The port to be closed must previously have been opened via
     * an AdsPortOpenEx() call.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsPortCloseEx(long port);

    /**
     * Establishes a connection (communication port) to the message
     * router. The port number returned by AdsPortOpenEx() is required as
     * parameter for further AdsLib function calls.
     * @return port number of a new Ads port or 0 if no more ports available
     */
    long AdsPortOpenEx();

    /**
     * Returns the local NetId and port number.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (Out) Reference to the structure of type AmsAddr.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsGetLocalAddressEx(long port, AmsAddr pAddr);

    /**
     * Change local NetId.
     * @param ams (In) local AmsNetId
     */
    void AdsSetLocalAddress(AmsNetId ams);

    /**
     * Reads data synchronously from an ADS server.
     * @param port (In)  port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param indexGroup (In) Index Group.
     * @param indexOffset (In) Index Offset.
     * @param bufferLength (In) Length of the data in bytes.
     * @param buffer (Out) Pointer to a data buffer that will receive the data.
     * @param bytesRead (Out) Pointer to a variable. If successful, this variable will return the number of actually read data bytes.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncReadReqEx2(long      port,
                           AmsAddr   pAddr,
                           int       indexGroup,
                           int       indexOffset,
                           int       bufferLength,
                           byte[]    buffer,
                           int[]     bytesRead);

    /**
     * Reads the identification and version number of an ADS server.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param devName (Out) Pointer to a character string of at least 16 bytes, that will receive the name of the ADS device.
     * @param version (Out) Address of a variable of type AdsVersion, which will receive the version number, revision number and the build number.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncReadDeviceInfoReqEx(
            long port,
            AmsAddr pAddr,
            String devName,
            AdsVersion version);

    /**
     * Reads the ADS status and the device status from an ADS server.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param adsState (Out) Address of a variable that will receive the ADS status (see data type [ADSSTATE](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/tcadsdll_enumadsstate.htm?id=17630)).
     * @param devState (Out) Address of a variable that will receive the device status.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncReadStateReqEx(long port, AmsAddr pAddr, short[] adsState, short[] devState);

    /**
     * Writes data synchronously into an ADS server and receives data back from the ADS server.
     * @param port  (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param indexGroup (In) Index Group.
     * @param indexOffset (In) Index Offset.
     * @param readLength (In) Length, in bytes, of the read buffer readData.
     * @param readData (Out) Buffer for data read from the ADS server.
     * @param writeLength (In) Length of the data, in bytes, send to the ADS server.
     * @param writeData (In) Buffer with data send to the ADS server.
     * @param bytesRead (Out) pointer to a variable. If successful, this variable will return the number of actually read data bytes.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncReadWriteReqEx2(long           port,
                                AmsAddr        pAddr,
                                int            indexGroup,
                                int            indexOffset,
                                int            readLength,
                                byte[]         readData,
                                int            writeLength,
                                byte[]         writeData,
                                int[]          bytesRead);

    /**
     * Writes data synchronously to an ADS server.
     * @param port  (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param indexGroup (In) Index Group.
     * @param indexOffset (In) Index Offset.
     * @param bufferLength (In) Length, in bytes, of the read buffer readData.
     * @param buffer (Out) Buffer for data read from the ADS server.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncWriteReqEx(long           port,
                           AmsAddr        pAddr,
                           int            indexGroup,
                           int            indexOffset,
                           int            bufferLength,
                           byte[]         buffer);

    /**
     * Changes the ADS status and the device status of an ADS server.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param adsState (In) New ADS status.
     * @param devState (In) New device status.
     * @param bufferLength (In) Length of the additional data, in bytes, send to the ADS server.
     * @param buffer (In) Buffer with additional data send to the ADS server.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncWriteControlReqEx(long           port,
                                  AmsAddr        pAddr,
                                  short          adsState,
                                  short          devState,
                                  int            bufferLength,
                                  byte[]         buffer);

    /**
     * A notification is defined within an ADS server (e.g. PLC). When a
     * certain event occurs a function (the callback function) is invoked in
     * the ADS client (C program).
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param indexGroup (In) Index Group.
     * @param indexOffset (In) Index Offset.
     * @param pAttrib (In) Pointer to the structure that contains further information.
     * @param pFunc (In) Pointer to the structure describing the callback function.
     * @param hUser (In) 32-bit value that is passed to the callback function.
     * @param pNotification (Out) Address of the variable that will receive the handle of the notification.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncAddDeviceNotificationReqEx(long                         port,
                                           AmsAddr                      pAddr,
                                           int                          indexGroup,
                                           int                          indexOffset,
                                           AdsNotificationAttrib        pAttrib,
                                           PAdsNotificationFuncEx       pFunc,
                                           int                          hUser,
                                           int[]                        pNotification);

    /**
     * A notification defined previously is deleted from an ADS server.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param pAddr (In) Structure with NetId and port number of the ADS server.
     * @param hNotification (In) The handle of the notification.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncDelDeviceNotificationReqEx(long port, AmsAddr pAddr, int hNotification);

    /**
     * Read the configured timeout for the ADS functions. The standard value is 5000 ms.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param timeout (Out) Buffer to store timeout value in ms.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncGetTimeoutEx(long port, int[] timeout);

    /**
     * Alters the timeout for the ADS functions. The standard value is 5000 ms.
     * @param port (In) port number of an Ads port that had previously been opened with AdsPortOpenEx().
     * @param timeout (In) Timeout in ms.
     * @return [ADS Return Code](http://infosys.beckhoff.de/content/1033/tc3_adsdll2/html/ads_returncodes.htm?id=17663)
     */
    long AdsSyncSetTimeoutEx(long port, int timeout);

    /**
     * Transmission modes for ADS notifications.
     */
    public static interface TransMode {
        
        /**
         * Send the notification whenever the cycle time specified
         * in the notification request has elapsed, regardless of whether the
         * variable being monitored has changed its value.
         */
        int    SERVERCYCLE = 3;
        
        /**
         * Send the notification whenever the cycle time specified
         * in the notification request has elapsed, but only if the
         * variable being monitored has changed its value.
         */
        int    SERVERONCHA = 4;
    }

    /**
     * Possible states for an ADS device.
     */
    public static interface DeviceState {
        int    INVALID = 0;
        int    IDLE = 1;
        int    RESET = 2;
        int    INIT = 3;
        int    START = 4;
        int    RUN = 5;
        int    STOP = 6;
        int    SAVECFG = 7;
        int    LOADCFG = 8;
        int    POWERFAILURE = 9;
        int    POWERGOOD = 10;
        int    ERROR = 11;
        int    SHUTDOWN = 12;
        int    SUSPEND = 13;
        int    RESUME = 14;
        int    CONFIG = 15;
        int    RECONFIG = 16;
        int    STOPPING = 17;
        int    INCOMPATIBLE = 18;
        int    EXCEPTION = 19;
    }

    /**
     * ADS error codes.
     */
    public static interface ErrorCode {
        int NOERR = (int)0;
        int ADS_ERROR_BASE = 1792;
        int DEVICE_LICENSEINVALID = (int)(39 + ADS_ERROR_BASE);
        int DEVICE_INVALIDOFFSET = (int)(3 + ADS_ERROR_BASE);
        int DEVICE_SYNTAX = (int)(13 + ADS_ERROR_BASE);
        int DEVICE_SYMBOLVERSIONINVALID = (int)(17 + ADS_ERROR_BASE);
        int DEVICE_LICENSEDUPLICATED = (int)(45 + ADS_ERROR_BASE);
        int DEVICE_SYMBOLNOTFOUND = (int)(16 + ADS_ERROR_BASE);
        int DEVICE_LICENSETIMETOLONG = (int)(43 + ADS_ERROR_BASE);
        int CLIENT_VARUSED = (int)(67 + ADS_ERROR_BASE);
        int DEVICE_NOTINIT = (int)(24 + ADS_ERROR_BASE);
        int DEVICE_INVALIDACCESS = (int)(4 + ADS_ERROR_BASE);
        int DEVICE_BUSY = (int)(8 + ADS_ERROR_BASE);
        int CLIENT_NOMORESYM = (int)(83 + ADS_ERROR_BASE);
        int CLIENT_REMOVEHASH = (int)(82 + ADS_ERROR_BASE);
        int DEVICE_INVALIDCONTEXT = (int)(9 + ADS_ERROR_BASE);
        int CLIENT_SYNCTIMEOUT = (int)(69 + ADS_ERROR_BASE);
        int DEVICE_NOTIFYHNDINVALID = (int)(20 + ADS_ERROR_BASE);
        int DEVICE_LICENSEEXCEEDED = (int)(38 + ADS_ERROR_BASE);
        int CLIENT_SYNCRESINVALID = (int)(84 + ADS_ERROR_BASE);
        int DEVICE_CERTIFICATEINVALID = (int)(47 + ADS_ERROR_BASE);
        int DEVICE_PENDING = (int)(30 + ADS_ERROR_BASE);
        int DEVICE_INVALIDOBJID = (int)(29 + ADS_ERROR_BASE);
        int DEVICE_NOMEMORY = (int)(10 + ADS_ERROR_BASE);
        int DEVICE_NOTREADY = (int)(7 + ADS_ERROR_BASE);
        int DEVICE_WARNING = (int)(32 + ADS_ERROR_BASE);
        int DEVICE_LICENSESYSTEMID = (int)(40 + ADS_ERROR_BASE);
        int DEVICE_ABORTED = (int)(31 + ADS_ERROR_BASE);
        int CLIENT_LISTEMPTY = (int)(66 + ADS_ERROR_BASE);
        int DEVICE_CLIENTUNKNOWN = (int)(21 + ADS_ERROR_BASE);
        int CLIENT_TIMEOUTINVALID = (int)(71 + ADS_ERROR_BASE);
        int DEVICE_INVALIDWATCHSIZE = (int)(23 + ADS_ERROR_BASE);
        int DEVICE_ERROR = (int)(0 + ADS_ERROR_BASE);
        int DEVICE_LICENSENOTIMELIMIT = (int)(41 + ADS_ERROR_BASE);
        int DEVICE_SIGNATUREINVALID = (int)(46 + ADS_ERROR_BASE);
        int DEVICE_INVALIDSIZE = (int)(5 + ADS_ERROR_BASE);
        int CLIENT_PORTNOTOPEN = (int)(72 + ADS_ERROR_BASE);
        int DEVICE_LICENSEFUTUREISSUE = (int)(42 + ADS_ERROR_BASE);
        int DEVICE_INVALIDPARM = (int)(11 + ADS_ERROR_BASE);
        int CLIENT_SYNCINTERNAL = (int)(80 + ADS_ERROR_BASE);
        int DEVICE_INVALIDCLSID = (int)(28 + ADS_ERROR_BASE);
        int DEVICE_NOINTERFACE = (int)(26 + ADS_ERROR_BASE);
        int DEVICE_LICENSEEXPIRED = (int)(37 + ADS_ERROR_BASE);
        int CLIENT_W32ERROR = (int)(70 + ADS_ERROR_BASE);
        int DEVICE_SYMBOLNOTACTIVE = (int)(34 + ADS_ERROR_BASE);
        int CLIENT_ADDHASH = (int)(81 + ADS_ERROR_BASE);
        int DEVICE_INCOMPATIBLE = (int)(14 + ADS_ERROR_BASE);
        int DEVICE_ACCESSDENIED = (int)(35 + ADS_ERROR_BASE);
        int DEVICE_INVALIDDATA = (int)(6 + ADS_ERROR_BASE);
        int DEVICE_INVALIDINTERFACE = (int)(27 + ADS_ERROR_BASE);
        int DEVICE_EXISTS = (int)(15 + ADS_ERROR_BASE);
        int DEVICE_INVALIDSTATE = (int)(18 + ADS_ERROR_BASE);
        int DEVICE_LICENSENOTFOUND = (int)(36 + ADS_ERROR_BASE);
        int DEVICE_INVALIDARRAYIDX = (int)(33 + ADS_ERROR_BASE);
        int DEVICE_NOTFOUND = (int)(12 + ADS_ERROR_BASE);
        int DEVICE_INVALIDGRP = (int)(2 + ADS_ERROR_BASE);
        int DEVICE_TRANSMODENOTSUPP = (int)(19 + ADS_ERROR_BASE);
        int CLIENT_INVALIDPARM = (int)(65 + ADS_ERROR_BASE);
        int CLIENT_NOAMSADDR = (int)(73 + ADS_ERROR_BASE);
        int DEVICE_NOMOREHDLS = (int)(22 + ADS_ERROR_BASE);
        int DEVICE_SRVNOTSUPP = (int)(1 + ADS_ERROR_BASE);
        int CLIENT_DUPLINVOKEID = (int)(68 + ADS_ERROR_BASE);
        int DEVICE_TIMEOUT = (int)(25 + ADS_ERROR_BASE);
        int DEVICE_EXCEPTION = (int)(44 + ADS_ERROR_BASE);
        int CLIENT_SYNCPORTLOCKED = (int)(85 + ADS_ERROR_BASE);
        int CLIENT_ERROR = (int)(64 + ADS_ERROR_BASE);
    }

    /**
     * The index group numbers meaningful to ADS.
     */
    public static interface IndexGroup {
        /** Area of variables mapped with %M in PLC code. */
        int PLC_MEMORY_AREA = 0x4020;
        
        int SYM_INFOBYNAME = (int)61447;
        int SYM_DOWNLOAD = (int)61450;
        int SYMTAB = (int)61440;
        int DEVICE_DATA = (int)61696;
        int SYM_RELEASEHND = (int)61446;
        int SYM_UPLOADINFO = (int)61452;
        int IOIMAGE_ROSIZE = (int)61493;
        int SYMNAME = (int)61441;
        int IOIMAGE_RWOB = (int)61488;
        int SYM_UPLOAD = (int)61451;
        int IOIMAGE_RWIOB = (int)61536;
        int SYM_UPLOADINFO2 = (int)61455;
        int SYM_DOWNLOAD2 = (int)61453;
        int IOIMAGE_CLEARI = (int)61504;
        int IOIMAGE_RWIB = (int)61472;
        int IOIMAGE_CLEARO = (int)61520;
        int SUMUP_READEX2 = (int)61572;
        int SUMUP_DELDEVNOTE = (int)61574;
        int SUMUP_READ = (int)61568;
        int SYMVAL = (int)61442;
        int SYM_VALBYHND = (int)61445;
        int SUMUP_WRITE = (int)61569;
        int IOIMAGE_RWOX = (int)61489;
        int SUMUP_READEX = (int)61571;
        int SYM_VALBYNAME = (int)61444;
        int SYM_INFOBYNAMEEX = (int)61449;
        int SYM_VERSION = (int)61448;
        int SYMNOTE = (int)61456;
        int SYM_DT_UPLOAD = (int)61454;
        int SUMUP_ADDDEVNOTE = (int)61573;
        int SYM_HNDBYNAME = (int)61443;
        int SUMUP_READWRITE = (int)61570;
        int IOIMAGE_RISIZE = (int)61477;
        int IOIMAGE_RWIX = (int)61473;
    }

    public static interface AmsPort {
        short LOGGER                  = 100;
        short R0_RTIME                = 200;
        short R0_TRACE                = (R0_RTIME + 90);
        short R0_IO                   = 300;
        short R0_SPS                  = 400;
        short R0_NC                   = 500;
        short R0_ISG                  = 550;
        short R0_PCS                  = 600;
        short R0_PLC                  = 801;
        short R0_PLC_RTS1             = 801; // TwinCAT2 PLC task.
        short R0_PLC_RTS2             = 811;
        short R0_PLC_RTS3             = 821;
        short R0_PLC_RTS4             = 831;
        short R0_PLC_TC3              = 851; // TwinCAT3 PLC task.
    }
}
