package org.lsst.ccs.drivers.ads.wrapper;

import com.sun.jna.Structure;
import java.util.Arrays;
import java.util.List;


/**
 * This structure contains all the attributes for the definition of a
 * notification request.
 * A TwinCAT ADS server maintains a FIFO into which notifications are written.
 * The notifications are actually sent if the FIFO fills up or it's been
 * too long since the last time the FIFO was emptied, whichever
 * come first.
<p>
* Warning: Too many notifications  can load the system so heavily that
 * the user interface becomes much slower.
 * <p>
 * Tip: Set the cycle time to the most appropriate values, and always
 * close connections when they are no longer required.
 */
public class AdsNotificationAttrib extends Structure {

    public int cbLength;
    public int nTransMode;
    public int nMaxDelay;
    public int nCycleTime;
    
    /**
     * @param cbLength Length in bytes of the data that is to be transmitted
     * from the server and passed to the client's callback function.
     * @param nTransMode See {@link ADSLibrary.TransMode}
     * @param nMaxDelay The maximum length of time that a notification
     * may spend in the server's FIFO before being sent to the client.
     * Normally the server will wait until the FIFO is full. The time unit
     * is 100 nanosec.
     * @param nCycleTime Specifies the interval in 100 ns units between the times
     * when the ADS server decides whether or not to send a new notification.
     * The smallest possible value is the cycle
     * time of the ADS server; for the PLC, this is the task cycle time.
     * This is also what you get when you use a value of zero.
     * */
    public AdsNotificationAttrib(int cbLength, int nTransMode, int nMaxDelay, int nCycleTime) {
        super(ALIGN_NONE);  // pack(1) pragma was used in AdsDef.h
        this.cbLength = cbLength;
        this.nTransMode = nTransMode;
        this.nMaxDelay = nMaxDelay;
        this.nCycleTime = nCycleTime;
    }

    /** For use by JNA. */
    public AdsNotificationAttrib() {
        super(ALIGN_NONE);
        this.cbLength = 0;
        this.nTransMode = 0;
        this.nMaxDelay = 0;
        this.nCycleTime = 0;
    }

    @Override
    protected List<String> getFieldOrder() {
        return Arrays.asList("cbLength", "nTransMode", "nMaxDelay", "nCycleTime");
    }
}
