package org.lsst.ccs.drivers.ad;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/*.
 *  Program to test the AD7794 chip.
 *
 *  @author Owen Saxton
 */
public class TestAd7794 {

    /*
     *  Enumerations and generated inverse maps
     */
    public enum ChanNames {

        AIN1(Ad7794Eval.CHAN_AIN1),
        AIN2(Ad7794Eval.CHAN_AIN2),
        AIN3(Ad7794Eval.CHAN_AIN3),
        AIN4(Ad7794Eval.CHAN_AIN4),
        AIN5(Ad7794Eval.CHAN_AIN5),
        AIN6(Ad7794Eval.CHAN_AIN6),
        TEMP(Ad7794Eval.CHAN_TEMP),
        VDD(Ad7794Eval.CHAN_VDD);

        private final int value;

        ChanNames(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }

    }

    private final static Map<Integer, String> chanValues = new HashMap<>();
    static {
        for (ChanNames en : ChanNames.values()) {
            chanValues.put(en.getValue(), en.name().toLowerCase());
        }
    }

    public enum CnvModeNames {

        CONTINUOUS(Ad7794Eval.MODE_CONT),
        SINGLE(Ad7794Eval.MODE_SINGLE),
        IDLE(Ad7794Eval.MODE_IDLE),
        POWERDOWN(Ad7794Eval.MODE_PWR_DOWN),
        INTOFFCAL(Ad7794Eval.MODE_INT_Z_CAL),
        INTGAINCAL(Ad7794Eval.MODE_INT_F_CAL),
        EXTOFFCAL(Ad7794Eval.MODE_SYS_Z_CAL),
        EXTGAINCAL(Ad7794Eval.MODE_SYS_F_CAL);

        private final int value;

        CnvModeNames(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }

    }

    private static final Map<Integer, String> cnvModeValues = new HashMap<>();
    static {
        for (CnvModeNames en : CnvModeNames.values()) {
            cnvModeValues.put(en.getValue(), en.name().toLowerCase());
        }
    }

    public enum RateNames {

        R470(Ad7794Eval.FRS_RATE_470),
        R242(Ad7794Eval.FRS_RATE_242),
        R123(Ad7794Eval.FRS_RATE_123),
        R62(Ad7794Eval.FRS_RATE_62),
        R50(Ad7794Eval.FRS_RATE_50),
        R39(Ad7794Eval.FRS_RATE_39),
        R33(Ad7794Eval.FRS_RATE_33),
        R19(Ad7794Eval.FRS_RATE_19),
        R17(Ad7794Eval.FRS_RATE_17),
        R16(Ad7794Eval.FRS_RATE_16),
        R12(Ad7794Eval.FRS_RATE_12),
        R10(Ad7794Eval.FRS_RATE_10),
        R8(Ad7794Eval.FRS_RATE_8),
        R6(Ad7794Eval.FRS_RATE_6),
        R4(Ad7794Eval.FRS_RATE_4);

        private final int value;

        RateNames(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }

    }

    private static final Map<Integer, String> rateValues = new HashMap<>();
    static {
        for (RateNames en : RateNames.values()) {
            rateValues.put(en.getValue(), en.name().toLowerCase());
        }
    }

    public enum ClockNames {

        INTERNAL(Ad7794Eval.CLOCK_INT),
        INTAVAIL(Ad7794Eval.CLOCK_INT_AVL),
        EXTERNAL(Ad7794Eval.CLOCK_EXT),
        EXTHALF(Ad7794Eval.CLOCK_EXT_HALF);

        private final int value;

        ClockNames(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }

    }

    private static final Map<Integer, String> clockValues = new HashMap<>();
    static {
        for (ClockNames en : ClockNames.values()) {
            clockValues.put(en.getValue(), en.name().toLowerCase());
        }
    }

    public enum GainNames {

        X1(Ad7794Eval.GAIN_1),
        X2(Ad7794Eval.GAIN_2),
        X4(Ad7794Eval.GAIN_4),
        X8(Ad7794Eval.GAIN_8),
        X16(Ad7794Eval.GAIN_16),
        X32(Ad7794Eval.GAIN_32),
        X64(Ad7794Eval.GAIN_64),
        X128(Ad7794Eval.GAIN_128);

        private final int value;

        GainNames(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }

    }

    private static final Map<Integer, String> gainValues = new HashMap<>();
    static {
        for (GainNames en : GainNames.values()) {
            gainValues.put(en.getValue(), en.name().toLowerCase());
        }
    }

    public enum RefsNames {

        EXT1(Ad7794Eval.REFSEL_EXT1),
        EXT2(Ad7794Eval.REFSEL_EXT2),
        INTERNAL(Ad7794Eval.REFSEL_INT);

        private final int value;

        RefsNames(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }

    }

    private static final Map<Integer, String> refsValues = new HashMap<>();
    static {
        for (RefsNames en : RefsNames.values()) {
            refsValues.put(en.getValue(), en.name().toLowerCase());
        }
    }

    public enum CalibNames {

        INTOFF(Ad7794Eval.CAL_INT_ZERO),
        INTGAIN(Ad7794Eval.CAL_INT_FULL),
        EXTOFF(Ad7794Eval.CAL_SYS_ZERO),
        EXTGAIN(Ad7794Eval.CAL_SYS_FULL);

        private final int value;

        CalibNames(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }

    }

    private static final Map<Integer, String> calibValues = new HashMap<>();
    static {
        for (CalibNames en : CalibNames.values()) {
            calibValues.put(en.getValue(), en.name().toLowerCase());
        }
    }

    public enum OptionNames {

        CONFIG(Ad7794Eval.OPTN_CONFIG),
        SINGLE(Ad7794Eval.OPTN_SINGLE),
        IMMED(Ad7794Eval.OPTN_IMMED);

        private final int value;

        OptionNames(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }

    }

    public enum OnOff { OFF, ON }

    /*
     *  Private fields
     */
    private final Ad7794Eval ad;


    /*.
     *  Constructor.
     */
    public TestAd7794() throws DriverException
    {
        ad = new Ad7794Eval();
    }


    /*.
     *  Connects to a board
     *
     *  @param  index  The board index
     */
    @Command(description="Connect to a board")
    public void connect(@Argument(description = "Board index")
                        int index) throws DriverException
    {
        ad.connect(index);
    }

    
    /*.
     *  Initializes the chip.
     */
    @Command(name="initialize", description="Initialize the chip")
    public void initialize() throws DriverException
    {
        ad.setupStandard();
    }


    /*.
     *  Shows the status register.
     */
    @Command(name="showstatus", description="Show the status register")
    public String showStatus() throws DriverException
    {
        int status = ad.getStatus();
        return String.format("Status: %sready, %serror, %sextref, %s, chan = %s",
                             (status & Ad7794Eval.STS_READY) == 0 ? "" : "not ",
                             (status & Ad7794Eval.STS_ERROR) != 0 ? "" : "no ",
                             (status & Ad7794Eval.STS_NOXREF) == 0 ? "" : "no ",
                             (status & Ad7794Eval.STS_AD7794) != 0 ? "AD7794" : "AD7795",
                             chanValues.get(status & Ad7794Eval.STS_CHANNEL_M));
    }


    /*.
     *  Sets the channel.
     */
    @Command(name="setchannel", description="Set the channel")
    public void setChan(@Argument(name="chan", description="Channel name")
                        ChanNames chan) throws DriverException
    {
        ad.setChannel(chan.getValue());
    }


    /*.
     *  Shows the channel.
     */
    @Command(name="showchannel", description="Show the channel")
    public String showChan() throws DriverException
    {
        int chan = ad.getChannel();
        return String.format("Channel = %s (%s)", chan, chanValues.get(chan));
    }


    /*.
     *  Sets the conversion mode.
     */
    @Command(name="setconversion", description="Set the conversion mode")
    public void setConv(@Argument(name="mode", description="Conversion mode")
                        CnvModeNames mode) throws DriverException
    {
        ad.setConvMode(mode.getValue());
    }


    /*.
     *  Shows the conversion mode.
     */
    @Command(name="showconversion", description="Show the conversion mode")
    public String showConv() throws DriverException
    {
        int mode = ad.getConvMode();
        return String.format("Conversion mode = %s (%s)", mode,
                             cnvModeValues.get(mode));
    }


    /*.
     *  Sets the conversion rate.
     */
    @Command(name="setrate", description="Set the conversion rate")
    public void setRate(@Argument(name="rate", description="Conversion rate")
                        RateNames rate) throws DriverException
    {
        ad.setConvRate(rate.getValue());
    }


    /*.
     *  Shows the conversion rate.
     */
    @Command(name="showrate", description="Show the conversion rate")
    public String showRate() throws DriverException
    {
        int rate = ad.getConvRate();
        return String.format("Conversion rate = %s (%s)", rate,
                             rateValues.get(rate).substring(1) + " Hz");
    }


    /*.
     *  Sets the clock source.
     */
    @Command(name="setclock", description="Set the clock source")
    public void setClock(@Argument(name="clock", description="Clock source")
                         ClockNames clock) throws DriverException
    {
        ad.setClockSource(clock.getValue());
    }


    /*.
     *  Shows the clock source.
     */
    @Command(name="showclock", description="Show the clock source")
    public String showClock() throws DriverException
    {
        int clock = ad.getClockSource();
        return String.format("Clock source = %s (%s)", clock,
                             clockValues.get(clock));
    }


    /*.
     *  Sets the gain.
     */
    @Command(name="setgain", description="Set the gain")
    public void setGain(@Argument(name="gain", description="The gain name")
                        GainNames gain) throws DriverException
    {
        ad.setGain(gain.getValue());
    }


    /*.
     *  Shows the gain.
     */
    @Command(name="showgain", description="Show the gain")
    public String showGain() throws DriverException
    {
        int gain = ad.getGain();
        return String.format("ADC gain = %s (%sx)", gain,
                             gainValues.get(gain).substring(1));
    }


    /*.
     *  Sets the reference source.
     */
    @Command(name="setreference", description="Set the reference source")
    public void setReference(@Argument(name="source", description="Reference source")
                             RefsNames source) throws DriverException
    {
        ad.setRefSelect(source.getValue());
    }


    /*.
     *  Shows the reference source.
     */
    @Command(name="showreference", description="Show the reference source")
    public String showRefsel() throws DriverException
    {
        int refs = ad.getRefSelect();
        return String.format("Reference source = %s (%s)", refs,
                             refsValues.get(refs));
    }


    /*.
     *  Sets the reference detection.
     */
    @Command(name="setrefdet", description="Set reference detection")
    public void setRefdet(@Argument(name="state", description="State to set")
                          OnOff det) throws DriverException
    {
        ad.setRefDetect(det == OnOff.ON);
    }


    /*.
     *  Shows the reference detection state.
     */
    @Command(name="showrefdet", description="Show reference detection state")
    public String showRefdet() throws DriverException
    {
        return String.format("Reference detection is %s",
                             onOff(ad.isRefDetect()));
    }


    /*.
     *  Sets buffered mode state.
     */
    @Command(name="setbuffered", description="Set buffered mode state")
    public void setBuff(@Argument(name="state", description="State to set")
                        OnOff on) throws DriverException
    {
        ad.setBuffered(on == OnOff.ON);
    }


    /*.
     *  Shows buffered mode state
     */
    @Command(name="showbuffered", description="Show buffered mode state")
    public String showBuff() throws DriverException
    {
        return String.format("Buffered mode is %s", onOff(ad.isBuffered()));
    }


    /*.
     *  Sets unipolar state.
     */
    @Command(name="setunipolar", description="Set unipolar state")
    public void setUnipol(@Argument(name="state", description="State to set")
                          OnOff on) throws DriverException
    {
        ad.setUnipolar(on == OnOff.ON);
    }


    /*.
     *  Shows unipolar state.
     */
    @Command(name="showunipolar", description="Show unipolar state")
    public String showUnipol() throws DriverException
    {
        return String.format("Unipolar mode is %s", onOff(ad.isUnipolar()));
    }


    /*.
     *  Sets the power switch.
     */
    @Command(name="setswitch", description="Set the power switch")
    public void setSwitch(@Argument(name="state", description="Switch state")
                          OnOff on) throws DriverException
    {
        ad.setPowerSwitch(on == OnOff.ON);
    }


    /*.
     *  Shows the power switch.
     */
    @Command(name="showswitch", description="Show the power switch")
    public String showSwitch() throws DriverException
    {
        return String.format("Power switch is %s", onOff(ad.isPowerSwitch()));
    }


    /*.
     *  Reads an ADC.
     */
    @Command(name="readadc", description="Read an ADC")
    public String readAdc(@Argument(name="chan", description="Channel number")
                          int chan,
                          @Argument(name="count", description="Sample count")
                          int count,
                          @Argument(name="optns", description="Option names")
                          OptionNames... optns) throws DriverException
    {
        int option = getOption(optns);
        double sum = 0, sumsq = 0;
        for (int j = 0; j < count; j++) {
            double value = ad.readAdc(chan, option);
            sum += value;
            sumsq += value * value;
        }
        return showStats("Raw " + chanValues.get(chan), count, sum, sumsq);
    }


    /*.
     *  Reads the chip temperature.
     */
    @Command(name="readtemp", description="Read the chip temperature")
    public String readTemp(@Argument(name="count", description="Sample count")
                           int count,
                           @Argument(name="optns", description="Option names")
                           OptionNames... optns) throws DriverException
    {
        int option = getOption(optns);
        double sum = 0, sumsq = 0;
        for (int j = 0; j < count; j++) {
            double value = ad.readTemperature(option);
            sum += value;
            sumsq += value * value;
        }
        return showStats("Temperature", count, sum, sumsq);
    }


    /*.
     *  Reads the chip VDD.
     */
    @Command(name="readvdd", description="Read the chip VDD")
    public String ReadVdd(@Argument(name="count", description="Sample count")
                          int count,
                          @Argument(name="optns", description="Option names")
                          OptionNames... optns) throws DriverException
    {
        int option = getOption(optns);
        double sum = 0, sumsq = 0;
        for (int j = 0; j < count; j++) {
            double value = ad.readVdd(option);
            sum += value;
            sumsq += value * value;
        }
        return showStats("Voltage", count, sum, sumsq);
    }


    /*.
     *  Reads the on-board thermistor.
     */
    @Command(name="readtherm", description="Read the thermistor")
    public String readTherm(@Argument(name="count", description="Sample count")
                            int count,
                            @Argument(name="optns", description="Option names")
                            OptionNames... optns) throws DriverException
    {
        int option = getOption(optns);
        ad.setupThermistor();
        double sum = 0, sumsq = 0;
        for (int j = 0; j < count; j++) {
            double value = ad.readThermistor(option);
            sum += value;
            sumsq += value * value;
        }
        ad.cleanupThermistor();
        return showStats("Thermistor", count, sum, sumsq);
    }


    /*.
     *  Performs a calibration.
     */
    @Command(name="calibrate", description="Perform a calibration")
    public String calibrate(@Argument(name="chan", description="Channel number")
                            int chan,
                            @Argument(name="type", description="Calibration type")
                            int type) throws DriverException
    {
        int value = ad.calibrate(chan, type);
        return String.format("New %s value = %s",
                             (type & 1) == 0 ? "offset" : "scale", value);
    }


    /*.
     *  Shows calibration values.
     */
    @Command(name="showcalib", description="Show calibration values")
    public String showCalib(@Argument(name="chan", description="Channel number")
                            int chan) throws DriverException
    {
        int[] value = ad.getCalibration(chan);
        return String.format("Offset = %s, scale = %s", value[0], value[1]);
    }


    /*.
     *  Shows all the register values.
     */
    private static final String[] regName = {"STAT", "MODE", "CONF", "DATA",
                                             " ID ", " IO ", "OFFS", "FSCL"};
    private final static String[]
        regDesc0 = {"| ~RDY |  ERR |NOXREF|  --  | 7794 | CHAN2| CHAN1| CHAN0|",
                    "| MOD2 | MOD1 | MOD0 |  PSW |  --  |  --  |AMPCOM|  --  |",
                    "|VBIAS1|VBIAS0|BRNOUT|UNIPOL| BOOST| GAIN2| GAIN1| GAIN0|",
                    "",
                    "",
                    "|  --  |DIOENA|IODAT2|IODAT1|CSDIR1|CSDIR0|CSENA1|CSENA0|",
                    "",
                    ""},
        regDesc1 = {"",
                    "| CLK1 | CLK0 |  --  |CHPDIS| FRS3 | FRS2 | FRS1 | FRS0 |",
                    "|RFSEL1|RFSEL0|REFDET|BUFFRD| CHAN3| CHAN2| CHAN1| CHAN0|",
                    "",
                    "",
                    "",
                    "",
                    ""};
    
    @Command(name="showregs", description="Show all register values")
    public String showRegs() throws DriverException
    {
        StringBuilder st = new StringBuilder();
        int[] regSize = ad.getRegSizes();
        for (int j = 0; j < 8; j++) {
            int value = ad.readRegister(j);
            int size = 2 * regSize[j];
            String fmt = String.format("%%d (%%s): %%s%%0%sx  %%s", size);
            if (j > 0) {
                st.append('\n');
            }
            st.append(String.format(fmt, j, regName[j],
                                    "      ".substring(size), value,
                                    regDesc0[j]));
            if (!regDesc1[j].isEmpty()) {
                st.append(String.format("\n                  %s", regDesc1[j]));
            }
        }
        return st.toString();
    }


    /*.
     *  Reads a register.
     */
    @Command(name="readregister", description="Read a register")
    public String readReg(@Argument(name="regnum", description="Register number")
                          int regnum) throws DriverException
    {
        regnum &= 0x07;
        String fmt = String.format("Register %%s: %%0%sx",
                                   2 * ad.getRegSizes()[regnum]);
        return String.format(fmt, regnum, ad.readRegister(regnum));
    }


    /*.
     *  Writes a register.
     */
    @Command(name="writeregister", description="Write a register")
    public void writeReg(@Argument(name="regnum", description="Register number")
                         int regnum,
                         @Argument(name="value", description="Value to write")
                         int value) throws DriverException
    {
        ad.writeRegister(regnum, value);
    }


    /*.
     *  Sets the reset state.
     */
    @Command(name="reset", description="Set the reset state")
    public void reset(@Argument(name="state", description="Reset state")
                      OnOff on) throws DriverException
    {
        ad.setReset(on == OnOff.ON);
    }


    /*.
     *  Reads memory
     */
    @Command(name="readmemory", description="Read memory")
    private String readMemory(@Argument(name="addr", description="Memory address")
                              int addr,
                              @Argument(name="count", description="Number of bytes")
                              int count,
                              @Argument(name="timeout", description="Timeout (ms)")
                              int timeout) throws DriverException
    {
        byte[] data = new byte[count];
        return dispData(addr, data, ad.readMemory(addr, data, timeout));
    }


    /*.
     *  Writes memory
     */
    @Command(name="writememory", description="Write memory")
    private void writeMemory(@Argument(name="addr", description="Memory address")
                             int addr,
                             @Argument(name="data", description="Date to write")
                             byte... data) throws DriverException
    {
        ad.writeMemory(addr, data);
    }


    /*.
     *  Loads memory from a file.
     */
    @Command(name="load", description="Load memory from a file")
    public void load(@Argument(name="file", description="File name")
                     String fileName,
                     @Argument(name="force",
                               description="Whether to force the load")
                     boolean force) throws DriverException
    {
        ad.load(fileName, force);
    }


    /*.
     *  Sets debug state.
     */
    @Command(name="setdebug", description="Set debug state on or off")
    public void setDelay(@Argument(name="state", description="Debug state")
                         OnOff state) throws DriverException
    {
        ad.setDebug(state == OnOff.ON);
    }


    /*.
     *  Forms options word.
     */
    private int getOption(OptionNames[] optns)
    {
        int option = 0;
        for (OptionNames optn : optns) {
            option |= optn.getValue();
        }
        return option;
    }


    /*.
     *  Displays mean and sigma values.
     */
    private static String showStats(String name, int count, double sum,
                                    double sumsq)
    {
        double mean = sum / count;
        double sigsq = sumsq / count - mean * mean;
        double sigma = sigsq <= 0 ? 0 : Math.sqrt(sigsq);
        return String.format("%s: mean = %.7g, sigma = %.7g", name, mean, sigma);
    }


    /*.
     *  Displays memory data
     */
    private static String dispData(int addr, byte[] data, int count)
    {
        StringBuilder sb = new StringBuilder();
        for (int j = 0; j < count; j++) {
            if ((j & 0x0f) == 0) {
                if (j != 0) {
                    sb.append('\n');
                }
                sb.append(String.format("%04x:", addr + j));
            }
            sb.append(String.format(" %02x", data[j] & 0xff));
        }
        return sb.toString();
    }


    /*.
     *  Generates on/off text from boolean.
     */
    private static String onOff(boolean on)
    {
        return on ? "on" : "off";
    }

}
