package org.lsst.ccs.drivers.ad;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines to access a SPI device on an Analog Devices evaluation board
 *
 *  @author Owen Saxton
 */
public abstract class SpiBus extends Cypress {

    /**
     *  Private constants
     */
    private static final int
        READ_TIMEOUT      = 1000,
        RQST_SPI          = 0xda;

    private static final String
        STD_LOAD_FILE     = "/Icv_spi.hex";

    /**
     *  Private fields
     */
    private int
        rdyTimeout = 0;


    /**
     *  Performs any needed initialization
     */
    void initialize() throws DriverException
    {
        load(false);
    }


    /**
     *  Generates communications register value
     *
     *  @param  regnum  The register number.
     *  @param  read    True for a read operation, false for a write.
     *  @return  The (8-bit) value to be put into the communications register.
     */
    public abstract int commReg(int regnum, boolean read);


    /**
     *  Sets the RDY timeout period
     *
     *  @param  period  The timeout period in units of ~82 microseconds, up
     *                  to a maximum of 65535.
     */
    public void setRdyTimeout(int period)
    {
        rdyTimeout = period;
    }


    /**
     *  Gets the RDY timeout period
     *
     *  @return  The timeout period in units of ~82 microseconds.
     */
    public int getRdyTimeout()
    {
        return rdyTimeout;
    }


    /**
     *  Reads a register
     *
     *  @param  regnum   The register to read
     *  @param  size     The register size (bytes)
     *  @return  The value read
     *  @throws  DriverException
     */
    public int readRegister(int regnum, int size) throws DriverException
    {
        byte[] data = new byte[size + 1];
        usb.controlRead(RTYP_VENDOR_READ, RQST_SPI, commReg(regnum, true), rdyTimeout,
                        data, 0, data.length, READ_TIMEOUT);
        int value = 0;
        for (int j = 1; j <= size; j++) {
            value = (value << 8) | (data[j] & 0xff);
        }
        return value;
    }


    /**
     *  Writes a register
     *
     *  @param  regnum  The register to write
     *  @param  size    The register size (bytes)
     *  @param  value   The value to write
     *  @throws  DriverException
     */
    public void writeRegister(int regnum, int size, int value) throws DriverException
    {
        byte data[] = new byte[size + 1];
        data[0] = (byte)commReg(regnum, false);
        for (int j = 0; j < size; j++) {
            data[size - j] = (byte)value;
            value >>= 8;
        }
        usb.controlWrite(RTYP_VENDOR_WRITE, RQST_SPI, 0, rdyTimeout, data, 0, data.length, 0);
    }


    /**
     *  Updates a register
     *
     *  @param  regnum  The register to update
     *  @param  size    The register size (bytes)
     *  @param  mask    The mask of bits to update
     *  @param  value   The value to write, under the mask
     *  @throws  DriverException
     */
    public void updateRegister(int regnum, int size, int mask, int value) throws DriverException
    {
        int data = readRegister(regnum, size);
        writeRegister(regnum, size, (data & ~mask) | (value & mask));
    }


    /**
     *  Loads memory from standard file
     *
     *  @param  force  If true, force the load even if memory appears to be
     *                 already loaded correctly.
     *  @throws  DriverException
     */
    public void load(boolean force) throws DriverException
    {
        load(STD_LOAD_FILE, force);
    }

}
