package org.lsst.ccs.drivers.ad;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines to access an I2C device on an Analog Devices evaluation board
 *
 *  @author Owen Saxton
 */
public class I2cBus extends Cypress {

    /**
     *  Private constants
     */
    private static final int
        READ_TIMEOUT      = 1000,
        RQST_I2C          = 0xdd,
        ADDR_I2C_READ     = 0x91,
        ADDR_I2C_WRITE    = 0x90;

    private static final String
        STD_LOAD_FILE     = "/Icv_usb.hex";


    /**
     *  Performs any needed initialization
     */
    void initialize() throws DriverException
    {
        load(false);
    }


    /**
     *  Reads 16-bit value from a pair of registers
     *
     *  @param  regnum  The number of the first (high-order) register to read
     *  @return  The 16-bit raw value
     *  @throws  DriverException
     */
    public int readRegister2(int regnum) throws DriverException
    {
        byte[] data = new byte[2];
        readRegister(regnum, data);

        return (data[1] & 0xff) | ((data[0] & 0xff) << 8);
    }


    /**
     *  Reads 24-bit value from a triplet of registers
     *
     *  @param  regnum  The number of the first (high-order) register to read
     *  @return  The 24-bit raw value
     *  @throws  DriverException
     */
    public int readRegister3(int regnum) throws DriverException
    {
        byte[] data = new byte[3];
        readRegister(regnum, data);

        return (data[2] & 0xff) | ((data[1] & 0xff) << 8)
                 | ((data[0] & 0xff) << 16);
    }


    /**
     *  Reads byte array from consecutive registers
     *
     *  @param  regnum   The first register to read
     *  @param  data     The array of bytes to read into
     *  @return  The number of bytes read.
     *  @throws  DriverException
     */
    public int readRegister(int regnum, byte[] data) throws DriverException
    {
        return usb.controlRead(RTYP_VENDOR_READ, RQST_I2C, ADDR_I2C_READ, regnum, data, 0, data.length, READ_TIMEOUT);
    }


    /**
     *  Reads a register
     *
     *  @param  regnum   The register to read
     *  @return  The value read
     *  @throws  DriverException
     */
    public int readRegister(int regnum) throws DriverException
    {
        byte[] data = new byte[1];
        readRegister(regnum, data);
        return data[0] & 0xff;
    }


    /**
     *  Writes byte array to consecutive registers
     *
     *  @param  regnum  The first register to write
     *  @param  data    The array of bytes to write
     *  @return  The number of bytes written.
     *  @throws  DriverException
     */
    public int writeRegister(int regnum, byte[] data) throws DriverException
    {
        return usb.controlWrite(RTYP_VENDOR_WRITE, RQST_I2C, ADDR_I2C_WRITE, regnum, data, 0, data.length, 0);
    }


    /**
     *  Writes a register
     *
     *  @param  regnum  The register to write
     *  @param  value   The value to write
     *  @throws  DriverException
     */
    public void writeRegister(int regnum, int value) throws DriverException
    {
        writeRegister(regnum, new byte[]{(byte)value});
    }


    /**
     *  Updates a register
     *
     *  @param  regnum  The register to update
     *  @param  mask    The mask of bits to update
     *  @param  value   The value to write, under the mask
     *  @throws  DriverException
     */
    public void updateRegister(int regnum, int mask, int value) throws DriverException
    {
        int data = readRegister(regnum);
        writeRegister(regnum, (data & ~mask) | (value & mask));
    }


    /**
     *  Loads memory from standard file
     *
     *  @param  force  If true, force the load even if memory appears to be
     *                 already loaded correctly.
     *  @throws  DriverException
     */
    public void load(boolean force) throws DriverException
    {
        load(STD_LOAD_FILE, force);
    }

}
