package org.lsst.ccs.drivers.ad;

import java.io.FileNotFoundException;
import java.io.InputStream;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.usb.UsbDevice;

/**
 *  Routines to access Cypress EZUSB chip functions on an Analog Devices evaluation board
 *
 *  @author Owen Saxton
 */
public class Cypress {

    /**
     *  Package constants
     */
    static final int
        RTYP_VENDOR_READ  = 0xc0,
        RTYP_VENDOR_WRITE = 0x40,
        RQST_CYPRESS      = 0xa0,
        RESET_ADDR        = 0xe600,
        CACHE_SIZE        = 4096,
        PACKET_SIZE       = 256;

    static final byte
        BOL_MARKER        = 0x3a;    // Colon

    /**
     *  Package fields
     */
    UsbDevice usb;

    /**
     *  Private fields
     */
    private String loadFile;
    private final int[] loadMap = new int[CACHE_SIZE / 32 + 1];
    private final byte[] loadImage = new byte[CACHE_SIZE + 1];


    /**
     *  Sets the USB device for subsequent operations.
     *
     *  @param  usb  The USB device to set
     */
    public void setUsbDevice(UsbDevice usb)
    {
        this.usb = usb;
    }


    /**
     *  Sets or clears the RESET state
     *
     *  @param  on  If true, sets RESET, otherwise clears RESET
     *  @throws  DriverException
     */
    public void setReset(boolean on) throws DriverException
    {
        byte data[] = {(byte)(on ? 1 : 0)};
        usb.controlWrite(RTYP_VENDOR_WRITE, RQST_CYPRESS, RESET_ADDR, 0, data, 0, data.length, 0);
    }


    /**
     *  Reads byte array from memory
     *
     *  @param  addr     The first address to read
     *  @param  data     The array of bytes to read into
     *  @param  timeout  The maximum time to wait (ms), or 0 for no timeout
     *  @return  The number of bytes read.
     *  @throws  DriverException
     */
    public int readMemory(int addr, byte[] data, int timeout) throws DriverException
    {
        return usb.controlRead(RTYP_VENDOR_READ, RQST_CYPRESS, addr, 0, data, 0, data.length, timeout);
    }


    /**
     *  Reads byte array from memory
     *
     *  @param  addr     The first address to read
     *  @param  data     The array of bytes to read into
     *  @param  offs     The offset in data to the first byte to read into
     *  @param  leng     The number of bytes to read
     *  @param  timeout  The maximum time to wait (ms), or 0 for no timeout
     *  @return  The number of bytes read.
     *  @throws  DriverException
     */
    public int readMemory(int addr, byte[] data, int offs, int leng, int timeout) throws DriverException
    {
        return usb.controlRead(RTYP_VENDOR_READ, RQST_CYPRESS, addr, 0, data, offs, leng, timeout);
    }


    /**
     *  Writes byte array to memory
     *
     *  @param  addr  The first address to write
     *  @param  data  The array of bytes to write
     *  @return  The number of bytes written.
     *  @throws  DriverException
     */
    public int writeMemory(int addr, byte[] data) throws DriverException
    {
        return usb.controlWrite(RTYP_VENDOR_WRITE, RQST_CYPRESS, addr, 0, data, 0, data.length, 0);
    }


    /**
     *  Writes byte array to memory
     *
     *  @param  addr  The first address to write
     *  @param  data  The array of bytes to write
     *  @param  offs  The offset in data to the first byte to write from
     *  @param  leng  The number of bytes to write
     *  @return  The number of bytes written.
     *  @throws  DriverException
     */
    public int writeMemory(int addr, byte[] data, int offs, int leng) throws DriverException
    {
        return usb.controlWrite(RTYP_VENDOR_WRITE, RQST_CYPRESS, addr, 0, data, offs, leng, 0);
    }


    /**
     *  Loads memory from a file
     *
     *  @param  name  The name of the "hex" file to load
     *  @param  force  If true, force the load even if memory appears to be
     *                 already loaded correctly.
     *  @throws  DriverException
     */
    public void load(String name, boolean force) throws DriverException
    {
        if (!name.equals(loadFile)) {
            cacheFile(name);
        }
        if (force || !checkCache()) {
            loadCache();
        }
    }


    /**
     *  Caches a load file's contents
     *
     *  @param  name  The name of the "hex" file to cache
     */
    private void cacheFile(String name) throws DriverException
    {
        byte[] data;
        int fileSize = 0, nRead;
        try {
            InputStream file = Cypress.class.getResourceAsStream(name);
            if (file == null) {
                throw new FileNotFoundException(name);
            }
            data = new byte[1024];
            while (true) {
                nRead = file.read(data);
                if (nRead < 0) break;
                fileSize += nRead;
            }
            file.close();
        }
        catch (Exception e) {
            throw new DriverException("Error reading file: " + e);
        }
        try {
            InputStream file = Cypress.class.getResourceAsStream(name);
            if (file == null) {
                throw new FileNotFoundException(name);
            }
            data = new byte[fileSize];
            nRead = file.read(data);
            file.close();
        }
        catch (Exception e) {
            throw new DriverException("Error reading file: " + e);
        }
        if (nRead != fileSize) {
            throw new DriverException("Inconsistent data amount read");
        }

        loadFile = null;
        for (int j = 0; j < loadMap.length; j++)
            loadMap[j] = 0;
        int posn = 0;
        while (posn >= 0 && posn < fileSize)
            posn = processLine(data, posn);
        if (posn >= 0) {
            throw new DriverException("EOF record missing");
        }
        if (-posn < fileSize) {
            throw new DriverException("EOF record not last in file");
        }
        loadFile = name;
    }


    /**
     *  Processes a line of data
     *
     *  @param  data  Array containing the complete file data
     *  @param  posn  The current position in the data
     *  @return  The updated position in the data
     */
    private int processLine(byte[] data, int posn) throws DriverException
    {
        if (posn + 11 > data.length) {
            throw new DriverException("Record too short");
        }
        if (data[posn] != BOL_MARKER) {
            throw new DriverException("Beginning-of-line marker not found");
        }
        int leng = decodeHex(data, posn + 1);
        if (posn + 11 + 2 * leng > data.length) {
            throw new DriverException("Record too short");
        }

        posn += 3;
        int checkSum = leng, addr = 0, type = 0;
        for (int j = 0; j < leng + 4; j++, posn += 2) {
            int value = decodeHex(data, posn);
            if (j == 0) {
                addr = value << 8;
            }
            else if (j == 1) {
                addr |= value;
            }
            else if (j == 2) {
                type = value;
                if (type != 0 && type != 1) {
                    throw new DriverException("Invalid record type");
                }
            }
            else if (j < leng + 3) {
                int ad = addr + j - 3;
                if (ad >= CACHE_SIZE) {
                    throw new DriverException("Address too large");
                }
                loadImage[ad] = (byte)value;
                loadMap[ad >> 5] |= 1 << (ad & 0x1f);
            }
            checkSum += value;
        }
        if ((checkSum & 0xff) != 0) {
            throw new DriverException("Checksum failed");
        }

        if (data[posn] == 13 || data[posn] == 10) {
            posn++;
            if (data[posn] == 13 || data[posn] == 10) {
                posn++;
            }
        }

        return (type == 0) ? posn : -posn;
    }


    /**
     *  Decodes two hexadecimal bytes into an integer
     *
     *  @param  data  Array containing the data bytes
     *  @param  offs  The offset to the bytes to decode
     */
    private int decodeHex(byte[] data, int offs) throws DriverException
    {
        try {
            return Integer.parseInt(new String(data, offs, 2), 16);
        }
        catch (Exception e) {
            throw new DriverException("Invalid hexadecimal number");
        }
    }


    /**
     *  Loads memory from the cache
     */
    private void loadCache() throws DriverException
    {
        setReset(true);

        int start = 0;
        boolean skip = true;
        for (int j = 0; j < loadImage.length; j++) {
            boolean set = (loadMap[j >> 5] & (1 << (j & 0x1f))) != 0;
            if (skip ^ set) continue;
            if (skip) {
                start = j;
            }
            else {
                int leng = j - start;
                while (leng > 0) {
                    int wLeng = leng > PACKET_SIZE ? PACKET_SIZE : leng;
                    writeMemory(start, loadImage, start, wLeng);
                    leng -= wLeng;
                    start += wLeng;
                }
            }
            skip = !skip;
        }

        setReset(false);
    }


    /**
     *  Checks whether memory contents matches the cache
     */
    private boolean checkCache() throws DriverException
    {
        setReset(true);

        boolean match = true, skip = true;
        int start = 0;
        for (int j = 0; j < loadImage.length; j++) {
            boolean set = (loadMap[j >> 5] & (1 << (j & 0x1f))) != 0;
            if (skip ^ set) continue;
            if (skip) {
                start = j;
            }
            else {
                int leng = j - start;
                if (leng > PACKET_SIZE) {
                    leng = PACKET_SIZE;
                }
                byte[] data = new byte[leng];
                readMemory(start, data, 1000);
                for (int k = 0; k < leng; k++) {
                    if (data[k] != loadImage[start + k]) {
                        match = false;
                    }
                }
                break;
            }
            skip = !skip;
        }

        setReset(false);

        return match;
    }

}
