package org.lsst.ccs.drivers.ad;

import java.util.ArrayList;
import javax.usb.UsbException;
import jline.console.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProcess;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to test an Analog Devices 7794 evaluation board
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestAd7794 implements CmndProcess.Dispatch {

    /*
    **  Command codes
    */
    private final static int
        CMD_READMEM     = 0,
        CMD_WRITEMEM    = 1,
        CMD_LOAD        = 2,
        CMD_RESET       = 3,
        CMD_READREG     = 4,
        CMD_WRITEREG    = 5,
        CMD_READADC     = 6,
        CMD_READTEMP    = 7,
        CMD_READVDD     = 8,
        CMD_INIT        = 9,
        CMD_SHOWREGS    = 10,
        CMD_SETCONV     = 11,
        CMD_SHOWCONV    = 12,
        CMD_SETRATE     = 13,
        CMD_SHOWRATE    = 14,
        CMD_SETCLOCK    = 15,
        CMD_SHOWCLOCK   = 16,
        CMD_SETGAIN     = 17,
        CMD_SHOWGAIN    = 18,
        CMD_SETREFSEL   = 19,
        CMD_SHOWREFSEL  = 20,
        CMD_SETREFDET   = 21,
        CMD_SHOWREFDET  = 22,
        CMD_SETBUFF     = 23,
        CMD_SHOWBUFF    = 24,
        CMD_SETUNIPOL   = 25,
        CMD_SHOWUNIPOL  = 26,
        CMD_SETSWITCH   = 27,
        CMD_SHOWSWITCH  = 28,
        CMD_SHOWSTATUS  = 29,
        CMD_READTHERM   = 30,
        NUM_CMDS        = 31;

    /*
    **  Command help text
    */
    private final static String[] helpInit = {
        "Initializes setup registers to standard values",
        "init",
    };

    private final static String[] helpShowstatus = {
        "Display status register contents",
        "showstatus",
    };

    private final static String[] helpSetconv = {
        "Set conversion mode",
        "setconv <value>",
        "value  The encoded conversion mode (0 - 7)",
    };

    private final static String[] helpShowconv = {
        "Display conversion mode",
        "showconv",
    };

    private final static String[] helpSetrate = {
        "Set conversion rate",
        "setrate <value>",
        "value  The encoded conversion rate (0 - 15)",
    };

    private final static String[] helpShowrate = {
        "Display conversion rate",
        "showrate",
    };

    private final static String[] helpSetclock = {
        "Set clock source",
        "setclock <value>",
        "value  The encoded clock source value (0 - 3)",
    };

    private final static String[] helpShowclock = {
        "Display clock source",
        "showclock",
    };

    private final static String[] helpSetgain = {
        "Set the gain",
        "setgain <value>",
        "value  The gain value (0 - 7)",
    };

    private final static String[] helpShowgain = {
        "Display the gain",
        "showgain",
    };

    private final static String[] helpSetrefsel = {
        "Set the reference source",
        "setrefsel <value>",
        "value  The encoded reference source (0 - 3)",
    };

    private final static String[] helpShowrefsel = {
        "Display reference source",
        "showrefsel",
    };

    private final static String[] helpSetrefdet = {
        "Set reference detection on or off",
        "setrefdet {on | off}",
    };

    private final static String[] helpShowrefdet = {
        "Display reference detection",
        "showrefdet",
    };

    private final static String[] helpSetbuff = {
        "Set buffered mode on or off",
        "setbuff {on | off}",
    };

    private final static String[] helpShowbuff = {
        "Display conversion mode",
        "showbuff",
    };

    private final static String[] helpSetunipol = {
        "Set unipolar mode on or off",
        "setunipol {on | off}",
    };

    private final static String[] helpShowunipol = {
        "Display unipolar mode",
        "showunipol",
    };

    private final static String[] helpSetswitch = {
        "Set power switch on or off",
        "setswitch {on | off}",
    };

    private final static String[] helpShowswitch = {
        "Display power switch state",
        "showswitch",
    };

    private final static String[] helpReadadc = {
        "Read an ADC channel and display the raw value",
        "readadc <chan> [<config>] [<single>] [<immed>] [<count>]",
        "chan    The ADC channel name or number (0 - 7)",
        "config  If present and non-zero, configure conversion mode",
        "single  If present and non-zero, single conversion, not continuous",
        "immed   If present and non-zero, don't wait for conversion",
        "count   The number of reads to average (default 1)",
    };

    private final static String[] helpReadtemp = {
        "Read the internal temperature sensor and display the value",
        "readtemp [<config>] [<single>] [<immed>] [<count>]",
        "config  If present and non-zero, configure conversion mode",
        "single  If present and non-zero, single conversion, not continuous",
        "immed   If present and non-zero, don't wait for conversion",
        "count   The number of reads to average (default 1)",
    };

    private final static String[] helpReadvdd = {
        "Read the internal voltage and display the value",
        "readvdd [<config>] [<single>] [<immed>] [<count>]",
        "config  If present and non-zero, configure conversion mode",
        "single  If present and non-zero, single conversion, not continuous",
        "immed   If present and non-zero, don't wait for conversion",
        "count   The number of reads to average (default 1)",
    };

    private final static String[] helpReadtherm = {
        "Read the on-board thermistor and display the value",
        "readtherm [<config>] [<single>] [<immed>] [<count>]",
        "config  If present and non-zero, configure conversion mode",
        "single  If present and non-zero, single conversion, not continuous",
        "immed   If present and non-zero, don't wait for conversion",
        "count   The number of reads to average (default 1)",
    };

    private final static String[] helpShowregs = {
        "Show formatted display of all registers",
        "showregs",
    };

    private final static String[] helpReadreg = {
        "Read register and display the contents",
        "readreg <regnum>",
        "regnum   The number of the register to read",
    };

    private final static String[] helpWritereg = {
        "Write to register",
        "writereg <regnum> <value>",
        "regnum  The number of the register to write",
        "value   The value to write",
    };

    private final static String[] helpReadmem = {
        "Read data from memory and display it",
        "readmem <addr> <leng> [<timwout>]",
        "addr     The address to read from",
        "leng     The number of bytes to read",
        "timeout  The timeout value (secs, default 1.0)",
    };

    private final static String[] helpWritemem = {
        "Write data to memory",
        "writemem <addr> [<data1>]... [<data8>]",
        "addr   The address to write to",
        "datan  String of (an even number of) hexadecimal digits to write",
    };

    private final static String[] helpReset = {
        "Set or clear the RESET state",
        "reset  <opern>",
        "opern  If non-zero, RESET is set; if zero, RESET is cleared",
    };

    private final static String[] helpLoad = {
        "Load memory from a file",
        "load  [<file>] [<force>]",
        "file   The name of the .hex file to load (default: standard)",
        "force  If present and non-zero, do the load even if unnecessary",
    };

    /*
    **  Lookup tables
    */
    private final static CmndProcess.Lookup chanNames;
    static {
        chanNames = new CmndProcess.Lookup(8);
        chanNames.add("ain1",        Ad7794Eval.CHAN_AIN1);
        chanNames.add("ain2",        Ad7794Eval.CHAN_AIN2);
        chanNames.add("ain3",        Ad7794Eval.CHAN_AIN3);
        chanNames.add("ain4",        Ad7794Eval.CHAN_AIN4);
        chanNames.add("ain5",        Ad7794Eval.CHAN_AIN5);
        chanNames.add("ain6",        Ad7794Eval.CHAN_AIN6);
        chanNames.add("temperature", Ad7794Eval.CHAN_TEMP);
        chanNames.add("vdd",         Ad7794Eval.CHAN_VDD);
    }

    private final static CmndProcess.Lookup cnvModeNames;
    static {
        cnvModeNames = new CmndProcess.Lookup(8);
        cnvModeNames.add("continuous", Ad7794Eval.MODE_CONT);
        cnvModeNames.add("single",     Ad7794Eval.MODE_SINGLE);
        cnvModeNames.add("idle",       Ad7794Eval.MODE_IDLE);
        cnvModeNames.add("powerdown",  Ad7794Eval.MODE_PWR_DOWN);
        cnvModeNames.add("intoffcal",  Ad7794Eval.MODE_INT_Z_CAL);
        cnvModeNames.add("intgaincal", Ad7794Eval.MODE_INT_F_CAL);
        cnvModeNames.add("extoffcal",  Ad7794Eval.MODE_SYS_Z_CAL);
        cnvModeNames.add("extgaincal", Ad7794Eval.MODE_SYS_F_CAL);
    }

    private final static CmndProcess.Lookup rateNames;
    static {
        rateNames = new CmndProcess.Lookup(15);
        rateNames.add("470Hz", Ad7794Eval.FRS_RATE_470);
        rateNames.add("242Hz", Ad7794Eval.FRS_RATE_242);
        rateNames.add("123Hz", Ad7794Eval.FRS_RATE_123);
        rateNames.add("62Hz",  Ad7794Eval.FRS_RATE_62);
        rateNames.add("50Hz",  Ad7794Eval.FRS_RATE_50);
        rateNames.add("39Hz",  Ad7794Eval.FRS_RATE_39);
        rateNames.add("33Hz",  Ad7794Eval.FRS_RATE_33);
        rateNames.add("19Hz",  Ad7794Eval.FRS_RATE_19);
        rateNames.add("17Hz",  Ad7794Eval.FRS_RATE_17);
        rateNames.add("16Hz",  Ad7794Eval.FRS_RATE_16);
        rateNames.add("12Hz",  Ad7794Eval.FRS_RATE_12);
        rateNames.add("10Hz",  Ad7794Eval.FRS_RATE_10);
        rateNames.add("8Hz",   Ad7794Eval.FRS_RATE_8);
        rateNames.add("6Hz",   Ad7794Eval.FRS_RATE_6);
        rateNames.add("4Hz",   Ad7794Eval.FRS_RATE_4);
    }

    private final static CmndProcess.Lookup clockNames;
    static {
        clockNames = new CmndProcess.Lookup(4);
        clockNames.add("internal", Ad7794Eval.CLOCK_INT);
        clockNames.add("intavail", Ad7794Eval.CLOCK_INT_AVL);
        clockNames.add("external", Ad7794Eval.CLOCK_EXT);
        clockNames.add("exthalf",  Ad7794Eval.CLOCK_EXT_HALF);
    }

    private final static CmndProcess.Lookup gainNames;
    static {
        gainNames = new CmndProcess.Lookup(8);
        gainNames.add("1X",   Ad7794Eval.GAIN_1);
        gainNames.add("2X",   Ad7794Eval.GAIN_2);
        gainNames.add("4X",   Ad7794Eval.GAIN_4);
        gainNames.add("8X",   Ad7794Eval.GAIN_8);
        gainNames.add("16X",  Ad7794Eval.GAIN_16);
        gainNames.add("32X",  Ad7794Eval.GAIN_32);
        gainNames.add("64X",  Ad7794Eval.GAIN_64);
        gainNames.add("128X", Ad7794Eval.GAIN_128);
    }

    private final static CmndProcess.Lookup refsNames;
    static {
        refsNames = new CmndProcess.Lookup(3);
        refsNames.add("ext1",     Ad7794Eval.REFSEL_EXT1);
        refsNames.add("ext2",     Ad7794Eval.REFSEL_EXT2);
        refsNames.add("internal", Ad7794Eval.REFSEL_INT);
    }

    private final static CmndProcess.Lookup onOffNames;
    static {
        onOffNames = new CmndProcess.Lookup(2);
        onOffNames.add("on",  1);
        onOffNames.add("off", 0);
    }

    /*
    **  Command table
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("init",       CMD_INIT,       helpInit,       "");
        cmnd.add("readadc",    CMD_READADC,    helpReadadc,    "jiiii",
                 chanNames);
        cmnd.add("readtemp",   CMD_READTEMP,   helpReadtemp,   "iiii");
        cmnd.add("readvdd",    CMD_READVDD,    helpReadvdd,    "iiii");
        cmnd.add("readtherm",  CMD_READTHERM,  helpReadtherm,  "iiii");
        cmnd.add("showregs",   CMD_SHOWREGS,   helpShowregs,   "");
        cmnd.add("readreg",    CMD_READREG,    helpReadreg,    "I");
        cmnd.add("writereg",   CMD_WRITEREG,   helpWritereg,   "II");
        cmnd.add("showstatus", CMD_SHOWSTATUS, helpShowstatus, "");
        cmnd.add("setconv",    CMD_SETCONV,    helpSetconv,    "j",
                 cnvModeNames);
        cmnd.add("showconv",   CMD_SHOWCONV,   helpShowconv,   "");
        cmnd.add("setrate",    CMD_SETRATE,    helpSetrate,    "j",
                 rateNames);
        cmnd.add("showrate",   CMD_SHOWRATE,   helpShowrate,   "");
        cmnd.add("setclock",   CMD_SETCLOCK,   helpSetclock,   "j",
                 clockNames);
        cmnd.add("showclock",  CMD_SHOWCLOCK,  helpShowclock,  "");
        cmnd.add("setgain",    CMD_SETGAIN,    helpSetgain,    "j",
                 gainNames);
        cmnd.add("showgain",   CMD_SHOWGAIN,   helpShowgain,   "");
        cmnd.add("setrefsel",  CMD_SETREFSEL,  helpSetrefsel,  "j",
                 refsNames);
        cmnd.add("showrefsel", CMD_SHOWREFSEL, helpShowrefsel, "");
        cmnd.add("setrefdet",  CMD_SETREFDET,  helpSetrefdet,  "K",
                 onOffNames);
        cmnd.add("showrefdet", CMD_SHOWREFDET, helpShowrefdet, "");
        cmnd.add("setbuff",    CMD_SETBUFF,    helpSetbuff,    "K",
                 onOffNames);
        cmnd.add("showbuff",   CMD_SHOWBUFF,   helpShowbuff,   "");
        cmnd.add("setunipol",  CMD_SETUNIPOL,  helpSetunipol,  "K",
                 onOffNames);
        cmnd.add("showunipol", CMD_SHOWUNIPOL, helpShowunipol, "");
        cmnd.add("setswitch",  CMD_SETSWITCH,  helpSetswitch,  "K",
                 onOffNames);
        cmnd.add("showswitch", CMD_SHOWSWITCH, helpShowswitch, "");
        cmnd.add("readmem",    CMD_READMEM,    helpReadmem,    "IIf");
        cmnd.add("writemem",   CMD_WRITEMEM,   helpWritemem,   "Issssssss");
        cmnd.add("reset",      CMD_RESET,      helpReset,      "I");
        cmnd.add("load",       CMD_LOAD,       helpLoad,       "si");
    }

    /*
    **  Private fields
    */
    private final static Output out = new ConsOut();
    private final CmndProcess proc = new CmndProcess();
    private final boolean debug;
    private Ad7794Eval ad;


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public TestAd7794(boolean debug)
    {
        this.debug = debug;
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        int index = (args.length >= 1) ? Integer.decode(args[0]) : 0;
        (new TestAd7794(args.length >= 2)).run(index);
    }


   /**
    ***************************************************************************
    **
    **  Runs the test
    **
    ***************************************************************************
    */
    public void run(int index)
    {
        proc.add(this, cmnd);

        try {
            ad = new Ad7794Eval(index);

            ConsoleReader reader = new ConsoleReader();
            while (true) {
                String line = reader.readLine(">> ");
                if (line == null || !proc.process(line)) break;
            }
        }
        catch (Exception e) {
            showException(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatches command for processing
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        try {
            switch (code) {
            case CMD_INIT:
                procInit(found, args); break;
            case CMD_SHOWSTATUS:
                procShowStatus(found, args); break;
            case CMD_SETCONV:
                procSetConv(found, args); break;
            case CMD_SHOWCONV:
                procShowConv(found, args); break;
            case CMD_SETRATE:
                procSetRate(found, args); break;
            case CMD_SHOWRATE:
                procShowRate(found, args); break;
            case CMD_SETCLOCK:
                procSetClock(found, args); break;
            case CMD_SHOWCLOCK:
                procShowClock(found, args); break;
            case CMD_SETGAIN:
                procSetGain(found, args); break;
            case CMD_SHOWGAIN:
                procShowGain(found, args); break;
            case CMD_SETREFSEL:
                procSetRefsel(found, args); break;
            case CMD_SHOWREFSEL:
                procShowRefsel(found, args); break;
            case CMD_SETREFDET:
                procSetRefdet(found, args); break;
            case CMD_SHOWREFDET:
                procShowRefdet(found, args); break;
            case CMD_SETBUFF:
                procSetBuff(found, args); break;
            case CMD_SHOWBUFF:
                procShowBuff(found, args); break;
            case CMD_SETUNIPOL:
                procSetUnipol(found, args); break;
            case CMD_SHOWUNIPOL:
                procShowUnipol(found, args); break;
            case CMD_SETSWITCH:
                procSetSwitch(found, args); break;
            case CMD_SHOWSWITCH:
                procShowSwitch(found, args); break;
            case CMD_READADC:
                procReadAdc(found, args); break;
            case CMD_READTEMP:
                procReadTemp(found, args); break;
            case CMD_READVDD:
                procReadVdd(found, args); break;
            case CMD_READTHERM:
                procReadTherm(found, args); break;
            case CMD_SHOWREGS:
                procShowRegs(found, args); break;
            case CMD_READREG:
                procReadReg(found, args); break;
            case CMD_WRITEREG:
                procWriteReg(found, args); break;
            case CMD_READMEM:
                procReadMem(found, args); break;
            case CMD_WRITEMEM:
                procWriteMem(found, args); break;
            case CMD_RESET:
                procReset(found, args); break;
            case CMD_LOAD:
                procLoad(found, args); break;
            default:
                out.println("Command not fully implemented");
            }
        }
        catch (UsbException e) {
            showException(e);
        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the INIT command
    **
    ***************************************************************************
    */
    private void procInit(int found, Object[] args) throws UsbException
    {
        ad.setupStandard();
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWSTATUS command
    **
    ***************************************************************************
    */
    private void procShowStatus(int found, Object[] args) throws UsbException
    {
        int status = ad.getStatus();
        out.format("Status: %sready, %serror, %sextref, %s, chan = %s\n",
                   (status & Ad7794Eval.STS_READY) == 0 ? "" : "not ",
                   (status & Ad7794Eval.STS_ERROR) != 0 ? "" : "no ",
                   (status & Ad7794Eval.STS_NOXREF) == 0 ? "" : "no ",
                   (status & Ad7794Eval.STS_AD7794) != 0 ? "AD7794" : "AD7795",
                   chanNames.decode(status & Ad7794Eval.STS_CHANNEL_M));
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETCONV command
    **
    ***************************************************************************
    */
    private void procSetConv(int found, Object[] args) throws UsbException
    {
        if ((found & 0x01) == 0) {
            showKeywords("conversion mode", cnvModeNames);
            return;
        }
        ad.setConvMode((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWCONV command
    **
    ***************************************************************************
    */
    private void procShowConv(int found, Object[] args) throws UsbException
    {
        int mode = ad.getConvMode();
        out.format("Conversion mode = %s (%s)\n", mode,
                   cnvModeNames.decode(mode));
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETRATE command
    **
    ***************************************************************************
    */
    private void procSetRate(int found, Object[] args) throws UsbException
    {
        if ((found & 0x01) == 0) {
            showKeywords("conversion rate", rateNames);
            return;
        }
        ad.setConvRate((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWRATE command
    **
    ***************************************************************************
    */
    private void procShowRate(int found, Object[] args) throws UsbException
    {
        int rate = ad.getConvRate();
        out.format("Conversion rate = %s (%s)\n", rate, rateNames.decode(rate));
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETCLOCK command
    **
    ***************************************************************************
    */
    private void procSetClock(int found, Object[] args) throws UsbException
    {
        if ((found & 0x01) == 0) {
            showKeywords("clock source", clockNames);
            return;
        }
        ad.setClockSource((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWCLOCK command
    **
    ***************************************************************************
    */
    private void procShowClock(int found, Object[] args) throws UsbException
    {
        int clock = ad.getClockSource();
        out.format("Clock source = %s (%s)\n", clock, clockNames.decode(clock));
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETGAIN command
    **
    ***************************************************************************
    */
    private void procSetGain(int found, Object[] args) throws UsbException
    {
        if ((found & 0x01) == 0) {
            showKeywords("gain", gainNames);
            return;
        }
        ad.setGain((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWGAIN command
    **
    ***************************************************************************
    */
    private void procShowGain(int found, Object[] args) throws UsbException
    {
        int gain = ad.getGain();
        out.format("ADC gain = %s (%s)\n", gain, gainNames.decode(gain));
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETREFSEL command
    **
    ***************************************************************************
    */
    private void procSetRefsel(int found, Object[] args) throws UsbException
    {
        if ((found & 0x01) == 0) {
            showKeywords("reference source", refsNames);
            return;
        }
        ad.setRefSelect((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWREFSEL command
    **
    ***************************************************************************
    */
    private void procShowRefsel(int found, Object[] args) throws UsbException
    {
        int refs = ad.getRefSelect();
        out.format("Reference source = %s (%s)\n", refs,
                   refsNames.decode(refs));
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETREFDET command
    **
    ***************************************************************************
    */
    private void procSetRefdet(int found, Object[] args) throws UsbException
    {
        ad.setRefDetect((Integer)args[0] != 0);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWREFDET command
    **
    ***************************************************************************
    */
    private void procShowRefdet(int found, Object[] args) throws UsbException
    {
        out.format("Reference detection is %s\n", onOff(ad.isRefDetect()));
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETBUFF command
    **
    ***************************************************************************
    */
    private void procSetBuff(int found, Object[] args) throws UsbException
    {
        ad.setBuffered((Integer)args[0] != 0);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWBUFF command
    **
    ***************************************************************************
    */
    private void procShowBuff(int found, Object[] args) throws UsbException
    {
        out.format("Buffered mode is %s\n", onOff(ad.isBuffered()));
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETUNIPOL command
    **
    ***************************************************************************
    */
    private void procSetUnipol(int found, Object[] args) throws UsbException
    {
        ad.setUnipolar((Integer)args[0] != 0);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWUNIPOL command
    **
    ***************************************************************************
    */
    private void procShowUnipol(int found, Object[] args) throws UsbException
    {
        out.format("Unipolar mode is %s\n", onOff(ad.isUnipolar()));
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETSWITCH command
    **
    ***************************************************************************
    */
    private void procSetSwitch(int found, Object[] args) throws UsbException
    {
        ad.setPowerSwitch((Integer)args[0] != 0);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWSWITCH command
    **
    ***************************************************************************
    */
    private void procShowSwitch(int found, Object[] args) throws UsbException
    {
        out.format("Power switch is %s\n", onOff(ad.isPowerSwitch()));
    }


   /**
    ***************************************************************************
    **
    **  Processes the READADC command
    **
    ***************************************************************************
    */
    private void procReadAdc(int found, Object[] args) throws UsbException
    {
        if ((found & 0x01) == 0) {
            showKeywords("channel", chanNames);
            return;
        }
        int chan = (Integer)args[0];
        int[] params = getParams(1, found, args);
        double sum = 0, sumsq = 0;
        for (int j = 0; j < params[1]; j++) {
            double value = ad.readAdc(chan, params[0]);
            sum += value;
            sumsq += value * value;
        }
        showStats("Raw " + chanNames.decode(chan), params[1], sum, sumsq);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READTEMP command
    **
    ***************************************************************************
    */
    private void procReadTemp(int found, Object[] args) throws UsbException
    {
        int[] params = getParams(0, found, args);
        double sum = 0, sumsq = 0;
        for (int j = 0; j < params[1]; j++) {
            double value = ad.readTemperature(params[0]);
            sum += value;
            sumsq += value * value;
        }
        showStats("Temperature", params[1], sum, sumsq);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READVDD command
    **
    ***************************************************************************
    */
    private void procReadVdd(int found, Object[] args) throws UsbException
    {
        int[] params = getParams(0, found, args);
        double sum = 0, sumsq = 0;
        for (int j = 0; j < params[1]; j++) {
            double value = ad.readVdd(params[0]);
            sum += value;
            sumsq += value * value;
        }
        showStats("Voltage", params[1], sum, sumsq);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READTHERM command
    **
    ***************************************************************************
    */
    private void procReadTherm(int found, Object[] args) throws UsbException
    {
        ad.setupThermistor();
        int[] params = getParams(0, found, args);
        double sum = 0, sumsq = 0;
        for (int j = 0; j < params[1]; j++) {
            double value = ad.readThermistor(params[0]);
            sum += value;
            sumsq += value * value;
        }
        showStats("Thermistor", params[1], sum, sumsq);
        ad.cleanupThermistor();
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWREGS command
    **
    ***************************************************************************
    */
    private static final String[] regName = {"STAT", "MODE", "CONF", "DATA",
                                             " ID ", " IO ", "OFFS", "FSCL"};
    private final static String[]
        regDesc0 = {"| ~RDY |  ERR |NOXREF|  --  | 7794 | CHAN2| CHAN1| CHAN0|",
                    "| MOD2 | MOD1 | MOD0 |  PSW |  --  |  --  |AMPCOM|  --  |",
                    "|VBIAS1|VBIAS0|BRNOUT|UNIPOL| BOOST| GAIN2| GAIN1| GAIN0|",
                    "",
                    "",
                    "|  --  |DIOENA|IODAT2|IODAT1|CSDIR1|CSDIR0|CSENA1|CSENA0|",
                    "",
                    ""},
        regDesc1 = {"",
                    "| CLK1 | CLK0 |  --  |CHPDIS| FRS3 | FRS2 | FRS1 | FRS0 |",
                    "|RFSEL1|RFSEL0|REFDET|BUFFRD| CHAN3| CHAN2| CHAN1| CHAN0|",
                    "",
                    "",
                    "",
                    "",
                    ""};
    
    private void procShowRegs(int found, Object[] args) throws UsbException
    {
        int[] regSize = ad.getRegSizes();
        for (int j = 0; j < 8; j++) {
            int value = ad.readRegister(j);
            int size = 2 * regSize[j];
            String fmt = String.format("%%d (%%s): %%s%%0%sx  %%s\n", size);
            out.format(fmt, j, regName[j], "      ".substring(size), value,
                       regDesc0[j]);
            if (!regDesc1[j].equals("")) {
                out.format("                  %s\n", regDesc1[j]);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the READREG command
    **
    ***************************************************************************
    */
    private void procReadReg(int found, Object[] args) throws UsbException
    {
        int regnum = (Integer)args[0];
        String fmt = String.format("Register %%s: %%0%sx\n",
                                   2 * ad.getRegSizes()[regnum]);
        out.format(fmt, regnum, ad.readRegister(regnum));
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITEREG command
    **
    ***************************************************************************
    */
    private void procWriteReg(int found, Object[] args) throws UsbException
    {
        ad.writeRegister((Integer)args[0], (Integer)args[1]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the RESET command
    **
    ***************************************************************************
    */
    private void procReset(int found, Object[] args) throws UsbException
    {
        ad.setReset((Integer)args[0] != 0);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READMEM command
    **
    ***************************************************************************
    */
    private void procReadMem(int found, Object[] args) throws UsbException
    {
        int addr = (Integer)args[0];
        float tmo = 1000F * ((found & 0x04) != 0 ? (Float)args[2] : 1F);
        byte[] data = new byte[(Integer)args[1]];
        dispData(addr, data, ad.readMemory(addr, data, (int)tmo));
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITEMEM command
    **
    ***************************************************************************
    */
    private void procWriteMem(int found, Object[] args) throws UsbException
    {
        ad.writeMemory((Integer)args[0], genData(found & 0x1fe, args));
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOAD command
    **
    ***************************************************************************
    */
    private void procLoad(int found, Object[] args) throws UsbException
    {
        boolean force = false;
        if ((found & 0x02) != 0 && (Integer)args[1] != 0) {
            force = true;
        }
        if ((found & 0x01) == 0) {
            ad.load(force);
        }
        else {
            ad.load((String)args[0], force);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets parameters for ADC read commands
    **
    ***************************************************************************
    */
    private static int[] getParams(int offs, int found, Object[] args)
    {
        int[] params = {0, 1};
        if ((found & (1 << offs)) != 0 && (Integer)args[offs] != 0) {
            params[0] |= Ad7794Eval.OPTN_CONFIG;
        }
        if ((found & (1 << ++offs)) != 0 && (Integer)args[offs] != 0) {
            params[0] |= Ad7794Eval.OPTN_SINGLE;
        }
        if ((found & (1 << ++offs)) != 0 && (Integer)args[offs] != 0) {
            params[0] |= Ad7794Eval.OPTN_IMMED;
        }
        if ((found & (1 << ++offs)) != 0) {
            params[1] = (Integer)args[offs];
        }

        return params;
    }


   /**
    ***************************************************************************
    **
    **  Displays mean and sigma values
    **
    ***************************************************************************
    */
    private static void showStats(String name, int count, double sum,
                                  double sumsq)
    {
        double mean = sum / count;
        double sigsq = sumsq / count - mean * mean;
        double sigma = sigsq <= 0 ? 0 : Math.sqrt(sigsq);
        out.format("%s: mean = %.7g, sigma = %.7g\n", name, mean, sigma);
    }


   /**
    ***************************************************************************
    **
    **  Displays memory data
    **
    ***************************************************************************
    */
    private static void dispData(int addr, byte[] data, int count)
    {
        for (int j = 0; j < count; j++) {
            if ((j & 0x0f) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%04x:", addr + j);
            }
            out.format(" %02x", data[j] & 0xff);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Generates write data from hexadecimal string arguments
    **
    ***************************************************************************
    */
    private static byte[] genData(int found, Object[] args)
    {
        int count = 0, mask = found;
        ArrayList list = new ArrayList();
        for (int j = 0; found != 0; found >>= 1, j++) {
            if ((found & 1) == 0) continue;
            String hex = (String)args[j];
            int leng = hex.length();
            if ((leng & 1) != 0) break;
            byte[] sData = new byte[leng / 2];
            try {
                for (int k = 0; k < leng; k += 2) {
                    String num = hex.substring(k, k + 2);
                    sData[k / 2] = (byte)Integer.parseInt(num, 16);
                }
            }
            catch (NumberFormatException e) {
                break;
            }
            count += leng / 2;
            list.add(sData);
            mask ^= (1 << j);
        }
        if (mask != 0) {
            out.println("Invalid hexadecimal string");
            return null;
        }
        byte[] data = new byte[count];
        int l = 0;
        for (int j = 0; j < list.size(); j++) {
            byte[] sData = (byte [])list.get(j);
            for (int k = 0; k < sData.length; k++) {
                data[l++] = sData[k];
            }
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Generates on/off text from boolean
    **
    ***************************************************************************
    */
    private static String onOff(boolean on)
    {
        return on ? "on" : "off";
    }


   /**
    ***************************************************************************
    **
    **  Displays allowed keyword names and codes
    **
    ***************************************************************************
    */
    private static void showKeywords(String text, CmndProcess.Lookup kwds)
    {
        out.format("Valid %s values are:\n", text);
        for (int j = 0; j < kwds.count(); j++) {
            out.format("  %s (%s)\n", kwds.name(j), kwds.code(j));
        }
    }


   /**
    ***************************************************************************
    **
    **  Displays an exception
    **
    ***************************************************************************
    */
    private void showException(Exception e)
    {
        if (debug) {
            e.printStackTrace();
        }
        else {
            out.println(e);
        }
    }

}
