package org.lsst.ccs.drivers.ad;

import javax.usb.UsbException;
import org.lsst.ccs.drivers.usb.UsbComm;

/**
 ***************************************************************************
 **
 **  Routines to access a SPI device on an Analog Devices evaluation board
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public abstract class SpiBus extends Cypress {

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int
        READ_TIMEOUT      = 1000,
        RQST_SPI          = 0xda;

    private final static String
        STD_LOAD_FILE     = "/Icv_spi.hex";

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private int
        rdyTimeout = 0;


   /**
    ***************************************************************************
    **
    **  Constructor
    **
    ***************************************************************************
    */
    public SpiBus(UsbComm com) throws UsbException
    {
        super(com);
    }


   /**
    ***************************************************************************
    **
    **  Performs any needed initialization
    **
    ***************************************************************************
    */
    void initialize() throws UsbException
    {
        load(false);
    }


   /**
    ***************************************************************************
    **
    **  Generates communications register value
    **
    **  @param  regnum  The register number.
    **
    **  @param  read    True for a read operation, false for a write.
    **
    **  @return  The (8-bit) value to be put into the communications register.
    **
    ***************************************************************************
    */
    public abstract int commReg(int regnum, boolean read);


   /**
    ***************************************************************************
    **
    **  Sets the RDY timeout period
    **
    **  @param  period  The timeout period in units of ~82 microseconds, up
    **                  to a maximum of 65535.
    **
    ***************************************************************************
    */
    public void setRdyTimeout(int period)
    {
        rdyTimeout = period;
    }


   /**
    ***************************************************************************
    **
    **  Gets the RDY timeout period
    **
    **  @return  The timeout period in units of ~82 microseconds.
    **
    ***************************************************************************
    */
    public int getRdyTimeout(int period)
    {
        return rdyTimeout;
    }


   /**
    ***************************************************************************
    **
    **  Reads a register
    **
    **  @param  regnum   The register to read
    **
    **  @param  size     The register size (bytes)
    **
    **  @return  The value read
    **
    ***************************************************************************
    */
    public int readRegister(int regnum, int size) throws UsbException
    {
        byte[] data = new byte[size + 1];
        com.read(RTYP_VENDOR_READ, RQST_SPI, commReg(regnum, true), rdyTimeout,
                 data, READ_TIMEOUT);
        int value = 0;
        for (int j = 1; j <= size; j++) {
            value = (value << 8) | (data[j] & 0xff);
        }
        return value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a register
    **
    **  @param  regnum  The register to write
    **
    **  @param  size    The register size (bytes)
    **
    **  @param  value   The value to write
    **
    ***************************************************************************
    */
    public void writeRegister(int regnum, int size, int value)
        throws UsbException
    {
        byte data[] = new byte[size + 1];
        data[0] = (byte)commReg(regnum, false);
        for (int j = 0; j < size; j++) {
            data[size - j] = (byte)value;
            value >>= 8;
        }
        com.write(RTYP_VENDOR_WRITE, RQST_SPI, 0, rdyTimeout, data);
    }


   /**
    ***************************************************************************
    **
    **  Updates a register
    **
    **  @param  regnum  The register to update
    **
    **  @param  size    The register size (bytes)
    **
    **  @param  mask    The mask of bits to update
    **
    **  @param  value   The value to write, under the mask
    **
    ***************************************************************************
    */
    public void updateRegister(int regnum, int size, int mask, int value)
        throws UsbException
    {
        int data = readRegister(regnum, size);
        writeRegister(regnum, size, (data & ~mask) | (value & mask));
    }


   /**
    ***************************************************************************
    **
    **  Loads memory from standard file
    **
    **  @param  force  If true, force the load even if memory appears to be
    **                 already loaded correctly.
    **
    ***************************************************************************
    */
    public void load(boolean force) throws UsbException
    {
        load(STD_LOAD_FILE, force);
    }

}
