package org.lsst.ccs.drivers.ad;

import javax.usb.UsbException;
import org.lsst.ccs.drivers.usb.UsbComm;

/**
 ***************************************************************************
 **
 **  Routines to access an Analog Devices AD7794 evaluation board
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Ad7794Eval extends SpiBus {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        STATUS_REG     = 0,        // Status register (8-bit)
        MODE_REG       = 1,        // Mode register (16-bit)
        CONFIG_REG     = 2,        // Configuration register (16-bit)
        DATA_REG       = 3,        // Data register (24- or 16-bit)
        ID_REG         = 4,        // ID register (8-bit)
        IO_REG         = 5,        // IO register (8-bit)
        OFFSET_REG     = 6,        // Offset register (24- or 16-bit)
        SCALE_REG      = 7,        // (Full-)Scale register (24- or 16-bit)

        COM_READ       = 0x40,     // Comm reg: read operation
        COM_ADDRESS_M  = 0x38,     // Comm reg: address mask
        COM_ADDRESS_V  = 3,        // Comm reg: address offset
        COM_CONT_READ  = 0x04,     // Comm reg: continuous read

        STS_READY      = 0x80,     // Status: (not) ready
        STS_ERROR      = 0x40,     // Status: voltage/temp data ready
        STS_NOXREF     = 0x20,     // Status: no external reference
        STS_AD7794     = 0x08,     // Status: AD7794 chip, not AD7795
        STS_CHANNEL_M  = 0x07,     // Status: channel mask
        STS_CHANNEL_V  = 0,        // Status: channel offset

        MOD_MODE_M     = 0xe000,   // Mode: mode select mask
        MOD_MODE_V     = 13,       // Mode: mode select offset
        MODE_CONT      = 0,        // Mode select: continuous conversion
        MODE_SINGLE    = 1,        // Mode select: single conversion
        MODE_IDLE      = 2,        // Mode select: idle
        MODE_PWR_DOWN  = 3,        // Mode select: power-down
        MODE_INT_Z_CAL = 4,        // Mode select: internal zero calibration
        MODE_INT_F_CAL = 5,        // Mode select: int. full-scale calibration
        MODE_SYS_Z_CAL = 6,        // Mode select: system zero calibration
        MODE_SYS_F_CAL = 7,        // Mode select: sys. full-scale calibration
        MOD_PSW        = 0x1000,   // Mode: power switch
        MOD_AMP_CM     = 0x0200,   // Mode: amplifier common mode
        MOD_CLOCK_M    = 0x00c0,   // Mode: clock source mask
        MOD_CLOCK_V    = 6,        // Mode: clock source offset
        CLOCK_INT      = 0,        // Clock source: internal 64 KHz
        CLOCK_INT_AVL  = 1,        // Clock source: int 64 KHz, avail on pin
        CLOCK_EXT      = 2,        // Clock source: external 64 KHz
        CLOCK_EXT_HALF = 3,        // Clock source: ext 64 KHz, halved
        MOD_CHOP_DIS   = 0x0010,   // Mode: chop disable
        MOD_FRS_M      = 0x000f,   // Mode: filter rate select mask
        MOD_FRS_V      = 0,        // Mode: filter rate select offset
        FRS_RATE_470   = 1,        // Filter rate: 470 Hz
        FRS_RATE_242   = 2,        // Filter rate: 242 Hz
        FRS_RATE_123   = 3,        // Filter rate: 123 Hz
        FRS_RATE_62    = 4,        // Filter rate: 62 Hz
        FRS_RATE_50    = 5,        // Filter rate: 50 Hz
        FRS_RATE_39    = 6,        // Filter rate: 39 Hz
        FRS_RATE_33    = 7,        // Filter rate: 33.2 Hz
        FRS_RATE_19    = 8,        // Filter rate: 19.6 Hz
        FRS_RATE_17    = 9,        // Filter rate: 16.7 Hz (80 dB rej'n)
        FRS_RATE_16    = 10,       // Filter rate: 16.7 Hz (65 dB rej'n)
        FRS_RATE_12    = 11,       // Filter rate: 12.5 Hz
        FRS_RATE_10    = 12,       // Filter rate: 10 Hz
        FRS_RATE_8     = 13,       // Filter rate: 8.33 Hz
        FRS_RATE_6     = 14,       // Filter rate: 6.25 Hz
        FRS_RATE_4     = 15,       // Filter rate: 4.17 Hz
        MOD_STANDARD   = FRS_RATE_16 << MOD_FRS_V,

        CFG_VBIAS_M    = 0xc000,   // Config: bias voltage mask
        CFG_VBIAS_V    = 14,       // Config: bias voltage offset
        VBIAS_OFF      = 0,        // Bias voltage: disabled
        VBIAS_AIN1     = 1,        // Bias voltage: connected to AIN1(-)
        VBIAS_AIN2     = 2,        // Bias voltage: connected to AIN2(-)
        VBIAS_AIN3     = 3,        // Bias voltage: connected to AIN3(-)
        CFG_BURNOUT    = 0x2000,   // Config: burnout current enabled
        CFG_UNIPOLAR   = 0x1000,   // Config: unipolar operation
        CFG_BOOST      = 0x0800,   // Config: boost bias voltage current
        CFG_GAIN_M     = 0x0700,   // Config: gain select mask
        CFG_GAIN_V     = 8,        // Config: gain select offset
        GAIN_1         = 0,        // ADC gain: 1
        GAIN_2         = 1,        // ADC gain: 2
        GAIN_4         = 2,        // ADC gain: 4
        GAIN_8         = 3,        // ADC gain: 8
        GAIN_16        = 4,        // ADC gain: 16
        GAIN_32        = 5,        // ADC gain: 32
        GAIN_64        = 6,        // ADC gain: 64
        GAIN_128       = 7,        // ADC gain: 128
        RANGE_2500     = GAIN_1,   // ADC range: 2.5 V
        RANGE_1250     = GAIN_2,   // ADC range: 1.25 V
        RANGE_625      = GAIN_4,   // ADC range: 0.625 V
        RANGE_312      = GAIN_8,   // ADC range: 0.3125 V
        RANGE_156      = GAIN_16,  // ADC range: 0.15625 V
        RANGE_78       = GAIN_32,  // ADC range: 0.078125 V
        RANGE_39       = GAIN_64,  // ADC range: 0.0390625 V
        RANGE_19       = GAIN_128, // ADC range: 0.01953125 V
        CFG_REFSEL_M   = 0x00c0,   // Config: reference select mask
        CFG_REFSEL_V   = 6,        // Config: reference select offset
        REFSEL_EXT1    = 0,        // Ref select: external on REFIN1
        REFSEL_EXT2    = 1,        // Ref select: external on REFIN2
        REFSEL_INT     = 2,        // Ref select: internal
        CFG_REF_DETECT = 0x0020,   // Config: reference detect enable
        CFG_BUFFERED   = 0x0010,   // Config: buffered mode
        CFG_CHANNEL_M  = 0x000f,   // Config: channel select mask
        CFG_CHANNEL_V  = 0,        // Config: channel select offset
        CHAN_AIN1      = 0,        // Channel: AIN1
        CHAN_AIN2      = 1,        // Channel: AIN2
        CHAN_AIN3      = 2,        // Channel: AIN3
        CHAN_AIN4      = 3,        // Channel: AIN4
        CHAN_AIN5      = 4,        // Channel: AIN5
        CHAN_AIN6      = 5,        // Channel: AIN6
        CHAN_TEMP      = 6,        // Channel: temperature sensor
        CHAN_VDD       = 7,        // Channel: VDD monitor
        CHAN_AIN1A     = 8,        // Channel: AIN1 (again)
        CFG_STANDARD   = (GAIN_128 << CFG_GAIN_V) | CFG_BUFFERED,

        IO_DIOENA      = 0x40,     // IO: digital IO enable
        IO_IO2DAT      = 0x20,     // IO: pin P2 data
        IO_IO1DAT      = 0x10,     // IO: pin P1 data
        IO_IEXCDIR_M   = 0x0c,     // IO: current sources direction mask
        IO_IEXCDIR_V   = 2,        // IO: current sources direction offset
        IEXCDIR_12     = 0,        // Sources direction: IOUT1 & IOUT2
        IEXCDIR_21     = 1,        // Sources direction: IOUT2 & IOUT1
        IEXCDIR_11     = 2,        // Sources direction: both to IOUT1
        IEXCDIR_22     = 3,        // Sources direction: both to IOUT2
        IO_IEXCENA_M   = 0x03,     // IO: current sources enable mask
        IO_IEXCENA_V   = 0,        // IO: current sources enable offset
        IEXCENA_OFF    = 0,        // Sources enable: disabled
        IEXCENA_10     = 1,        // Sources enable: 10 uA
        IEXCENA_210    = 2,        // Sources enable: 210 uA
        IEXCENA_1000   = 3,        // Sources enable: 1 mA
        IO_STANDARD    = 0,

        OPTN_CONFIG    = 0x01,     // Data read option: configure conv. mode
        OPTN_SINGLE    = 0x02,     // Data read option: single mode, not cont.
        OPTN_IMMED     = 0x04,     // Data read option: don't wait for ready

        DIO_ENABLE     = 0x01,     // DIO value: enable digital output
        DIO_P1DATA     = 0x02,     // DIO value: pin P1 on
        DIO_P2DATA     = 0x04;     // DIO value: pin P2 on

    public final static float
        HALF_RANGE      = 8388608F,
        REF_VOLTAGE     = 1.17F,
        INT_TEMP_SCALE  = 100F,
        INT_VOLT_SCALE  = 6F;

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int
        AD_VID            = 0x0456,
        AD_7794_PID       = 0xb480;

    private final static double
        REF_RESISTANCE    = 5000,
        BASE_RESISTANCE   = 1000,
        CELSIUS_OFFSET    = 273.15,
        BASE_TEMPERATURE  = CELSIUS_OFFSET + 25,
        THERM_B_COEFF     = 3470;

    private final static int[]
        REG_SIZE = {1, 2, 2, 3, 1, 1, 3, 3};   // Array of register sizes

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private int thermGain, thermRefsel, thermSrcdir, thermSrcena;
    private boolean thermSaved, thermUnipol;


   /**
    ***************************************************************************
    **
    **  Constructors
    **
    ***************************************************************************
    */
    public Ad7794Eval() throws UsbException
    {
        this(0);
    }

    public Ad7794Eval(int index) throws UsbException
    {
        /*
        **  Initialize parent, locate the board, load firmware
        */
        super(new UsbComm());
        if (com.findDevice(AD_VID, AD_7794_PID, index) == null) {
            throw new AdException("Board not found");
        }
        initialize();

        /*
        **  If board was powered down and back up it gets into a bad state
        **  which seems to be cured by writing a value with the lower order
        **  bit set to 1 to the mode register.  In this case, the mode
        **  register assumes its power-on value, which is not the same as
        **  the value just written.  But if the written value is read back,
        **  i.e. there was no problem, the original value must be restored.
        */  
        int mode = readRegister(MODE_REG);
        writeRegister(MODE_REG, 1);
        if (readRegister(MODE_REG) == 1) {
            writeRegister(MODE_REG, mode);
        }
    }


   /**
    ***************************************************************************
    **
    **  Generates communications register value
    **
    ***************************************************************************
    */
    @Override
    public int commReg(int regnum, boolean read)
    {
        return ((regnum << COM_ADDRESS_V) & COM_ADDRESS_M)
                 | (read ? COM_READ : 0);
    }


   /**
    ***************************************************************************
    **
    **  Performs standard board setup
    **
    ***************************************************************************
    */
    public void setupStandard() throws UsbException
    {
        writeRegister(MODE_REG, MOD_STANDARD);
        writeRegister(CONFIG_REG, CFG_STANDARD);
        writeRegister(IO_REG, IO_STANDARD);
    }


   /**
    ***************************************************************************
    **
    **  Gets the array of register sizes
    **
    ***************************************************************************
    */
    public int[] getRegSizes()
    {
        return REG_SIZE;
    }


   /**
    ***************************************************************************
    **
    **  Gets the board status
    **
    ***************************************************************************
    */
    public int getStatus() throws UsbException
    {
        return readRegister(STATUS_REG);
    }


   /**
    ***************************************************************************
    **
    **  Sets the conversion mode
    **
    **  @param  mode  The mode value to set
    **
    ***************************************************************************
    */
    public void setConvMode(int mode) throws UsbException
    {
        updateRegister(MODE_REG, MOD_MODE_M, mode << MOD_MODE_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the conversion mode
    **
    **  @return  The mode value
    **
    ***************************************************************************
    */
    public int getConvMode() throws UsbException
    {
        return (readRegister(MODE_REG) & MOD_MODE_M) >> MOD_MODE_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the clock source
    **
    **  @param  source  The source value to set
    **
    ***************************************************************************
    */
    public void setClockSource(int source) throws UsbException
    {
        updateRegister(MODE_REG, MOD_CLOCK_M, source << MOD_CLOCK_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the clock source
    **
    **  @return  The source value
    **
    ***************************************************************************
    */
    public int getClockSource() throws UsbException
    {
        return (readRegister(MODE_REG) & MOD_CLOCK_M) >> MOD_CLOCK_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the power switch state
    **
    **  @param  on  Whether or not to set power switch on
    **
    ***************************************************************************
    */
    public void setPowerSwitch(boolean on) throws UsbException
    {
        updateRegister(MODE_REG, MOD_PSW, on ? MOD_PSW : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the power switch state
    **
    **  @return  Whether or not power switch is on
    **
    ***************************************************************************
    */
    public boolean isPowerSwitch() throws UsbException
    {
        return (readRegister(MODE_REG) & MOD_PSW) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the conversion rate
    **
    **  @param  rate  The rate value to set
    **
    ***************************************************************************
    */
    public void setConvRate(int rate) throws UsbException
    {
        updateRegister(MODE_REG, MOD_FRS_M, rate << MOD_FRS_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the conversion rate
    **
    **  @return  The conversion rate
    **
    ***************************************************************************
    */
    public int getConvRate() throws UsbException
    {
        return (readRegister(MODE_REG) & MOD_FRS_M) >> MOD_FRS_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the unipolar mode
    **
    **  @param  enable  Whether or not to set unipolar mode
    **
    ***************************************************************************
    */
    public void setUnipolar(boolean enable) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_UNIPOLAR, enable ? CFG_UNIPOLAR : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the unipolar mode
    **
    **  @return  Whether or not unipolar mode is set
    **
    ***************************************************************************
    */
    public boolean isUnipolar() throws UsbException
    {
        return (readRegister(CONFIG_REG) & CFG_UNIPOLAR) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the buffered mode
    **
    **  @param  enable  Whether or not to set buffered mode
    **
    ***************************************************************************
    */
    public void setBuffered(boolean enable) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_BUFFERED, enable ? CFG_BUFFERED : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the buffered mode
    **
    **  @return  Whether or not buffered mode is set
    **
    ***************************************************************************
    */
    public boolean isBuffered() throws UsbException
    {
        return (readRegister(CONFIG_REG) & CFG_BUFFERED) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the reference detect mode
    **
    **  @param  enable  Whether or not to set reference detect mode
    **
    ***************************************************************************
    */
    public void setRefDetect(boolean enable) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_REF_DETECT, enable ? CFG_REF_DETECT : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the reference detect mode
    **
    **  @return  Whether or not reference detect mode is set
    **
    ***************************************************************************
    */
    public boolean isRefDetect() throws UsbException
    {
        return (readRegister(CONFIG_REG) & CFG_REF_DETECT) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the reference selection
    **
    **  @param  mode  The reference selection value to set
    **
    ***************************************************************************
    */
    public void setRefSelect(int mode) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_REFSEL_M, mode << CFG_REFSEL_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the reference selection
    **
    **  @return  The reference selection value
    **
    ***************************************************************************
    */
    public int getRefSelect() throws UsbException
    {
        return (readRegister(CONFIG_REG) & CFG_REFSEL_M) >> CFG_REFSEL_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the ADC gain
    **
    **  @param  gain  The gain value to set
    **
    ***************************************************************************
    */
    public void setGain(int gain) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_GAIN_M, gain << CFG_GAIN_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the ADC gain
    **
    **  @return  The gain value
    **
    ***************************************************************************
    */
    public int getGain() throws UsbException
    {
        return (readRegister(CONFIG_REG) & CFG_GAIN_M) >> CFG_GAIN_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the DIO values
    **
    **  @param  value  The DIO values to set, the OR of the following:
    **                   DIO_ENABLE: enable digital output
    **                   DIO_P1DATA: set 1 on pin P1
    **                   DIO_P2DATA: set 1 on pin P2
    **
    ***************************************************************************
    */
    public void setDio(int value) throws UsbException
    {
        int mask = ((value & DIO_ENABLE) != 0 ? IO_DIOENA : 0)
                     | ((value & DIO_P1DATA) != 0 ? IO_IO1DAT : 0)
                     | ((value & DIO_P2DATA) != 0 ? IO_IO2DAT : 0);
        updateRegister(IO_REG, IO_DIOENA | IO_IO1DAT | IO_IO2DAT, mask);
    }


   /**
    ***************************************************************************
    **
    **  Gets the DIO values
    **
    **  @return  The DIO values currently set, encoded as follows:
    **             DIO_ENABLE: digital output enabled
    **             DIO_P1DATA: 1 on pin P1
    **             DIO_P2DATA: 1 on pin P2
    **
    ***************************************************************************
    */
    public int getDio() throws UsbException
    {
        int mask = readRegister(IO_REG) & (IO_DIOENA | IO_IO1DAT | IO_IO2DAT);
        return ((mask & IO_DIOENA) != 0 ? DIO_ENABLE : 0)
                 | ((mask &IO_IO1DAT) != 0 ? DIO_P1DATA : 0)
                 | ((mask &IO_IO2DAT) != 0 ? DIO_P2DATA : 0);
    }


   /**
    ***************************************************************************
    **
    **  Sets the current source direction
    **
    **  @param  dirn  The direction value to set
    **
    ***************************************************************************
    */
    public void setSourceDirection(int dirn) throws UsbException
    {
        updateRegister(IO_REG, IO_IEXCDIR_M, dirn << IO_IEXCDIR_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the current source direction
    **
    **  @return  The direction value
    **
    ***************************************************************************
    */
    public int getSourceDirection() throws UsbException
    {
        return (readRegister(IO_REG) & IO_IEXCDIR_M) >> IO_IEXCDIR_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the current source enable value
    **
    **  @param  enab  The enable value to set
    **
    ***************************************************************************
    */
    public void setSourceEnable(int enab) throws UsbException
    {
        updateRegister(IO_REG, IO_IEXCENA_M, enab << IO_IEXCENA_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the current source enable value
    **
    **  @return  The enable value
    **
    ***************************************************************************
    */
    public int getSourceEnable() throws UsbException
    {
        return (readRegister(IO_REG) & IO_IEXCENA_M) >> IO_IEXCENA_V;
    }


   /**
    ***************************************************************************
    **
    **  Reads an ADC channel
    **
    **  @param  chan   The ADC channel to read (0 - 7)
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 OPTN_CONFIG: First configures the conversion mode.
    **
    **                 OPTN_SINGLE: Configures for a single conversion,
    **                              otherwise continuous.
    **
    **                 OPTN_IMMED:  Reads the data register immediately
    **                              instead of waiting for a conversion to
    **                              complete.
    **
    **  @return  The read value (raw counts)
    **
    ***************************************************************************
    */
    public int readAdc(int chan, int optns) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_CHANNEL_M, chan << CFG_CHANNEL_V);
        if ((optns & OPTN_CONFIG) != 0) {
            int mode = (optns & OPTN_SINGLE) != 0 ? MODE_SINGLE : MODE_CONT;
            updateRegister(MODE_REG, MOD_MODE_M, mode << MOD_MODE_V);
        }
        if ((optns & OPTN_IMMED) == 0) {
            waitReady();
        }

        return readRegister(DATA_REG);
    }


   /**
    ***************************************************************************
    **
    **  Reads the internal temperature sensor
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 OPTN_CONFIG: First configures the conversion mode.
    **
    **                 OPTN_SINGLE: Configures for a single conversion,
    **                              otherwise continuous.
    **
    **                 OPTN_IMMED:  Reads the data register immediately
    **                              instead of waiting for a conversion to
    **                              complete.
    **
    **  @return  The read temperature (Celsius)
    **
    ***************************************************************************
    */
    public float readTemperature(int optns) throws UsbException
    {
        return (readAdc(CHAN_TEMP, optns) / HALF_RANGE - 1F) * REF_VOLTAGE
                 * INT_TEMP_SCALE;
    }


   /**
    ***************************************************************************
    **
    **  Reads the internal voltage monitor
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 OPTN_CONFIG: First configures the conversion mode.
    **
    **                 OPTN_SINGLE: Configures for a single conversion,
    **                              otherwise continuous.
    **
    **                 OPTN_IMMED:  Reads the data register immediately
    **                              instead of waiting for a conversion to
    **                              complete.
    **
    **  @return  The read voltage
    **
    ***************************************************************************
    */
    public float readVdd(int optns) throws UsbException
    {
        return (readAdc(CHAN_VDD, optns) / HALF_RANGE - 1F) * REF_VOLTAGE
                 * INT_VOLT_SCALE;
    }


   /**
    ***************************************************************************
    **
    **  Reads the on-board thermistor
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 OPTN_CONFIG: First configures the conversion mode.
    **
    **                 OPTN_SINGLE: Configures for a single conversion,
    **                              otherwise continuous.
    **
    **                 OPTN_IMMED:  Reads the data register immediately
    **                              instead of waiting for a conversion to
    **                              complete.
    **
    **  @return  The read temperature (Celsius)
    **
    ***************************************************************************
    */
    public float readThermistor(int optns) throws UsbException
    {
        double r = (readAdc(CHAN_AIN4, optns) / HALF_RANGE - 1)
                     * REF_RESISTANCE;
        double t = 1 / (1 / BASE_TEMPERATURE
                          + Math.log(r / BASE_RESISTANCE) / THERM_B_COEFF)
                     - CELSIUS_OFFSET;

        return (float)t;
    }


   /**
    ***************************************************************************
    **
    **  Sets up for reading the thermistor
    **
    ***************************************************************************
    */
    public void setupThermistor() throws UsbException
    {
        thermGain = getGain();
        thermUnipol = isUnipolar();
        thermRefsel = getRefSelect();
        thermSrcdir = getSourceDirection();
        thermSrcena = getSourceEnable();
        thermSaved = true;
        setGain(GAIN_1);
        setUnipolar(false);
        setRefSelect(REFSEL_EXT1);
        setSourceDirection(IEXCDIR_22);
        setSourceEnable(IEXCENA_210);
    }


   /**
    ***************************************************************************
    **
    **  Cleans up after reading the thermistor
    **
    ***************************************************************************
    */
    public void cleanupThermistor() throws UsbException
    {
        if (thermSaved) {
            thermSaved = false;
            setGain(thermGain);
            setUnipolar(thermUnipol);
            setRefSelect(thermRefsel);
            setSourceDirection(thermSrcdir);
            setSourceEnable(thermSrcena);
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a register
    **
    **  @param  regnum   The register to read
    **
    **  @return  The value read
    **
    ***************************************************************************
    */
    public int readRegister(int regnum) throws UsbException
    {
        return readRegister(regnum, REG_SIZE[regnum & 0x07]);
    }


   /**
    ***************************************************************************
    **
    **  Writes a register
    **
    **  @param  regnum  The register to write
    **
    **  @param  value   The value to write
    **
    ***************************************************************************
    */
    public void writeRegister(int regnum, int value) throws UsbException
    {
        writeRegister(regnum, REG_SIZE[regnum & 0x07], value);
    }


   /**
    ***************************************************************************
    **
    **  Updates a register
    **
    **  @param  regnum  The register to update
    **
    **  @param  mask    The mask of bits to update
    **
    **  @param  value   The value to write, under the mask
    **
    ***************************************************************************
    */
    public void updateRegister(int regnum, int mask, int value)
        throws UsbException
    {
        updateRegister(regnum, REG_SIZE[regnum & 0x07], mask, value);
    }


   /**
    ***************************************************************************
    **
    **  Waits for the board to become ready after a conversion
    **
    ***************************************************************************
    */
    private void waitReady() throws UsbException
    {
        long start = System.currentTimeMillis();
        boolean ready = false;
        while (!(ready = (getStatus() & STS_READY) == 0)
                 && System.currentTimeMillis() - start <= 5000) {
            try {
                Thread.sleep(5);
            }
            catch (InterruptedException e) {
            }
        }
        if (!ready) {
            throw new AdException("Conversion timed out");
        }
    }

}
