package org.lsst.ccs.drivers.ad;

import javax.usb.UsbException;
import org.lsst.ccs.drivers.usb.UsbComm;

/**
 ***************************************************************************
 **
 **  Routines to access an I2C device on an Analog Devices evaluation board
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class I2cBus extends Cypress {

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int
        READ_TIMEOUT      = 1000,
        RQST_I2C          = 0xdd,
        ADDR_I2C_READ     = 0x91,
        ADDR_I2C_WRITE    = 0x90;

    private final static String
        STD_LOAD_FILE     = "/Icv_usb.hex";

   /**
    ***************************************************************************
    **
    **  Constructor
    **
    ***************************************************************************
    */
    public I2cBus(UsbComm com) throws UsbException
    {
        super(com);
    }


   /**
    ***************************************************************************
    **
    **  Performs any needed initialization
    **
    ***************************************************************************
    */
    void initialize() throws UsbException
    {
        load(false);
    }


   /**
    ***************************************************************************
    **
    **  Reads 16-bit value from a pair of registers
    **
    **  @param  regnum  The number of the first (high-order) register to read
    **
    **  @return  The 16-bit raw value
    **
    ***************************************************************************
    */
    public int readRegister2(int regnum) throws UsbException
    {
        byte[] data = new byte[2];
        readRegister(regnum, data);

        return (data[1] & 0xff) | ((data[0] & 0xff) << 8);
    }


   /**
    ***************************************************************************
    **
    **  Reads 24-bit value from a triplet of registers
    **
    **  @param  regnum  The number of the first (high-order) register to read
    **
    **  @return  The 24-bit raw value
    **
    ***************************************************************************
    */
    public int readRegister3(int regnum) throws UsbException
    {
        byte[] data = new byte[3];
        readRegister(regnum, data);

        return (data[2] & 0xff) | ((data[1] & 0xff) << 8)
                 | ((data[0] & 0xff) << 16);
    }


   /**
    ***************************************************************************
    **
    **  Reads byte array from consecutive registers
    **
    **  @param  regnum   The first register to read
    **
    **  @param  data     The array of bytes to read into
    **
    **  @param  timeout  The maximum time to wait (ms), or 0 for no timeout
    **
    **  @return  The number of bytes read.
    **
    ***************************************************************************
    */
    public int readRegister(int regnum, byte[] data) throws UsbException
    {
        return com.read(RTYP_VENDOR_READ, RQST_I2C, ADDR_I2C_READ, regnum,
                        data, READ_TIMEOUT);
    }


   /**
    ***************************************************************************
    **
    **  Reads a register
    **
    **  @param  regnum   The register to read
    **
    **  @param  timeout  The maximum time to wait (ms), or 0 for no timeout
    **
    **  @return  The value read
    **
    ***************************************************************************
    */
    public int readRegister(int regnum) throws UsbException
    {
        byte[] data = new byte[1];
        com.read(RTYP_VENDOR_READ, RQST_I2C, ADDR_I2C_READ, regnum, data,
                 READ_TIMEOUT);

        return data[0] & 0xff;
    }


   /**
    ***************************************************************************
    **
    **  Writes byte array to consecutive registers
    **
    **  @param  regnum  The first register to write
    **
    **  @param  data    The array of bytes to write
    **
    **  @return  The number of bytes written.
    **
    ***************************************************************************
    */
    public int writeRegister(int regnum, byte[] data) throws UsbException
    {
        return com.write(RTYP_VENDOR_WRITE, RQST_I2C, ADDR_I2C_WRITE, regnum,
                         data);
    }


   /**
    ***************************************************************************
    **
    **  Writes a register
    **
    **  @param  regnum  The register to write
    **
    **  @param  value   The value to write
    **
    ***************************************************************************
    */
    public void writeRegister(int regnum, int value) throws UsbException
    {
        byte data[] = {(byte)value};
        com.write(RTYP_VENDOR_WRITE, RQST_I2C, ADDR_I2C_WRITE, regnum, data);
    }


   /**
    ***************************************************************************
    **
    **  Updates a register
    **
    **  @param  regnum  The register to update
    **
    **  @param  mask    The mask of bits to update
    **
    **  @param  value   The value to write, under the mask
    **
    ***************************************************************************
    */
    public void updateRegister(int regnum, int mask, int value)
        throws UsbException
    {
        byte data[] = new byte[1];
        com.read(RTYP_VENDOR_READ, RQST_I2C, ADDR_I2C_READ, regnum, data,
                 READ_TIMEOUT);
        data[0] = (byte)((data[0] & ~mask) | (value & mask));
        com.write(RTYP_VENDOR_WRITE, RQST_I2C, ADDR_I2C_WRITE, regnum, data);
    }


   /**
    ***************************************************************************
    **
    **  Loads memory from standard file
    **
    **  @param  force  If true, force the load even if memory appears to be
    **                 already loaded correctly.
    **
    ***************************************************************************
    */
    public void load(boolean force) throws UsbException
    {
        load(STD_LOAD_FILE, force);
    }

}
