package org.lsst.ccs.drivers.ad;

import javax.usb.UsbException;
import org.lsst.ccs.drivers.usb.UsbComm;

/**
 ***************************************************************************
 **
 **  Routines to access an Analog Devices AD7747 evaluation board
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Ad7747Eval extends I2cBus {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        STATUS_REG     = 0,        // Status register
        CAP_DATA_REG   = 1,        // Capacitance data registers (3-byte)
        VT_DATA_REG    = 4,        // Voltage/temp data registers (3-byte)
        CAP_SETUP_REG  = 7,        // Capacitance setup register
        VT_SETUP_REG   = 8,        // Voltage/temperature setup register
        EXC_SETUP_REG  = 9,        // Excitation setup register
        CONFIG_REG     = 10,       // Configuration register
        CAP_DAC_A_REG  = 11,       // Capacitance DAC A register
        CAP_DAC_B_REG  = 12,       // Capacitance DAC B register
        CAP_OFFSET_REG = 13,       // Capacitance offset register
        CAP_GAIN_REG   = 15,       // Capacitance gain register
        VOLT_GAIN_REG  = 17,       // Voltage gain register

        STS_READY      = 0x04,     // Status: ready overall
        STS_RDYVT      = 0x02,     // Status: voltage/temp data ready
        STS_RDYCAP     = 0x01,     // Status: capacitance data ready

        CPSU_CAPEN     = 0x80,     // Cap setup: enable
        CPSU_CAPDIFF   = 0x20,     // Cap setup: differential input
        CPSU_STANDARD  = CPSU_CAPDIFF,  // Cal setup: standard

        VTSU_VTEN      = 0x80,     // Volt/temp setup: enable
        VTSU_VTMODE_M  = 0x60,     // Volt/temp setup: mode mask
        VTSU_VTMODE_V  = 5,        // Volt/temp setup: mode offset
        VTMD_INT_TEMP  = 0,        // Volt/temp mode: internal temperature
        VTMD_EXT_TEMP  = 1,        // Volt/temp mode: external temperature
        VTMD_VDD_MON   = 2,        // Volt/temp mode: monitor VDD
        VTMD_EXT_VOLT  = 3,        // Volt/temp mode: external voltage
        VTSU_EXTREF    = 0x10,     // Volt/temp setup: use external reference
        VTSU_VTSHORT   = 0x02,     // Volt/temp setup: input shorted
        VTSU_VTCHOP    = 0x01,     // Volt/temp setup: chopper on
        VTSU_STANDARD  = VTSU_VTCHOP,  // Volt/temp setup: standard

        EXSU_EXCDAC    = 0x08,     // Exc setup: DAC enabled
        EXSU_EXCEN     = 0x04,     // Exc setup: enabled
        EXSU_EXCLVL_M  = 0x03,     // Exc setup: level mask
        EXSU_EXCLVL_V  = 0,        // Exc setup: level offset
        EXCLVL_1_8     = 0,        // Exc level: 1/8 VDD
        EXCLVL_1_4     = 1,        // Exc level: 1/4 VDD
        EXCLVL_3_8     = 2,        // Exc level: 3/8 VDD
        EXCLVL_1_2     = 3,        // Exc level: 1/2 VDD
        EXSU_STANDARD  = EXSU_EXCDAC | EXSU_EXCEN | EXCLVL_3_8,
                                   // Exc setup: standard

        CFG_VTFS_M     = 0xc0,     // Config: volt/temp filter mask
        CFG_VTFS_V     = 6,        // Config: volt/temp filter offset
        VTFS_RATE_50   = 0,        // 49.8 Hz
        VTFS_RATE_31   = 1,        // 31.2 Hz
        VTFS_RATE_16   = 2,        // 16.1 Hz
        VTFS_RATE_8    = 3,        // 8.2 Hz
        VTFS_TIME_20   = VTFS_RATE_50,
        VTFS_TIME_32   = VTFS_RATE_31,
        VTFS_TIME_62   = VTFS_RATE_16,
        VTFS_TIME_122  = VTFS_RATE_8,
        CFG_CAPFS_M    = 0x38,     // Config: cap filter mask
        CFG_CAPFS_V    = 3,        // Config: cap filter offset
        CAPFS_RATE_45  = 0,        // 45.5 Hz
        CAPFS_RATE_42  = 1,        // 41.9 Hz
        CAPFS_RATE_25  = 2,        // 25.0 Hz
        CAPFS_RATE_13  = 3,        // 13.2 Hz
        CAPFS_RATE_8   = 4,        // 8.1 Hz
        CAPFS_RATE_7   = 5,        // 6.5 Hz
        CAPFS_RATE_6   = 6,        // 5.5 Hz
        CAPFS_RATE_5   = 7,        // 4.6 Hz
        CAPFS_TIME_22  = CAPFS_RATE_45,
        CAPFS_TIME_24  = CAPFS_RATE_42,
        CAPFS_TIME_40  = CAPFS_RATE_25,
        CAPFS_TIME_76  = CAPFS_RATE_13,
        CAPFS_TIME_124 = CAPFS_RATE_8,
        CAPFS_TIME_154 = CAPFS_RATE_7,
        CAPFS_TIME_184 = CAPFS_RATE_6,
        CAPFS_TIME_219 = CAPFS_RATE_5,
        CFG_MODE_M     = 0x07,     // Config: mode mask
        CFG_MODE_V     = 0,        // Config: mode offset
        MODE_IDLE      = 0,        // Mode: idle
        MODE_CONT      = 1,        // Mode: continuous
        MODE_SINGLE    = 2,        // Mode: single shot
        MODE_POWERDOWN = 3,        // Mode: power down
        MODE_OFFS_CAL  = 5,        // Mode: offset calibration
        MODE_GAIN_CAL  = 6,        // Mode: gain calibration

        DACA_ENABLE    = 0x80,
        DACA_VALUE_M   = 0x3f,

        DACB_ENABLE    = 0x80,
        DACB_VALUE_M   = 0x3f,

        RDO_CONFIG     = 0x01,     // Read option: configure VT type
        RDO_EXTERN     = 0x02,     // Read option: external temp or voltage
        RDO_SINGLE     = 0x04,     // Read option: set single mode
        RDO_IMMED      = 0x08;     // Read option: don't wait for ready

    public final static float
        HALF_RANGE3     = 8388608F,
        REF_VOLTAGE     = 1.17F,
        CAP_FULL_SCALE  = 8.192F,
        TEMP_FULL_SCALE = 4096F,
        INT_VOLT_SCALE  = 6F;

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int
        AD_VID            = 0x0456,
        AD_7747_PID       = 0xb481;

   /**
    ***************************************************************************
    **
    **  Constructors
    **
    ***************************************************************************
    */
    public Ad7747Eval() throws UsbException
    {
        this(0);
    }

    public Ad7747Eval(int index) throws UsbException
    {
        /*
        **  Initialize parent, locate the board, load firmware
        */
        super(new UsbComm());
        if (com.findDevice(AD_VID, AD_7747_PID, index) == null) {
            throw new AdException("Board not found");
        }
        initialize();
    }


   /**
    ***************************************************************************
    **
    **  Performs standard board setup
    **
    **  <p>
    **  The three setup register are written with capacitance and voltage
    **  channels disabled, and all other bits set to the values mandated for
    **  correct operation.  The configuration register is set to idle mode
    **  and the maximum conversion rates for the data channels.
    **
    ***************************************************************************
    */
    public void setupStandard() throws UsbException
    {
        writeRegister(CAP_SETUP_REG, CPSU_STANDARD);
        writeRegister(EXC_SETUP_REG, EXSU_STANDARD);
        writeRegister(VT_SETUP_REG, VTSU_STANDARD);
        writeRegister(CONFIG_REG, 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the board status
    **
    ***************************************************************************
    */
    public int getStatus() throws UsbException
    {
        return readRegister(STATUS_REG);
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables the capacitance channel
    **
    **  @param  set  If true, the channel is enabled; otherwise disabled
    **
    ***************************************************************************
    */
    public void setCapEnabled(boolean set) throws UsbException
    {
        updateRegister(CAP_SETUP_REG, CPSU_CAPEN, set ? CPSU_CAPEN : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the enabled state of the capacitance channel
    **
    **  @return  The enabled state
    **
    ***************************************************************************
    */
    public boolean isCapEnabled() throws UsbException
    {
        return (readRegister(CAP_SETUP_REG) & CPSU_CAPEN) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables the capacitance excitation
    **
    **  @param  set  If true, excitation is enabled; otherwise disabled
    **
    ***************************************************************************
    */
    public void setExcEnabled(boolean set) throws UsbException
    {
        updateRegister(EXC_SETUP_REG, EXSU_EXCEN, set ? EXSU_EXCEN : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the enabled state of the capacitance excitation
    **
    **  @return  The enabled state
    **
    ***************************************************************************
    */
    public boolean isExcEnabled() throws UsbException
    {
        return (readRegister(EXC_SETUP_REG) & EXSU_EXCEN) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables the voltage/temperature channel
    **
    **  @param  set  If true, the channel is enabled; otherwise disabled
    **
    ***************************************************************************
    */
    public void setVtEnabled(boolean set) throws UsbException
    {
        updateRegister(VT_SETUP_REG, VTSU_VTEN, set ? VTSU_VTEN : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the enabled state of the voltage/temperature channel
    **
    **  @return  The enabled state
    **
    ***************************************************************************
    */
    public boolean isVtEnabled() throws UsbException
    {
        return (readRegister(VT_SETUP_REG) & VTSU_VTEN) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the mode of the voltage/temperature channel
    **
    **  @param  mode  The mode value to set
    **
    ***************************************************************************
    */
    public void setVtMode(int mode) throws UsbException
    {
        updateRegister(VT_SETUP_REG, VTSU_VTMODE_M, mode << VTSU_VTMODE_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the mode of the voltage/temperature channel
    **
    **  @return  The mode value
    **
    ***************************************************************************
    */
    public int getVtMode() throws UsbException
    {
        return (readRegister(VT_SETUP_REG) & VTSU_VTMODE_M) >> VTSU_VTMODE_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets or clears the shorted state of the voltage/temperature channel
    **
    **  @param  set  If true, shorted state is set; otherwise cleared
    **
    ***************************************************************************
    */
    public void setVtShorted(boolean set) throws UsbException
    {
        updateRegister(VT_SETUP_REG, VTSU_VTSHORT, set ? VTSU_VTSHORT : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the shorted state of the voltage/temperature channel
    **
    **  @return  The shorted state
    **
    ***************************************************************************
    */
    public boolean isVtShorted() throws UsbException
    {
        return (readRegister(VT_SETUP_REG) & VTSU_VTSHORT) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets or clears the use of the external reference voltage
    **
    **  @param  set  If true, external reference is used; otherwise internal
    **
    ***************************************************************************
    */
    public void setExternalRef(boolean set) throws UsbException
    {
        updateRegister(VT_SETUP_REG, VTSU_EXTREF, set ? VTSU_EXTREF : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets whether the external reference voltage is being used
    **
    **  @return  The external reference state
    **
    ***************************************************************************
    */
    public boolean isExternalRef() throws UsbException
    {
        return (readRegister(VT_SETUP_REG) & VTSU_EXTREF) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the voltage/temperature conversion rate
    **
    **  @param  rate  The rate value to set
    **
    ***************************************************************************
    */
    public void setVtConvRate(int rate) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_VTFS_M, rate << CFG_VTFS_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the voltage/temperature conversion rate
    **
    **  @return  The conversion rate
    **
    ***************************************************************************
    */
    public int getVtConvRate() throws UsbException
    {
        return (readRegister(CONFIG_REG) & CFG_VTFS_M) >> CFG_VTFS_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the capacitance conversion rate
    **
    **  @param  rate  The rate value to set
    **
    ***************************************************************************
    */
    public void setCapConvRate(int rate) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_CAPFS_M, rate << CFG_CAPFS_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the capacitance conversion rate
    **
    **  @return  The conversion rate
    **
    ***************************************************************************
    */
    public int getCapConvRate() throws UsbException
    {
        return (readRegister(CONFIG_REG) & CFG_CAPFS_M) >> CFG_CAPFS_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the conversion mode
    **
    **  @param  mode  The mode value to set
    **
    ***************************************************************************
    */
    public void setConvMode(int mode) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_MODE_M, mode);
    }


   /**
    ***************************************************************************
    **
    **  Gets the conversion mode
    **
    **  @return  The mode value
    **
    ***************************************************************************
    */
    public int getConvMode() throws UsbException
    {
        return readRegister(CONFIG_REG) & CFG_MODE_M;
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables capacitance DAC A
    **
    **  @param  set  If true, capacitance DAC A is enabled; otherwise disabled
    **
    ***************************************************************************
    */
    public void setDacAEnabled(boolean set) throws UsbException
    {
        updateRegister(CAP_DAC_A_REG, DACA_ENABLE, set ? DACA_ENABLE : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets capacitance DAC A enabled state
    **
    **  @return  The enabled state
    **
    ***************************************************************************
    */
    public boolean isDacAEnabled() throws UsbException
    {
        return (readRegister(CAP_DAC_A_REG) & DACA_ENABLE) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets capacitance DAC A value
    **
    **  @param  value  The DAC value to set
    **
    ***************************************************************************
    */
    public void setDacAValue(int value) throws UsbException
    {
        updateRegister(CAP_DAC_A_REG, DACA_VALUE_M, value);
    }


   /**
    ***************************************************************************
    **
    **  Gets capacitance DAC A value
    **
    **  @return  The DAC value
    **
    ***************************************************************************
    */
    public int getDacAValue() throws UsbException
    {
        return readRegister(CAP_DAC_A_REG) & DACA_VALUE_M;
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables capacitance DAC B
    **
    **  @param  set  If true, capacitance DAC B is enabled; otherwise disabled
    **
    ***************************************************************************
    */
    public void setDacBEnabled(boolean set) throws UsbException
    {
        updateRegister(CAP_DAC_B_REG, DACB_ENABLE, set ? DACB_ENABLE : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets capacitance DAC B enabled state
    **
    **  @return  The enabled state
    **
    ***************************************************************************
    */
    public boolean isDacBEnabled() throws UsbException
    {
        return (readRegister(CAP_DAC_B_REG) & DACB_ENABLE) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets capacitance DAC B value
    **
    **  @param  value  The DAC value to set
    **
    ***************************************************************************
    */
    public void setDacBValue(int value) throws UsbException
    {
        updateRegister(CAP_DAC_B_REG, DACB_VALUE_M, value);
    }


   /**
    ***************************************************************************
    **
    **  Gets capacitance DAC B value
    **
    **  @return  The DAC value
    **
    ***************************************************************************
    */
    public int getDacBValue() throws UsbException
    {
        return readRegister(CAP_DAC_B_REG) & DACB_VALUE_M;
    }


   /**
    ***************************************************************************
    **
    **  Reads capacitance sensor
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 RDO_SINGLE: First configures for a single conversion.
    **
    **                 RDO_IMMED:  Reads the data registers immediately
    **                             instead of waiting for a conversion to
    **                             complete.
    **
    **  @return  The read capacitance (pF)
    **
    ***************************************************************************
    */
    public float readCapacitance(int optns) throws UsbException
    {
        if ((optns & RDO_SINGLE) != 0) {
            updateRegister(CONFIG_REG, CFG_MODE_M, MODE_SINGLE);
        }
        if ((optns & RDO_IMMED) == 0) {
            waitReady(STS_RDYCAP);
        }

        return (readRegister3(CAP_DATA_REG) / HALF_RANGE3 - 1F)
                 * CAP_FULL_SCALE;
    }


   /**
    ***************************************************************************
    **
    **  Reads temperature sensor
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 RDO_CONFIG: Configures to read either the internal or
    **                             external temperature sensor.
    **
    **                 RDO_EXTERN: If configuring, selects the external sensor
    **                             instead of the internal one.
    **
    **                 RDO_SINGLE: First configures for a single conversion.
    **
    **                 RDO_IMMED:  Reads the data registers immediately
    **                             instead of waiting for a conversion to
    **                             complete.
    **
    **  @return  The read temperature (Celsius)
    **
    ***************************************************************************
    */
    public float readTemperature(int optns) throws UsbException
    {
        if ((optns & RDO_CONFIG) != 0) {
            updateRegister(VT_SETUP_REG, VTSU_VTMODE_M,
                           (((optns & RDO_EXTERN) != 0) ? VTMD_EXT_TEMP
                               : VTMD_INT_TEMP) << VTSU_VTMODE_V);
        }
        if ((optns & RDO_SINGLE) != 0) {
            updateRegister(CONFIG_REG, CFG_MODE_M, MODE_SINGLE);
        }
        if ((optns & RDO_IMMED) == 0) {
            waitReady(STS_RDYVT);
        }

        return (readRegister3(VT_DATA_REG) / HALF_RANGE3 - 1F)
                 * TEMP_FULL_SCALE;
    }


   /**
    ***************************************************************************
    **
    **  Reads voltage
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 RDO_CONFIG: Configures to read either the internal or
    **                             external temperature sensor.
    **
    **                 RDO_EXTERN: Selects the external sensor instead of the
    **                             internal one (for both configuring and
    **                             conversion).
    **
    **                 RDO_SINGLE: First configures for a single conversion.
    **
    **                 RDO_IMMED:  Reads the data registers immediately
    **                             instead of waiting for a conversion to
    **                             complete.
    **
    **  @return  The read voltage
    **
    ***************************************************************************
    */
    public float readVoltage(int optns) throws UsbException
    {
        boolean ext = (optns & RDO_EXTERN) != 0;
        if ((optns & RDO_CONFIG) != 0) {
            updateRegister(VT_SETUP_REG, VTSU_VTMODE_M,
                           (ext ? VTMD_EXT_VOLT : VTMD_VDD_MON)
                             << VTSU_VTMODE_V);
        }
        if ((optns & RDO_SINGLE) != 0) {
            updateRegister(CONFIG_REG, CFG_MODE_M, MODE_SINGLE);
        }
        if ((optns & RDO_IMMED) == 0) {
            waitReady(STS_RDYVT);
        }
        float volts = (readRegister3(VT_DATA_REG) / HALF_RANGE3 - 1F)
                        * REF_VOLTAGE;
        if (!ext) {
            volts *= INT_VOLT_SCALE;
        }

        return volts;
    }


   /**
    ***************************************************************************
    **
    **  Waits for the board to become ready after a conversion
    **
    **  @param  mask  Mask of bits that must be clear
    **
    ***************************************************************************
    */
    private void waitReady(int mask) throws UsbException
    {
        long start = System.currentTimeMillis();
        boolean ready = false;
        while (!(ready = (getStatus() & mask) == 0)
                 && System.currentTimeMillis() - start <= 5000) {
            try {
                Thread.sleep(5);
            }
            catch (InterruptedException e) {
            }
        }
        if (!ready) {
            throw new AdException("Conversion timed out");
        }
    }

}
