package org.lsst.ccs.drivers.ad;

import java.util.ArrayList;
import javax.usb.UsbException;
import jline.console.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProcess;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to test an Analog Devices 7747 evaluation board
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestAd7747 implements CmndProcess.Dispatch {

    /*
    **  Command codes
    */
    private final static int
        CMD_READMEM     = 0,
        CMD_WRITEMEM    = 1,
        CMD_LOAD        = 2,
        CMD_RESET       = 3,
        CMD_READREG     = 4,
        CMD_WRITEREG    = 5,
        CMD_READCAP     = 6,
        CMD_READTEMP    = 7,
        CMD_READVOLT    = 8,
        CMD_READ2REG    = 9,
        CMD_READ3REG    = 10,
        CMD_SETCAP      = 11,
        CMD_SHOWCAP     = 12,
        CMD_SETVT       = 13,
        CMD_SHOWVT      = 14,
        CMD_SETCONV     = 15,
        CMD_SHOWCONV    = 16,
        CMD_SETDAC      = 17,
        CMD_SHOWDAC     = 18,
        CMD_INIT        = 19,
        CMD_SHOWREGS    = 20,
        NUM_CMDS        = 21;

    /*
    **  Command help text
    */
    private final static String[] helpInit = {
        "Initializes setup registers to standard values",
        "init",
    };

    private final static String[] helpSetcap = {
        "Set capacitance channel configuration",
        "setcap [<enable>] [<rate>]",
        "enable  If non-zero, enable capacitance channel; otherwise disable it",
        "rate    The encoded capacitance data conversion rate (0 - 7)",
    };

    private final static String[] helpShowcap = {
        "Display capacitance channel configuration",
        "showcap",
    };

    private final static String[] helpSetvt = {
        "Set voltage/temperature channel configuration",
        "setvt [<enable>] [<mode>] [<rate>] [<short>] [<extref>]",
        "enable  If non-zero, enable volt/temp channel; otherwise disable it",
        "mode    The encoded volt/temp channel mode (0 - 3)",
        "rate    The encoded volt/temp data conversion rate (0 - 3)",
        "short   If non-zero, short the input; otherwise open it",
        "extref  If non-zero, use external ref voltage; otherwise internal",
    };

    private final static String[] helpShowvt = {
        "Display voltage/temperature channel configuration",
        "showvt",
    };

    private final static String[] helpSetconv = {
        "Set conversion mode",
        "setconv <value>",
        "value  The encoded conversion mode (0 - 7)",
    };

    private final static String[] helpShowconv = {
        "Display conversion mode",
        "showconv",
    };

    private final static String[] helpSetdac = {
        "Set capacitance DAC state and value",
        "setdac <dac> [<enable>] [<value>]",
        "dac     If zero, use DAC A; otherwise DAC B",
        "enable  If non-zero enable the DAC; otherwise disable it",
        "value   The DAC value",
    };

    private final static String[] helpShowdac = {
        "Display DAC configuration",
        "showdac",
    };

    private final static String[] helpReadcap = {
        "Read the capacitance sensor and display the value",
        "readcap [<single>] [<immed>] [<count>]",
        "single  If present and non-zero, perform a single conversion",
        "immed   If present and non-zero, don't wait for conversion",
        "count   The number of reads to average (default 1)",
    };

    private final static String[] helpReadtemp = {
        "Read the internal or external temperature sensor and display the value",
        "readtemp [<extcfg>] [<extern>] [<single>] [<immed>] [<count>]",
        "extcfg  If present and non-zero, configure which sensor to read",
        "extern  If present and non-zero, read the external sensor",
        "single  If present and non-zero, perform a single conversion",
        "immed   If present and non-zero, don't wait for conversion",
        "count   The number of reads to average (default 1)",
    };

    private final static String[] helpReadvolt = {
        "Read the internal or external voltage and display the value",
        "readvolt [<extcfg>] [<extern>] [<single>] [<immed>] [<count>]",
        "extcfg  If present and non-zero, configure which voltage to read",
        "extern  If present and non-zero, read the external voltage",
        "single  If present and non-zero, perform a single conversion",
        "immed   If present and non-zero, don't wait for conversion",
        "count   The number of reads to average (default 1)",
    };

    private final static String[] helpReadmem = {
        "Read data from memory and display it",
        "readmem <addr> <leng> [<timwout>]",
        "addr     The address to read from",
        "leng     The number of bytes to read",
        "timeout  The timeout value (secs, default 1.0)",
    };

    private final static String[] helpWritemem = {
        "Write data to memory",
        "writemem <addr> [<data1>]... [<data8>]",
        "addr   The address to write to",
        "datan  String of (an even number of) hexadecimal digits to write",
    };

    private final static String[] helpShowregs = {
        "Show formatted display of all registers",
        "showregs",
    };

    private final static String[] helpReadreg = {
        "Read register(s) and display the contents",
        "readreg <regnum> [<count>]",
        "regnum   The number of the first register to read",
        "count    The number of registers to read (default: 1)",
    };

    private final static String[] helpWritereg = {
        "Write to register(s)",
        "writereg <regnum> <value1> [<value2>]... [<value8>]",
        "regnum  The number of the first register to write",
        "valuen  Values to write to consecutive registers",
    };

    private final static String[] helpRead2reg = {
        "Read two consecutive registers and display the resultant value",
        "readreg <regnum>",
        "regnum   The number of the first register to read",
    };

    private final static String[] helpRead3reg = {
        "Read three consecutive registers and display the resultant value",
        "readreg <regnum>",
        "regnum   The number of the first register to read",
    };

    private final static String[] helpReset = {
        "Set or clear the RESET state",
        "reset  <opern>",
        "opern  If non-zero, RESET is set; if zero, RESET is cleared",
    };

    private final static String[] helpLoad = {
        "Load memory from a file",
        "load  [<file>] [<force>]",
        "file   The name of the .hex file to load (default: standard)",
        "force  If present and non-zero, do the load even if unnecessary",
    };

    /*
    **  Lookup tables
    */
    private final static CmndProcess.Lookup capRateNames;
    static {
        capRateNames = new CmndProcess.Lookup(8);
        capRateNames.add("45.5", Ad7747Eval.CAPFS_RATE_45);
        capRateNames.add("41.9", Ad7747Eval.CAPFS_RATE_42);
        capRateNames.add("25.0", Ad7747Eval.CAPFS_RATE_25);
        capRateNames.add("13.2", Ad7747Eval.CAPFS_RATE_13);
        capRateNames.add("8.1",  Ad7747Eval.CAPFS_RATE_8);
        capRateNames.add("6.5",  Ad7747Eval.CAPFS_RATE_7);
        capRateNames.add("5.5",  Ad7747Eval.CAPFS_RATE_6);
        capRateNames.add("4.6",  Ad7747Eval.CAPFS_RATE_5);
    }
    private final static CmndProcess.Lookup vtRateNames;
    static {
        vtRateNames = new CmndProcess.Lookup(4);
        vtRateNames.add("49.8", Ad7747Eval.VTFS_RATE_50);
        vtRateNames.add("31.2", Ad7747Eval.VTFS_RATE_31);
        vtRateNames.add("16.1", Ad7747Eval.VTFS_RATE_16);
        vtRateNames.add("8.2",  Ad7747Eval.VTFS_RATE_8);
    }
    private final static CmndProcess.Lookup vtModeNames;
    static {
        vtModeNames = new CmndProcess.Lookup(4);
        vtModeNames.add("temperature", Ad7747Eval.VTMD_INT_TEMP);
        vtModeNames.add("exttemp",     Ad7747Eval.VTMD_EXT_TEMP);
        vtModeNames.add("vdd",         Ad7747Eval.VTMD_VDD_MON);
        vtModeNames.add("voltage",     Ad7747Eval.VTMD_EXT_VOLT);
    }
    private final static CmndProcess.Lookup cnvModeNames;
    static {
        cnvModeNames = new CmndProcess.Lookup(6);
        cnvModeNames.add("idle",       Ad7747Eval.MODE_IDLE);
        cnvModeNames.add("continuous", Ad7747Eval.MODE_CONT);
        cnvModeNames.add("single",     Ad7747Eval.MODE_SINGLE);
        cnvModeNames.add("powerdown",  Ad7747Eval.MODE_POWERDOWN);
        cnvModeNames.add("offsetcal",  Ad7747Eval.MODE_OFFS_CAL);
        cnvModeNames.add("gaincal",    Ad7747Eval.MODE_GAIN_CAL);
    }

    /*
    **  Command table
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("readcap",  CMD_READCAP,  helpReadcap,  "iii");
        cmnd.add("readtemp", CMD_READTEMP, helpReadtemp, "iiiii");
        cmnd.add("readvolt", CMD_READVOLT, helpReadvolt, "iiiii");
        cmnd.add("readmem",  CMD_READMEM,  helpReadmem,  "IIf");
        cmnd.add("writemem", CMD_WRITEMEM, helpWritemem, "Issssssss");
        cmnd.add("showregs", CMD_SHOWREGS, helpShowregs, "");
        cmnd.add("readreg",  CMD_READREG,  helpReadreg,  "Ii");
        cmnd.add("writereg", CMD_WRITEREG, helpWritereg, "Issssssss");
        cmnd.add("read2reg", CMD_READ2REG, helpRead2reg, "I");
        cmnd.add("read3reg", CMD_READ3REG, helpRead3reg, "I");
        cmnd.add("init",     CMD_INIT,     helpInit,     "");
        cmnd.add("setcap",   CMD_SETCAP,   helpSetcap,   "ij",
                 capRateNames);
        cmnd.add("setvt",    CMD_SETVT,    helpSetvt,    "ijjii",
                 vtModeNames, vtRateNames);
        cmnd.add("setconv",  CMD_SETCONV,  helpSetconv,  "J",
                 cnvModeNames);
        cmnd.add("setdac",   CMD_SETDAC,   helpSetdac,   "Iii");
        cmnd.add("showcap",  CMD_SHOWCAP,  helpShowcap,  "");
        cmnd.add("showvt",   CMD_SHOWVT,   helpShowvt,   "");
        cmnd.add("showconv", CMD_SHOWCONV, helpShowconv, "");
        cmnd.add("showdac",  CMD_SHOWDAC,  helpShowdac,  "");
        cmnd.add("reset",    CMD_RESET,    helpReset,    "I");
        cmnd.add("load",     CMD_LOAD,     helpLoad,     "si");
    }

    /*
    **  Private fields
    */
    private final static Output out = new ConsOut();
    private final CmndProcess proc = new CmndProcess();
    private final boolean debug;
    private Ad7747Eval ad;


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public TestAd7747(boolean debug)
    {
        this.debug = debug;
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        int index = (args.length >= 1) ? Integer.decode(args[0]) : 0;
        (new TestAd7747(args.length >= 2)).run(index);
    }


   /**
    ***************************************************************************
    **
    **  Run the test
    **
    ***************************************************************************
    */
    public void run(int index)
    {
        proc.add(this, cmnd);

        try {
            ad = new Ad7747Eval(index);

            ConsoleReader reader = new ConsoleReader();
            while (true) {
                String line = reader.readLine(">> ");
                if (line == null || !proc.process(line)) break;
            }
        }
        catch (Exception e) {
            showException(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatches command for processing
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        try {
            switch (code) {
            case CMD_INIT:
                procInit(found, args); break;
            case CMD_SETCAP:
                procSetCap(found, args); break;
            case CMD_SETVT:
                procSetVt(found, args); break;
            case CMD_SETCONV:
                procSetConv(found, args); break;
            case CMD_SETDAC:
                procSetDac(found, args); break;
            case CMD_SHOWCAP:
                procShowCap(found, args); break;
            case CMD_SHOWVT:
                procShowVt(found, args); break;
            case CMD_SHOWCONV:
                procShowConv(found, args); break;
            case CMD_SHOWDAC:
                procShowDac(found, args); break;
            case CMD_READCAP:
                procReadCap(found, args); break;
            case CMD_READTEMP:
                procReadTemp(found, args); break;
            case CMD_READVOLT:
                procReadVolt(found, args); break;
            case CMD_SHOWREGS:
                procShowRegs(found, args); break;
            case CMD_READREG:
                procReadReg(found, args); break;
            case CMD_READ2REG:
                procRead2Reg(found, args); break;
            case CMD_READ3REG:
                procRead3Reg(found, args); break;
            case CMD_WRITEREG:
                procWriteReg(found, args); break;
            case CMD_RESET:
                procReset(found, args); break;
            case CMD_READMEM:
                procReadMem(found, args); break;
            case CMD_WRITEMEM:
                procWriteMem(found, args); break;
            case CMD_LOAD:
                procLoad(found, args); break;
            default:
                out.println("Command not fully implemented");
            }
        }
        catch (UsbException e) {
            showException(e);
        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the INIT command
    **
    ***************************************************************************
    */
    private void procInit(int found, Object[] args) throws UsbException
    {
        ad.setupStandard();
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETCAP command
    **
    ***************************************************************************
    */
    private void procSetCap(int found, Object[] args) throws UsbException
    {
        if ((found & 0x01) != 0) {
            ad.setCapEnabled((Integer)args[0] != 0);
        }
        if ((found & 0x02) != 0) {
            ad.setCapConvRate((Integer)args[1]);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETVT command
    **
    ***************************************************************************
    */
    private void procSetVt(int found, Object[] args) throws UsbException
    {
        if ((found & 0x01) != 0) {
            ad.setVtEnabled((Integer)args[0] != 0);
        }
        if ((found & 0x02) != 0) {
            ad.setVtMode((Integer)args[1]);
        }
        if ((found & 0x04) != 0) {
            ad.setVtConvRate((Integer)args[2]);
        }
        if ((found & 0x08) != 0) {
            ad.setVtShorted((Integer)args[3] != 0);
        }
        if ((found & 0x10) != 0) {
            ad.setExternalRef((Integer)args[4] != 0);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETCONV command
    **
    ***************************************************************************
    */
    private void procSetConv(int found, Object[] args) throws UsbException
    {
        ad.setConvMode((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETDAC command
    **
    ***************************************************************************
    */
    private void procSetDac(int found, Object[] args) throws UsbException
    {
        int dac = (Integer)args[0];
        if ((found & 0x02) != 0) {
            if (dac == 0) {
                ad.setDacAEnabled((Integer)args[1] != 0);
            }
            else {
                ad.setDacBEnabled((Integer)args[1] != 0);
            }
        }
        if ((found & 0x04) != 0) {
            if (dac == 0) {
                ad.setDacAValue((Integer)args[2]);
            }
            else {
                ad.setDacBValue((Integer)args[2]);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWCAP command
    **
    ***************************************************************************
    */
    private void procShowCap(int found, Object[] args) throws UsbException
    {
        int rate = ad.getCapConvRate();
        out.format("%s, conv. rate = %s (%s Hz)\n",
                   ad.isCapEnabled() ? "Enabled" : "Disabled",
                   rate, capRateNames.decode(rate));
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWVT command
    **
    ***************************************************************************
    */
    private void procShowVt(int found, Object[] args) throws UsbException
    {
        int mode = ad.getVtMode();
        int rate = ad.getVtConvRate();
        out.format("%s, mode = %s (%s), conv. rate = %s (%s Hz)"
                     + "%s%s\n",
                   ad.isVtEnabled() ? "Enabled" : "Disabled",
                   mode, vtModeNames.decode(mode),
                   rate, vtRateNames.decode(rate),
                   ad.isVtShorted() ? ", shorted" : "",
                   ad.isExternalRef() ? ", external ref" : "");
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWCONV command
    **
    ***************************************************************************
    */
    private void procShowConv(int found, Object[] args) throws UsbException
    {
        int mode = ad.getConvMode();
        out.format("Conversion mode = %s (%s)\n", mode,
                   cnvModeNames.decode(mode));
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWDAC command
    **
    ***************************************************************************
    */
    private void procShowDac(int found, Object[] args) throws UsbException
    {
        out.format("DAC A: %s, value = %s; DAC B: %s, value = %s\n",
                   ad.isDacAEnabled() ? "Enabled" : "Disabled",
                   ad.getDacAValue(),
                   ad.isDacBEnabled() ? "Enabled" : "Disabled",
                   ad.getDacBValue());
    }


   /**
    ***************************************************************************
    **
    **  Processes the READCAP command
    **
    ***************************************************************************
    */
    private void procReadCap(int found, Object[] args) throws UsbException
    {
        int[] params = getParams(-2, found, args);
        double sum = 0, sumsq = 0;
        for (int j = 0; j < params[1]; j++) {
            double value = ad.readCapacitance(params[0]);
            sum += value;
            sumsq += value * value;
        }
        showStats("Capacitance", params[1], sum, sumsq);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READTEMP command
    **
    ***************************************************************************
    */
    private void procReadTemp(int found, Object[] args) throws UsbException
    {
        int[] params = getParams(0, found, args);
        double sum = 0, sumsq = 0;
        for (int j = 0; j < params[1]; j++) {
            double value = ad.readTemperature(params[0]);
            sum += value;
            sumsq += value * value;
        }
        showStats("Temperature", params[1], sum, sumsq);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READVOLT command
    **
    ***************************************************************************
    */
    private void procReadVolt(int found, Object[] args) throws UsbException
    {
        int[] params = getParams(0, found, args);
        double sum = 0, sumsq = 0;
        for (int j = 0; j < params[1]; j++) {
            double value = ad.readVoltage(params[0]);
            sum += value;
            sumsq += value * value;
        }
        showStats("Voltage", params[1], sum, sumsq);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWREGS command
    **
    ***************************************************************************
    */
    private static final int[] regSize = {1, 3, 3, 1, 1, 1, 1, 1, 1, 2, 2, 2};
    private static final String[] regName = {"STAT", "CPDT", "VTDT", "CPSU",
                                             "VTSU", "EXSU", "CONF", "DACA",
                                             "DACB", "COFF", "CPGN", "VTGN"};
    private final static String[]
        regDesc  = {"|  --  |  --  |  --  |  --  |  --  | ~RDY |~RDYVT|~RDYCP|",
                    "",
                    "",
                    "|CAPENA|  --  |CAPDIF|  --  |  --  |  --  |  --  |  --  |",
                    "| VTENA|VTMOD1|VTMOD0|EXTREF|  --  |  --  | SHORT| CHOP |",
                    "|  --  |  --  |  --  |  --  |EXCDAC|EXCENA|EXVLV1|EXCLV0|",
                    "| VTFS1| VTFS0|CAPFS2|CAPFS1|CAPFS0| MODE2| MODE1| MODE0|",
                    "|DAENAB|DAVAL6|DAVAL5|DAVAL4|DAVAL3|DAVAL2|DAVAL1|DAVAL0|",
                    "|DBENAB|DBVAL6|DBVAL5|DBVAL4|DBVAL3|DBVAL2|DBVAL1|DBVAL0|",
                    "",
                    "",
                    ""};
    
    private void procShowRegs(int found, Object[] args) throws UsbException
    {
        int reg = 0;
        for (int j = 0; j < regSize.length; j++) {
            int size = regSize[j];
            out.format("%2s (%s): ", reg, regName[j]);
            for (int k = size; k < 3; k++) {
                out.print("  ");
            }
            for (int k = 0; k < size; k++) {
                out.format("%02x", ad.readRegister(reg++));
            }
            out.println("  " + regDesc[j]);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the READREG command
    **
    ***************************************************************************
    */
    private void procReadReg(int found, Object[] args) throws UsbException
    {
        int regnum = (Integer)args[0];
        if ((found & 0x02) != 0) {
            byte[] data = new byte[(Integer)args[1]];
            dispRegs(regnum, data, ad.readRegister(regnum, data));
        }
        else {
            out.format("%02x: %02x\n", regnum, ad.readRegister(regnum));
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the READ2REG command
    **
    ***************************************************************************
    */
    private void procRead2Reg(int found, Object[] args) throws UsbException
    {
        out.format("Value = %d (0x%1$04x)\n",
                   ad.readRegister2((Integer)args[0]));
    }


   /**
    ***************************************************************************
    **
    **  Processes the READ3REG command
    **
    ***************************************************************************
    */
    private void procRead3Reg(int found, Object[] args) throws UsbException
    {
        out.format("Value = %d (0x%1$04x)\n",
                   ad.readRegister3((Integer)args[0]));
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITEREG command
    **
    ***************************************************************************
    */
    private void procWriteReg(int found, Object[] args) throws UsbException
    {
        int regnum = (Integer)args[0], count = 0, value = -1;
        for (int j = 1; j <= 8; j++) {
            if ((found & (1 << j)) != 0) {
                String number = (String)args[j];
                count += (number.length() + 1) / 2;
                if (count == 1) {
                    try {
                        value = Integer.parseInt(number, 16);
                    }
                    catch (NumberFormatException e) {
                        out.println("Invalid hexadecimal string");
                    }
                }
            }
        }
        if (count == 1) {
            if (value == -1) return;
            ad.writeRegister(regnum, value);
        }
        else {
            byte[] data = genData(found & 0x1fe, args);
            ad.writeMemory((Integer)args[0], data);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the RESET command
    **
    ***************************************************************************
    */
    private void procReset(int found, Object[] args) throws UsbException
    {
        ad.setReset((Integer)args[0] != 0);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READMEM command
    **
    ***************************************************************************
    */
    private void procReadMem(int found, Object[] args) throws UsbException
    {
        int addr = (Integer)args[0];
        float tmo = 1000F * ((found & 0x04) != 0 ? (Float)args[2] : 1F);
        byte[] data = new byte[(Integer)args[1]];
        dispData(addr, data, ad.readMemory(addr, data, (int)tmo));
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITEMEM command
    **
    ***************************************************************************
    */
    private void procWriteMem(int found, Object[] args) throws UsbException
    {
        ad.writeMemory((Integer)args[0], genData(found & 0x1fe, args));
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOAD command
    **
    ***************************************************************************
    */
    private void procLoad(int found, Object[] args) throws UsbException
    {
        boolean force = false;
        if ((found & 0x02) != 0 && (Integer)args[1] != 0) {
            force = true;
        }
        if ((found & 0x01) == 0) {
            ad.load(force);
        }
        else {
            ad.load((String)args[0], force);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets parameters for ADC read commands
    **
    ***************************************************************************
    */
    private static int[] getParams(int offs, int found, Object[] args)
    {
        int[] params = {0, 1};
        if ((found & (1 << offs)) != 0 && (Integer)args[offs] != 0) {
            params[0] |= Ad7747Eval.OPTN_EXTCFG;
        }
        if ((found & (1 << ++offs)) != 0 && (Integer)args[offs] != 0) {
            params[0] |= Ad7747Eval.OPTN_EXTERN;
        }
        if ((found & (1 << ++offs)) != 0 && (Integer)args[offs] != 0) {
            params[0] |= Ad7747Eval.OPTN_SINGLE;
        }
        if ((found & (1 << ++offs)) != 0 && (Integer)args[offs] != 0) {
            params[0] |= Ad7747Eval.OPTN_IMMED;
        }
        if ((found & (1 << ++offs)) != 0) {
            params[1] = (Integer)args[offs];
        }

        return params;
    }


   /**
    ***************************************************************************
    **
    **  Displays mean and sigma values
    **
    ***************************************************************************
    */
    private static void showStats(String name, int count, double sum,
                                  double sumsq)
    {
        double mean = sum / count;
        double sigsq = sumsq / count - mean * mean;
        double sigma = sigsq <= 0 ? 0 : Math.sqrt(sigsq);
        out.format("%s: mean = %.7g, sigma = %.7g\n", name, mean, sigma);
    }


   /**
    ***************************************************************************
    **
    **  Displays register values
    **
    ***************************************************************************
    */
    private static void dispRegs(int regnum, byte[] data, int count)
    {
        for (int j = 0; j < count; j++) {
            if ((j & 0x0f) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%02x:", regnum + j);
            }
            out.format(" %02x", data[j] & 0xff);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Displays memory data
    **
    ***************************************************************************
    */
    private static void dispData(int addr, byte[] data, int count)
    {
        for (int j = 0; j < count; j++) {
            if ((j & 0x0f) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%04x:", addr + j);
            }
            out.format(" %02x", data[j] & 0xff);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Generates write data from hexadecimal string arguments
    **
    ***************************************************************************
    */
    private static byte[] genData(int found, Object[] args)
    {
        int count = 0, mask = found;
        ArrayList list = new ArrayList();
        for (int j = 0; found != 0; found >>= 1, j++) {
            if ((found & 1) == 0) continue;
            String hex = (String)args[j];
            int leng = hex.length();
            if ((leng & 1) != 0) break;
            byte[] sData = new byte[leng / 2];
            try {
                for (int k = 0; k < leng; k += 2) {
                    String num = hex.substring(k, k + 2);
                    sData[k / 2] = (byte)Integer.parseInt(num, 16);
                }
            }
            catch (NumberFormatException e) {
                break;
            }
            count += leng / 2;
            list.add(sData);
            mask ^= (1 << j);
        }
        if (mask != 0) {
            out.println("Invalid hexadecimal string");
            return null;
        }
        byte[] data = new byte[count];
        int l = 0;
        for (int j = 0; j < list.size(); j++) {
            byte[] sData = (byte [])list.get(j);
            for (int k = 0; k < sData.length; k++) {
                data[l++] = sData[k];
            }
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Displays an exception
    **
    ***************************************************************************
    */
    private void showException(Exception e)
    {
        if (debug) {
            e.printStackTrace();
        }
        else {
            out.println(e);
        }
    }

}
