package org.lsst.ccs.drivers.ad;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Scanner;
import javax.usb.UsbException;
import jline.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProc;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to test an Analog Devices 7747 evaluation board
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestAd7747 implements CmndProc.Dispatch {

    /*
    **  Command codes
    */
    private final static int
        CMD_READMEM     = 0,
        CMD_WRITEMEM    = 1,
        CMD_LOAD        = 2,
        CMD_RESET       = 3,
        CMD_READREG     = 4,
        CMD_WRITEREG    = 5,
        CMD_READCAP     = 6,
        CMD_READTEMP    = 7,
        CMD_READVOLT    = 8,
        CMD_READ2REG    = 9,
        CMD_READ3REG    = 10,
        CMD_SETCAP      = 11,
        CMD_SHOWCAP     = 12,
        CMD_SETVT       = 13,
        CMD_SHOWVT      = 14,
        CMD_SETCONV     = 15,
        CMD_SHOWCONV    = 16,
        CMD_SETDAC      = 17,
        CMD_SHOWDAC     = 18,
        CMD_INIT        = 19,
        NUM_CMDS        = 20;

    /*
    **  Command help text
    */
    private final static String[] helpInit = {
        "Initializes setup registers to standard values",
        "init",
    };

    private final static String[] helpSetcap = {
        "Set capacitance channel configuration",
        "setcap [<enable>] [<rate>]",
        "enable  If non-zero, enable capacitance channel; otherwise disable it",
        "rate    The encoded capacitance data conversion rate (0 - 7)",
    };

    private final static String[] helpShowcap = {
        "Display capacitance channel configuration",
        "showcap",
    };

    private final static String[] helpSetvt = {
        "Set voltage/temperature channel configuration",
        "setvt [<enable>] [<mode>] [<rate>] [<short>] [<extref>]",
        "enable  If non-zero, enable volt/temp channel; otherwise disable it",
        "mode    The encoded volt/temp channel mode (0 - 3)",
        "rate    The encoded volt/temp data conversion rate (0 - 3)",
        "short   If non-zero, short the input; otherwise open it",
        "extref  If non-zero, use external ref voltage; otherwise internal",
    };

    private final static String[] helpShowvt = {
        "Display voltage/temperature channel configuration",
        "showvt",
    };

    private final static String[] helpSetconv = {
        "Set conversion mode",
        "setconv <value>",
        "value  The encoded conversion mode (0 - 7)",
    };

    private final static String[] helpShowconv = {
        "Display conversion mode",
        "showconv",
    };

    private final static String[] helpSetdac = {
        "Set capacitance DAC state and value",
        "setdac <dac> [<enable>] [<value>]",
        "dac     If zero, use DAC A; otherwise DAC B",
        "enable  If non-zero enable the DAC; otherwise disable it",
        "value   The DAC value",
    };

    private final static String[] helpShowdac = {
        "Display DAC configuration",
        "showdac",
    };

    private final static String[] helpReadcap = {
        "Read the capacitance sensor and display the value",
        "readcap [<single>] [<immed>] [<count>]",
        "single  If present and non-zero, perform a single conversion",
        "immed   If present and non-zero, don't wait for conversion",
        "count   The number of reads to average (default 1)",
    };

    private final static String[] helpReadtemp = {
        "Read the internal or external temperature sensor and display the value",
        "readtemp [<extcfg>] [<extern>] [<single>] [<immed>] [<count>]",
        "extcfg  If present and non-zero, configure which sensor to read",
        "extern  If present and non-zero, read the external sensor",
        "single  If present and non-zero, perform a single conversion",
        "immed   If present and non-zero, don't wait for conversion",
        "count   The number of reads to average (default 1)",
    };

    private final static String[] helpReadvolt = {
        "Read the internal or external voltage and display the value",
        "readvolt [<extcfg>] [<extern>] [<single>] [<immed>] [<count>]",
        "extcfg  If present and non-zero, configure which voltage to read",
        "extern  If present and non-zero, read the external voltage",
        "single  If present and non-zero, perform a single conversion",
        "immed   If present and non-zero, don't wait for conversion",
        "count   The number of reads to average (default 1)",
    };

    private final static String[] helpReadmem = {
        "Read data from memory and display it",
        "readmem <addr> <leng> [<timwout>]",
        "addr     The address to read from",
        "leng     The number of bytes to read",
        "timeout  The timeout value (secs, default 1.0)",
    };

    private final static String[] helpWritemem = {
        "Write data to memory",
        "writemem <addr> [<data1>]... [<data8>]",
        "addr   The address to write to",
        "datan  String of (an even number of) hexadecimal digits to write",
    };

    private final static String[] helpReadreg = {
        "Read register(s) and display the contents",
        "readreg <regnum> [<count>]",
        "regnum   The number of the first register to read",
        "count    The number of registers to read (default: 1)",
    };

    private final static String[] helpWritereg = {
        "Write to register(s)",
        "writereg <regnum> <value1> [<value2>]... [<value8>]",
        "regnum  The number of the first register to write",
        "valuen  Values to write to consecutive registers",
    };

    private final static String[] helpRead2reg = {
        "Read two consecutive registers and display the resultant value",
        "readreg <regnum>",
        "regnum   The number of the first register to read",
    };

    private final static String[] helpRead3reg = {
        "Read three consecutive registers and display the resultant value",
        "readreg <regnum>",
        "regnum   The number of the first register to read",
    };

    private final static String[] helpReset = {
        "Set or clear the RESET state",
        "reset  <opern>",
        "opern  If non-zero, RESET is set; if zero, RESET is cleared",
    };

    private final static String[] helpLoad = {
        "Load memory from a file",
        "load  [<file>] [<force>]",
        "file   The name of the .hex file to load (default: standard)",
        "force  If present and non-zero, do the load even if unnecessary",
    };

    /*
    **  Command table setup
    */
    private final static CmndProc.Command cmnd = new CmndProc.Command(NUM_CMDS);
    static {
        cmnd.add("readcap",     CMD_READCAP,     helpReadcap);
        cmnd.add("readtemp",    CMD_READTEMP,    helpReadtemp);
        cmnd.add("readvolt",    CMD_READVOLT,    helpReadvolt);
        cmnd.add("readmem",     CMD_READMEM,     helpReadmem);
        cmnd.add("writemem",    CMD_WRITEMEM,    helpWritemem);
        cmnd.add("readreg",     CMD_READREG,     helpReadreg);
        cmnd.add("writereg",    CMD_WRITEREG,    helpWritereg);
        cmnd.add("read2reg",    CMD_READ2REG,    helpRead2reg);
        cmnd.add("read3reg",    CMD_READ3REG,    helpRead3reg);
        cmnd.add("init",        CMD_INIT,        helpInit);
        cmnd.add("setcap",      CMD_SETCAP,      helpSetcap);
        cmnd.add("setvt",       CMD_SETVT,       helpSetvt);
        cmnd.add("setconv",     CMD_SETCONV,     helpSetconv);
        cmnd.add("setdac",      CMD_SETDAC,      helpSetdac);
        cmnd.add("showcap",     CMD_SHOWCAP,     helpShowcap);
        cmnd.add("showvt",      CMD_SHOWVT,      helpShowvt);
        cmnd.add("showconv",    CMD_SHOWCONV,    helpShowconv);
        cmnd.add("showdac",     CMD_SHOWDAC,     helpShowdac);
        cmnd.add("reset",       CMD_RESET,       helpReset);
        cmnd.add("load",        CMD_LOAD,        helpLoad);
    }

    /*
    **  Lookup table setup
    */
    private final static CmndProc.Lookup capRateNames = new CmndProc.Lookup(8);
    static {
        capRateNames.add("45.5", Ad7747Eval.CAPFS_RATE_45);
        capRateNames.add("41.9", Ad7747Eval.CAPFS_RATE_42);
        capRateNames.add("25.0", Ad7747Eval.CAPFS_RATE_25);
        capRateNames.add("13.2", Ad7747Eval.CAPFS_RATE_13);
        capRateNames.add("8.1",  Ad7747Eval.CAPFS_RATE_8);
        capRateNames.add("6.5",  Ad7747Eval.CAPFS_RATE_7);
        capRateNames.add("5.5",  Ad7747Eval.CAPFS_RATE_6);
        capRateNames.add("4.6",  Ad7747Eval.CAPFS_RATE_5);
    }
    private final static CmndProc.Lookup vtRateNames = new CmndProc.Lookup(4);
    static {
        vtRateNames.add("49.8", Ad7747Eval.VTFS_RATE_50);
        vtRateNames.add("31.2", Ad7747Eval.VTFS_RATE_31);
        vtRateNames.add("16.1", Ad7747Eval.VTFS_RATE_16);
        vtRateNames.add("8.2",  Ad7747Eval.VTFS_RATE_8);
    }
    private final static CmndProc.Lookup vtModeNames = new CmndProc.Lookup(4);
    static {
        vtModeNames.add("temperature", Ad7747Eval.VTMD_INT_TEMP);
        vtModeNames.add("exttemp",     Ad7747Eval.VTMD_EXT_TEMP);
        vtModeNames.add("vdd",         Ad7747Eval.VTMD_VDD_MON);
        vtModeNames.add("voltage",     Ad7747Eval.VTMD_EXT_VOLT);
    }
    private final static CmndProc.Lookup cnvModeNames = new CmndProc.Lookup(6);
    static {
        cnvModeNames.add("idle",       Ad7747Eval.MODE_IDLE);
        cnvModeNames.add("continuous", Ad7747Eval.MODE_CONT);
        cnvModeNames.add("single",     Ad7747Eval.MODE_SINGLE);
        cnvModeNames.add("powerdown",  Ad7747Eval.MODE_POWERDOWN);
        cnvModeNames.add("offsetcal",  Ad7747Eval.MODE_OFFS_CAL);
        cnvModeNames.add("gaincal",    Ad7747Eval.MODE_GAIN_CAL);
    }

    /*
    **  Private fields
    */
    private final Output out = new ConsOut();
    private final ConsoleReader reader = new ConsoleReader();
    private final CmndProc proc = new CmndProc();
    private final Ad7747Eval ad;
    private final boolean debug;


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public TestAd7747(int index, boolean dbg)
        throws UsbException, IOException
    {
        proc.add(this, cmnd);
        ad = new Ad7747Eval(index);
        debug = dbg;
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        int index = (args.length >= 1) ? Integer.decode(args[0]) : 0;
        try {
            (new TestAd7747(index, args.length >= 2)).run();
        }
        catch (UsbException e) {
            System.out.println(e);
        }
        catch (IOException e) {
            System.out.println(e);
        }

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Run the test
    **
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run() throws IOException
    {
        while (true) {
            String line = reader.readLine(">> ");
            if (line == null) break;
            try {
                if (!proc.process(line)) break;
            }
            catch (RuntimeException e) {
                if (debug) e.printStackTrace(System.out);
                else out.println(e);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatch command for processing - exception-free version
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, Scanner scan)
    {
        try {
            return dispatchCmnd(code, scan);
        }
        catch (Exception e) {
            out.println(e.toString());
            return true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatch command for processing
    **
    ***************************************************************************
    */
    private boolean dispatchCmnd(int code, Scanner scan) throws UsbException
    {
        int found;
        Object[] args = new Object[16];

        switch (code) {

        case CMD_INIT:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                ad.setupStandard();
            }
            break;
        
        case CMD_SETCAP:
            found = CmndProc.scanArgs(scan, "il", args, capRateNames);
            if (found >= 0) {
                if ((found & 0x01) != 0)
                    ad.setCapEnabled((Integer)args[0] != 0);
                if ((found & 0x02) != 0)
                    ad.setCapConvRate((Integer)args[1]);
            }
            break;
        
        case CMD_SETVT:
            found = CmndProc.scanArgs(scan, "illii", args, vtModeNames,
                                      vtRateNames);
            if (found >= 0) {
                if ((found & 0x01) != 0)
                    ad.setVtEnabled((Integer)args[0] != 0);
                if ((found & 0x02) != 0)
                    ad.setVtMode((Integer)args[1]);
                if ((found & 0x04) != 0)
                    ad.setVtConvRate((Integer)args[2]);
                if ((found & 0x08) != 0)
                    ad.setVtShorted((Integer)args[3] != 0);
                if ((found & 0x10) != 0)
                    ad.setExternalRef((Integer)args[4] != 0);
            }
            break;
        
        case CMD_SETCONV:
            if (CmndProc.scanArgs(scan, "L", args, cnvModeNames) >= 0) {
                ad.setConvMode((Integer)args[0]);
            }
            break;
        
        case CMD_SETDAC:
            if ((found = CmndProc.scanArgs(scan, "Iii", args)) >= 0) {
                int dac = (Integer)args[0];
                if ((found & 0x02) != 0) {
                    if (dac == 0)
                        ad.setDacAEnabled((Integer)args[1] != 0);
                    else
                        ad.setDacBEnabled((Integer)args[1] != 0);
                }
                if ((found & 0x04) != 0) {
                    if (dac == 0)
                        ad.setDacAValue((Integer)args[2]);
                    else
                        ad.setDacBValue((Integer)args[2]);
                }
            }
            break;
        
        case CMD_SHOWCAP:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                int rate = ad.getCapConvRate();
                out.format("%s, conv. rate = %s (%s Hz)\n",
                           ad.isCapEnabled() ? "Enabled" : "Disabled",
                           rate, capRateNames.decode(rate));
            }
            break;
        
        case CMD_SHOWVT:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                int mode = ad.getVtMode();
                int rate = ad.getVtConvRate();
                out.format("%s, mode = %s (%s), conv. rate = %s (%s Hz)"
                             + "%s%s\n",
                           ad.isVtEnabled() ? "Enabled" : "Disabled",
                           mode, vtModeNames.decode(mode),
                           rate, vtRateNames.decode(rate),
                           ad.isVtShorted() ? ", shorted" : "",
                           ad.isExternalRef() ? ", external ref" : "");
            }
            break;
        
        case CMD_SHOWCONV:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                int mode = ad.getConvMode();
                out.format("Conversion mode = %s (%s)\n", mode,
                           cnvModeNames.decode(mode));
            }
            break;
        
        case CMD_SHOWDAC:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                out.format("DAC A: %s, value = %s; DAC B: %s, value = %s\n",
                           ad.isDacAEnabled() ? "Enabled" : "Disabled",
                           ad.getDacAValue(),
                           ad.isDacBEnabled() ? "Enabled" : "Disabled",
                           ad.getDacBValue());
            }
            break;

        case CMD_READCAP:
            if ((found = CmndProc.scanArgs(scan, "iii", args)) >= 0) {
                int optns = 0;
                if ((found & 0x01) != 0 && (Integer)args[0] != 0)
                    optns |= Ad7747Eval.OPTN_SINGLE;
                if ((found & 0x02) != 0 && (Integer)args[1] != 0)
                    optns |= Ad7747Eval.OPTN_IMMED;
                int count = (found & 0x04) != 0 ? (Integer)args[2] : 1;
                double sum = 0, sumsq = 0;
                for (int j = 0; j < count; j++) {
                    double value = ad.readCapacitance(optns);
                    sum += value;
                    sumsq += value * value;
                }
                showStats("Capacitance", count, sum, sumsq);
            }
            break;
        
        case CMD_READTEMP:
            if ((found = CmndProc.scanArgs(scan, "iiiii", args)) >= 0) {
                int optns = 0;
                if ((found & 0x01) != 0 && (Integer)args[0] != 0)
                    optns |= Ad7747Eval.OPTN_EXTCFG;
                if ((found & 0x02) != 0 && (Integer)args[1] != 0)
                    optns |= Ad7747Eval.OPTN_EXTERN;
                if ((found & 0x04) != 0 && (Integer)args[2] != 0)
                    optns |= Ad7747Eval.OPTN_SINGLE;
                if ((found & 0x08) != 0 && (Integer)args[3] != 0)
                    optns |= Ad7747Eval.OPTN_IMMED;
                int count = (found & 0x10) != 0 ? (Integer)args[4] : 1;
                double sum = 0, sumsq = 0;
                for (int j = 0; j < count; j++) {
                    double value = ad.readTemperature(optns);
                    sum += value;
                    sumsq += value * value;
                }
                showStats("Temperature", count, sum, sumsq);
            }
            break;
        
        case CMD_READVOLT:
            if ((found = CmndProc.scanArgs(scan, "iiiii", args)) >= 0) {
                int optns = 0;
                if ((found & 0x01) != 0 && (Integer)args[0] != 0)
                    optns |= Ad7747Eval.OPTN_EXTCFG;
                if ((found & 0x02) != 0 && (Integer)args[1] != 0)
                    optns |= Ad7747Eval.OPTN_EXTERN;
                if ((found & 0x04) != 0 && (Integer)args[2] != 0)
                    optns |= Ad7747Eval.OPTN_SINGLE;
                if ((found & 0x08) != 0 && (Integer)args[3] != 0)
                    optns |= Ad7747Eval.OPTN_IMMED;
                int count = (found & 0x10) != 0 ? (Integer)args[4] : 1;
                double sum = 0, sumsq = 0;
                for (int j = 0; j < count; j++) {
                    double value = ad.readVoltage(optns);
                    sum += value;
                    sumsq += value * value;
                }
                showStats("Voltage", count, sum, sumsq);
            }
            break;

        case CMD_READREG:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0) {
                int regnum = (Integer)args[0];
                if ((found & 0x02) != 0) {
                    byte[] data = new byte[(Integer)args[1]];
                    dispRegs(regnum, data, ad.readRegister(regnum, data));
                }
                else
                    out.format("%02x: %02x\n", regnum, ad.readRegister(regnum));
            }
            break;

        case CMD_READ2REG:
            if (CmndProc.scanArgs(scan, "I", args) >= 0) {
                out.format("Value = %d (0x%1$04x)\n",
                           ad.readRegister2((Integer)args[0]));
            }
            break;
        
        case CMD_READ3REG:
            if (CmndProc.scanArgs(scan, "I", args) >= 0) {
                out.format("Value = %d (0x%1$04x)\n",
                           ad.readRegister3((Integer)args[0]));
            }
            break;
        
        case CMD_WRITEREG:
            if ((found = CmndProc.scanArgs(scan, "Issssssss", args)) >= 0) {
                int regnum = (Integer)args[0], count = 0, value = -1;
                for (int j = 1; j <= 8; j++) {
                    if ((found & (1 << j)) != 0) {
                        String number = (String)args[j];
                        count += (number.length() + 1) / 2;
                        if (count == 1) {
                            try {
                                value = Integer.parseInt(number, 16);
                            }
                            catch (NumberFormatException e) {
                                out.println("Invalid hexadecimal string");
                            }
                        }
                    }
                }
                if (count == 1) {
                    if (value == -1) break;
                    ad.writeRegister(regnum, value);
                }
                else {
                    byte[] data = genData(found & 0x1fe, args);
                    ad.writeMemory((Integer)args[0], data);
                }
            }
            break;

        case CMD_RESET:
            if (CmndProc.scanArgs(scan, "I", args) >= 0) {
                ad.setReset((Integer)args[0] != 0);
            }
            break;

        case CMD_READMEM:
            if ((found = CmndProc.scanArgs(scan, "IIf", args)) >= 0) {
                int addr = (Integer)args[0];
                float tmo = 1000F * ((found & 0x04) != 0 ? (Float)args[2] : 1F);
                byte[] data = new byte[(Integer)args[1]];
                dispData(addr, data, ad.readMemory(addr, data, (int)tmo));
            }
            break;
        
        case CMD_WRITEMEM:
            if ((found = CmndProc.scanArgs(scan, "Issssssss", args)) >= 0) {
                byte[] data = genData(found & 0x1fe, args);
                ad.writeMemory((Integer)args[0], data);
            }
            break;

        case CMD_LOAD:
            if ((found = CmndProc.scanArgs(scan, "si", args)) >= 0) {
                boolean force = false;
                if ((found & 0x02) != 0 && (Integer)args[1] != 0)
                    force = true;
                if ((found & 0x01) == 0)
                    ad.load(force);
                else
                    ad.load((String)args[0], force);
            }
            break;
        
        default:
            out.println("Command not fully implemented");

        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Display mean and sigma values
    **
    ***************************************************************************
    */
    private void showStats(String name, int count, double sum, double sumsq)
    {
        double mean = sum / count;
        double sigsq = sumsq / count - mean * mean;
        double sigma = sigsq <= 0 ? 0 : Math.sqrt(sigsq);
        out.format("%s: mean = %.7g, sigma = %.7g\n", name, mean, sigma);
    }


   /**
    ***************************************************************************
    **
    **  Display register values
    **
    ***************************************************************************
    */
    private void dispRegs(int regnum, byte[] data, int count)
    {
        for (int j = 0; j < count; j++) {
            if ((j & 0x0f) == 0) {
                if (j != 0) out.println();
                out.format("%02x:", regnum + j);
            }
            out.format(" %02x", data[j] & 0xff);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Display memory data
    **
    ***************************************************************************
    */
    private void dispData(int addr, byte[] data, int count)
    {
        for (int j = 0; j < count; j++) {
            if ((j & 0x0f) == 0) {
                if (j != 0) out.println();
                out.format("%04x:", addr + j);
            }
            out.format(" %02x", data[j] & 0xff);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Generate write data from hexadecimal string arguments
    **
    ***************************************************************************
    */
    private byte[] genData(int found, Object[] args)
    {
        int count = 0, mask = found;
        ArrayList list = new ArrayList();
        for (int j = 0; found != 0; found >>= 1, j++) {
            if ((found & 1) == 0) continue;
            String hex = (String)args[j];
            int leng = hex.length();
            if ((leng & 1) != 0) break;
            byte[] sData = new byte[leng / 2];
            try {
                for (int k = 0; k < leng; k += 2) {
                    String num = hex.substring(k, k + 2);
                    sData[k / 2] = (byte)Integer.parseInt(num, 16);
                }
            }
            catch (NumberFormatException e) {
                break;
            }
            count += leng / 2;
            list.add(sData);
            mask ^= (1 << j);
        }
        if (mask != 0) {
            out.println("Invalid hexadecimal string");
            return null;
        }
        byte[] data = new byte[count];
        int l = 0;
        for (int j = 0; j < list.size(); j++) {
            byte[] sData = (byte [])list.get(j);
            for (int k = 0; k < sData.length; k++) {
                data[l++] = sData[k];
            }
        }

        return data;
    }

}
