package org.lsst.ccs.drivers.ad;

import java.io.FileNotFoundException;
import java.io.InputStream;
import javax.usb.UsbException;
import org.lsst.ccs.drivers.usb.UsbComm;

/**
 ***************************************************************************
 **
 **  Routines to access an Analog Devices AD7747 evaluation board
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Ad7747Eval {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        STATUS_REG     = 0,        // Status register
        CAP_DATA_REG   = 1,        // Capacitance data registers (3-byte)
        VT_DATA_REG    = 4,        // Voltage/temp data registers (3-byte)
        CAP_SETUP_REG  = 7,        // Capacitance setup register
        VT_SETUP_REG   = 8,        // Voltage/temperature setup register
        EXC_SETUP_REG  = 9,        // Excitation setup register
        CONFIG_REG     = 10,       // Configuration register
        CAP_DAC_A_REG  = 11,       // Capacitance DAC A register
        CAP_DAC_B_REG  = 12,       // Capacitance DAC B register
        CAP_OFFSET_REG = 13,       // Capacitance offset register
        CAP_GAIN_REG   = 15,       // Capacitance gain register
        VOLT_GAIN_REG  = 17,       // Voltage gain register

        STS_READY      = 0x04,     // Status: ready overall
        STS_RDYVT      = 0x02,     // Status: voltage/temp data ready
        STS_RDYCAP     = 0x01,     // Status: capacitance data ready

        CPSU_CAPEN     = 0x80,     // Cap setup: enable
        CPSU_CAPDIFF   = 0x20,     // Cap setup: differential input
        CPSU_STANDARD  = CPSU_CAPDIFF,  // Cal setup: standard

        VTSU_VTEN      = 0x80,     // Volt/temp setup: enable
        VTSU_VTMODE_M  = 0x60,     // Volt/temp setup: mode mask
        VTSU_VTMODE_V  = 5,        // Volt/temp setup: mode offset
        VTMD_INT_TEMP  = 0,        // Volt/temp mode: internal temperature
        VTMD_EXT_TEMP  = 1,        // Volt/temp mode: external temperature
        VTMD_VDD_MON   = 2,        // Volt/temp mode: monitor VDD
        VTMD_EXT_VOLT  = 3,        // Volt/temp mode: external voltage
        VTSU_EXTREF    = 0x10,     // Volt/temp setup: use external reference
        VTSU_VTSHORT   = 0x02,     // Volt/temp setup: input shorted
        VTSU_VTCHOP    = 0x01,     // Volt/temp setup: chopper on
        VTSU_STANDARD  = VTSU_VTCHOP,  // Volt/temp setup: standard

        EXSU_EXCDAC    = 0x08,     // Exc setup: DAC enabled
        EXSU_EXCEN     = 0x04,     // Exc setup: enabled
        EXSU_EXCLVL_M  = 0x03,     // Exc setup: level mask
        EXSU_EXCLVL_V  = 0,        // Exc setup: level offset
        EXCLVL_1_8     = 0,        // Exc level: 1/8 VDD
        EXCLVL_1_4     = 1,        // Exc level: 1/4 VDD
        EXCLVL_3_8     = 2,        // Exc level: 3/8 VDD
        EXCLVL_1_2     = 3,        // Exc level: 1/2 VDD
        EXSU_STANDARD  = EXSU_EXCDAC | EXSU_EXCEN | EXCLVL_3_8,  // Exc setup: std

        CFG_VTFS_M     = 0xc0,     // Config: volt/temp filter mask
        CFG_VTFS_V     = 6,        // Config: volt/temp filter offset
        VTFS_RATE_50   = 0,        // 
        VTFS_RATE_31   = 1,
        VTFS_RATE_16   = 2,
        VTFS_RATE_8    = 3,
        VTFS_TIME_20   = VTFS_RATE_50,
        VTFS_TIME_32   = VTFS_RATE_31,
        VTFS_TIME_62   = VTFS_RATE_16,
        VTFS_TIME_122  = VTFS_RATE_8,
        CFG_CAPFS_M    = 0x38,     // Config: cap filter mask
        CFG_CAPFS_V    = 3,        // Config: cap filter offset
        CAPFS_RATE_45  = 0,
        CAPFS_RATE_42  = 1,
        CAPFS_RATE_25  = 2,
        CAPFS_RATE_13  = 3,
        CAPFS_RATE_8   = 4,
        CAPFS_RATE_7   = 5,
        CAPFS_RATE_6   = 6,
        CAPFS_RATE_5   = 7,
        CAPFS_TIME_22  = CAPFS_RATE_45,
        CAPFS_TIME_24  = CAPFS_RATE_42,
        CAPFS_TIME_40  = CAPFS_RATE_25,
        CAPFS_TIME_76  = CAPFS_RATE_13,
        CAPFS_TIME_124 = CAPFS_RATE_8,
        CAPFS_TIME_154 = CAPFS_RATE_7,
        CAPFS_TIME_184 = CAPFS_RATE_6,
        CAPFS_TIME_219 = CAPFS_RATE_5,
        CFG_MODE_M     = 0x07,     // Config: mode mask
        CFG_MODE_V     = 0,        // Config: mode offset
        MODE_IDLE      = 0,        // Mode: idle
        MODE_CONT      = 1,        // Mode: continuous
        MODE_SINGLE    = 2,        // Mode: single shot
        MODE_POWERDOWN = 3,        // Mode: power down
        MODE_OFFS_CAL  = 5,        // Mode: offset calibration
        MODE_GAIN_CAL  = 6,        // Mode: gain calibration

        DACA_ENABLE    = 0x80,
        DACA_VALUE_M   = 0x3f,

        DACB_ENABLE    = 0x80,
        DACB_VALUE_M   = 0x3f,

        OPTN_EXTCFG    = 0x01,
        OPTN_EXTERN    = 0x02,
        OPTN_SINGLE    = 0x04,
        OPTN_IMMED     = 0x08;

    public final static float
        HALF_RANGE3     = 8388608F,
        REF_VOLTAGE     = 1.17F,
        CAP_FULL_SCALE  = 8.192F,
        TEMP_FULL_SCALE = 4096F,
        INT_VOLT_SCALE  = 6F;

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int
        AD_VID            = 0x0456,
        AD_7747_PID       = 0xb481,
        READ_TIMEOUT      = 1000,
        RTYP_VENDOR_READ  = 0xc0,
        RTYP_VENDOR_WRITE = 0x40,
        RQST_CYPRESS      = 0xa0,
        RQST_I2C          = 0xdd,
        ADDR_I2C_READ     = 0x91,
        ADDR_I2C_WRITE    = 0x90,
        CACHE_SIZE        = 4096,
        PACKET_SIZE       = 256;

    private final static String
        STD_LOAD_FILE     = "/Icv_usb.hex";

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private static String loadFile;
    private static int[] loadMap = new int[CACHE_SIZE / 32 + 1];
    private static byte[] loadImage = new byte[CACHE_SIZE + 1];

    private final UsbComm com;


   /**
    ***************************************************************************
    **
    **  Constructors
    **
    ***************************************************************************
    */
    public Ad7747Eval() throws UsbException
    {
        this(0);
    }

    public Ad7747Eval(int index) throws UsbException
    {
        com = new UsbComm();
        if (com.findDevice(AD_VID, AD_7747_PID, index) == null)
            throw new Ad7747Exception("Board not found");
    }


   /**
    ***************************************************************************
    **
    **  Performs standard board setup
    **
    **  <p>
    **  The three setup register are written with capacitance and voltage
    **  channels disabled, and all other bits set to the values mandated for
    **  correct operation.  The configuration register is set to idle mode
    **  and the maximum conversion rates for the data channels.
    **
    ***************************************************************************
    */
    public void setupStandard() throws UsbException
    {
        writeRegister(CAP_SETUP_REG, CPSU_STANDARD);
        writeRegister(EXC_SETUP_REG, EXSU_STANDARD);
        writeRegister(VT_SETUP_REG, VTSU_STANDARD);
        writeRegister(CONFIG_REG, 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the board status
    **
    ***************************************************************************
    */
    public int getStatus() throws UsbException
    {
        return readRegister(STATUS_REG);
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables the capacitance channel
    **
    **  @param  set  If true, the channel is enabled; otherwise disabled
    **
    ***************************************************************************
    */
    public void setCapEnabled(boolean set) throws UsbException
    {
        updateRegister(CAP_SETUP_REG, CPSU_CAPEN, set ? CPSU_CAPEN : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the enabled state of the capacitance channel
    **
    **  @return  The enabled state
    **
    ***************************************************************************
    */
    public boolean isCapEnabled() throws UsbException
    {
        return (readRegister(CAP_SETUP_REG) & CPSU_CAPEN) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables the capacitance excitation
    **
    **  @param  set  If true, excitation is enabled; otherwise disabled
    **
    ***************************************************************************
    */
    public void setExcEnabled(boolean set) throws UsbException
    {
        updateRegister(EXC_SETUP_REG, EXSU_EXCEN, set ? EXSU_EXCEN : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the enabled state of the capacitance excitation
    **
    **  @return  The enabled state
    **
    ***************************************************************************
    */
    public boolean isExcEnabled() throws UsbException
    {
        return (readRegister(EXC_SETUP_REG) & EXSU_EXCEN) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables the voltage/temperature channel
    **
    **  @param  set  If true, the channel is enabled; otherwise disabled
    **
    ***************************************************************************
    */
    public void setVtEnabled(boolean set) throws UsbException
    {
        updateRegister(VT_SETUP_REG, VTSU_VTEN, set ? VTSU_VTEN : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the enabled state of the voltage/temperature channel
    **
    **  @return  The enabled state
    **
    ***************************************************************************
    */
    public boolean isVtEnabled() throws UsbException
    {
        return (readRegister(VT_SETUP_REG) & VTSU_VTEN) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the mode of the voltage/temperature channel
    **
    **  @param  mode  The mode value to set
    **
    ***************************************************************************
    */
    public void setVtMode(int mode) throws UsbException
    {
        updateRegister(VT_SETUP_REG, VTSU_VTMODE_M, mode << VTSU_VTMODE_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the mode of the voltage/temperature channel
    **
    **  @return  The mode value
    **
    ***************************************************************************
    */
    public int getVtMode() throws UsbException
    {
        return (readRegister(VT_SETUP_REG) & VTSU_VTMODE_M) >> VTSU_VTMODE_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets or clears the shorted state of the voltage/temperature channel
    **
    **  @param  set  If true, shorted state is set; otherwise cleared
    **
    ***************************************************************************
    */
    public void setVtShorted(boolean set) throws UsbException
    {
        updateRegister(VT_SETUP_REG, VTSU_VTSHORT, set ? VTSU_VTSHORT : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the shorted state of the voltage/temperature channel
    **
    **  @return  The shorted state
    **
    ***************************************************************************
    */
    public boolean isVtShorted() throws UsbException
    {
        return (readRegister(VT_SETUP_REG) & VTSU_VTSHORT) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets or clears the use of the external reference voltage
    **
    **  @param  set  If true, external reference is used; otherwise internal
    **
    ***************************************************************************
    */
    public void setExternalRef(boolean set) throws UsbException
    {
        updateRegister(VT_SETUP_REG, VTSU_EXTREF, set ? VTSU_EXTREF : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets whether the external reference voltage is being used
    **
    **  @return  The external reference state
    **
    ***************************************************************************
    */
    public boolean isExternalRef() throws UsbException
    {
        return (readRegister(VT_SETUP_REG) & VTSU_EXTREF) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the voltage/temperature conversion rate
    **
    **  @param  rate  The rate value to set
    **
    ***************************************************************************
    */
    public void setVtConvRate(int rate) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_VTFS_M, rate << CFG_VTFS_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the voltage/temperature conversion rate
    **
    **  @return  The conversion rate
    **
    ***************************************************************************
    */
    public int getVtConvRate() throws UsbException
    {
        return (readRegister(CONFIG_REG) & CFG_VTFS_M) >> CFG_VTFS_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the capacitance conversion rate
    **
    **  @param  rate  The rate value to set
    **
    ***************************************************************************
    */
    public void setCapConvRate(int rate) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_CAPFS_M, rate << CFG_CAPFS_V);
    }


   /**
    ***************************************************************************
    **
    **  Gets the capacitance conversion rate
    **
    **  @return  The conversion rate
    **
    ***************************************************************************
    */
    public int getCapConvRate() throws UsbException
    {
        return (readRegister(CONFIG_REG) & CFG_CAPFS_M) >> CFG_CAPFS_V;
    }


   /**
    ***************************************************************************
    **
    **  Sets the conversion mode
    **
    **  @param  mode  The mode value to set
    **
    ***************************************************************************
    */
    public void setConvMode(int mode) throws UsbException
    {
        updateRegister(CONFIG_REG, CFG_MODE_M, mode);
    }


   /**
    ***************************************************************************
    **
    **  Gets the conversion mode
    **
    **  @return  The mode value
    **
    ***************************************************************************
    */
    public int getConvMode() throws UsbException
    {
        return readRegister(CONFIG_REG) & CFG_MODE_M;
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables capacitance DAC A
    **
    **  @param  set  If true, capacitance DAC A is enabled; otherwise disabled
    **
    ***************************************************************************
    */
    public void setDacAEnabled(boolean set) throws UsbException
    {
        updateRegister(CAP_DAC_A_REG, DACA_ENABLE, set ? DACA_ENABLE : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets capacitance DAC A enabled state
    **
    **  @return  The enabled state
    **
    ***************************************************************************
    */
    public boolean isDacAEnabled() throws UsbException
    {
        return (readRegister(CAP_DAC_A_REG) & DACA_ENABLE) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets capacitance DAC A value
    **
    **  @param  value  The DAC value to set
    **
    ***************************************************************************
    */
    public void setDacAValue(int value) throws UsbException
    {
        updateRegister(CAP_DAC_A_REG, DACA_VALUE_M, value);
    }


   /**
    ***************************************************************************
    **
    **  Gets capacitance DAC A value
    **
    **  @return  The DAC value
    **
    ***************************************************************************
    */
    public int getDacAValue() throws UsbException
    {
        return readRegister(CAP_DAC_A_REG) & DACA_VALUE_M;
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables capacitance DAC B
    **
    **  @param  set  If true, capacitance DAC B is enabled; otherwise disabled
    **
    ***************************************************************************
    */
    public void setDacBEnabled(boolean set) throws UsbException
    {
        updateRegister(CAP_DAC_B_REG, DACB_ENABLE, set ? DACB_ENABLE : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets capacitance DAC B enabled state
    **
    **  @return  The enabled state
    **
    ***************************************************************************
    */
    public boolean isDacBEnabled() throws UsbException
    {
        return (readRegister(CAP_DAC_B_REG) & DACB_ENABLE) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets capacitance DAC B value
    **
    **  @param  value  The DAC value to set
    **
    ***************************************************************************
    */
    public void setDacBValue(int value) throws UsbException
    {
        updateRegister(CAP_DAC_B_REG, DACB_VALUE_M, value);
    }


   /**
    ***************************************************************************
    **
    **  Gets capacitance DAC B value
    **
    **  @return  The DAC value
    **
    ***************************************************************************
    */
    public int getDacBValue() throws UsbException
    {
        return readRegister(CAP_DAC_B_REG) & DACB_VALUE_M;
    }


   /**
    ***************************************************************************
    **
    **  Reads capacitance sensor
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 OPTN_SINGLE: First configures for a single conversion.
    **
    **                 OPTN_IMMED:  Reads the data registers immediately
    **                              instead of waiting for a conversion to
    **                              complete.
    **
    **  @return  The read capacitance (pF)
    **
    ***************************************************************************
    */
    public float readCapacitance(int optns) throws UsbException
    {
        if ((optns & OPTN_SINGLE) != 0)
            updateRegister(CONFIG_REG, CFG_MODE_M, MODE_SINGLE);
        if ((optns & OPTN_IMMED) == 0)
            waitReady(STS_RDYCAP);

        return (readRegister3(CAP_DATA_REG) / HALF_RANGE3 - 1F)
                 * CAP_FULL_SCALE;
    }


   /**
    ***************************************************************************
    **
    **  Reads temperature sensor
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 OPTN_EXTCFG: Configures to read either the internal or
    **                              external temperature sensor.
    **
    **                 OPTN_EXTERN: If configuring, selects the external sensor
    **                              instead of the internal one.
    **
    **                 OPTN_SINGLE: First configures for a single conversion.
    **
    **                 OPTN_IMMED:  Reads the data registers immediately
    **                              instead of waiting for a conversion to
    **                              complete.
    **
    **  @return  The read temperature (Celsius)
    **
    ***************************************************************************
    */
    public float readTemperature(int optns) throws UsbException
    {
        if ((optns & OPTN_EXTCFG) != 0)
            updateRegister(VT_SETUP_REG, VTSU_VTMODE_M,
                           (((optns & OPTN_EXTERN) != 0)
                             ? VTMD_EXT_TEMP : VTMD_INT_TEMP) << VTSU_VTMODE_V);
        if ((optns & OPTN_SINGLE) != 0)
            updateRegister(CONFIG_REG, CFG_MODE_M, MODE_SINGLE);
        if ((optns & OPTN_IMMED) == 0)
            waitReady(STS_RDYVT);

        return (readRegister3(VT_DATA_REG) / HALF_RANGE3 - 1F)
                 * TEMP_FULL_SCALE;
    }


   /**
    ***************************************************************************
    **
    **  Reads voltage
    **
    **  @param  optns  Options word, composed of the following possible bit
    **                 masks:
    **
    **                 OPTN_EXTCFG: Configures to read either the internal or
    **                              external temperature sensor.
    **
    **                 OPTN_EXTERN: Selects the external sensor instead of the
    **                              internal one (for both configuring and
    **                              conversion).
    **
    **                 OPTN_SINGLE: First configures for a single conversion.
    **
    **                 OPTN_IMMED:  Reads the data registers immediately
    **                              instead of waiting for a conversion to
    **                              complete.
    **
    **  @return  The read voltage
    **
    ***************************************************************************
    */
    public float readVoltage(int optns) throws UsbException
    {
        boolean ext = (optns & OPTN_EXTERN) != 0;
        if ((optns & OPTN_EXTCFG) != 0)
            updateRegister(VT_SETUP_REG, VTSU_VTMODE_M,
                           (ext ? VTMD_EXT_VOLT : VTMD_VDD_MON) << VTSU_VTMODE_V);
        if ((optns & OPTN_SINGLE) != 0)
            updateRegister(CONFIG_REG, CFG_MODE_M, MODE_SINGLE);
        if ((optns & OPTN_IMMED) == 0)
            waitReady(STS_RDYVT);
        float volts = (readRegister3(VT_DATA_REG) / HALF_RANGE3 - 1F)
                        * REF_VOLTAGE;
        if (!ext) volts *= INT_VOLT_SCALE;

        return volts;
    }


   /**
    ***************************************************************************
    **
    **  Reads 16-bit value from a pair of registers
    **
    **  @param  regnum  The number of the first (high-order) register to read
    **
    **  @return  The 16-bit raw value
    **
    ***************************************************************************
    */
    public int readRegister2(int regnum) throws UsbException
    {
        byte[] data = new byte[2];
        readRegister(regnum, data);

        return (data[1] & 0xff) | ((data[0] & 0xff) << 8);
    }


   /**
    ***************************************************************************
    **
    **  Reads 24-bit value from a triplet of registers
    **
    **  @param  regnum  The number of the first (high-order) register to read
    **
    **  @return  The 24-bit raw value
    **
    ***************************************************************************
    */
    public int readRegister3(int regnum) throws UsbException
    {
        byte[] data = new byte[3];
        readRegister(regnum, data);

        return (data[2] & 0xff) | ((data[1] & 0xff) << 8)
                 | ((data[0] & 0xff) << 16);
    }


   /**
    ***************************************************************************
    **
    **  Reads byte array from consecutive registers
    **
    **  @param  regnum   The first register to read
    **
    **  @param  data     The array of bytes to read into
    **
    **  @param  timeout  The maximum time to wait (ms), or 0 for no timeout
    **
    **  @return  The number of bytes read.
    **
    ***************************************************************************
    */
    public int readRegister(int regnum, byte[] data) throws UsbException
    {
        return com.read(RTYP_VENDOR_READ, RQST_I2C, ADDR_I2C_READ, regnum,
                        data, READ_TIMEOUT);
    }


   /**
    ***************************************************************************
    **
    **  Reads a register
    **
    **  @param  regnum   The register to read
    **
    **  @param  timeout  The maximum time to wait (ms), or 0 for no timeout
    **
    **  @return  The value read
    **
    ***************************************************************************
    */
    public int readRegister(int regnum) throws UsbException
    {
        byte[] data = new byte[1];
        com.read(RTYP_VENDOR_READ, RQST_I2C, ADDR_I2C_READ, regnum, data,
                 READ_TIMEOUT);

        return data[0] & 0xff;
    }


   /**
    ***************************************************************************
    **
    **  Writes byte array to consecutive registers
    **
    **  @param  regnum  The first register to write
    **
    **  @param  data    The array of bytes to write
    **
    **  @return  The number of bytes written.
    **
    ***************************************************************************
    */
    public int writeRegister(int regnum, byte[] data) throws UsbException
    {
        return com.write(RTYP_VENDOR_WRITE, RQST_I2C, ADDR_I2C_WRITE, regnum,
                         data);
    }


   /**
    ***************************************************************************
    **
    **  Writes a register
    **
    **  @param  regnum  The register to write
    **
    **  @param  value   The value to write
    **
    ***************************************************************************
    */
    public void writeRegister(int regnum, int value) throws UsbException
    {
        byte data[] = {(byte)value};
        com.write(RTYP_VENDOR_WRITE, RQST_I2C, ADDR_I2C_WRITE, regnum, data);
    }


   /**
    ***************************************************************************
    **
    **  Updates a register
    **
    **  @param  regnum  The register to update
    **
    **  @param  mask    The mask of bits to update
    **
    **  @param  value   The value to write, under the mask
    **
    ***************************************************************************
    */
    public void updateRegister(int regnum, int mask, int value)
        throws UsbException
    {
        byte data[] = new byte[1];
        com.read(RTYP_VENDOR_READ, RQST_I2C, ADDR_I2C_READ, regnum, data,
                 READ_TIMEOUT);
        data[0] = (byte)((data[0] & ~mask) | (value & mask));
        com.write(RTYP_VENDOR_WRITE, RQST_I2C, ADDR_I2C_WRITE, regnum, data);
    }


   /**
    ***************************************************************************
    **
    **  Sets or clears the RESET state
    **
    **  @param  on  If true, sets RESET, otherwise clears RESET
    **
    ***************************************************************************
    */
    public void setReset(boolean on) throws UsbException
    {
        byte data[] = {(byte)(on ? 1 : 0)};
        com.write(RTYP_VENDOR_WRITE, RQST_CYPRESS, 0xe600, 0, data);
    }


   /**
    ***************************************************************************
    **
    **  Reads byte array from memory
    **
    **  @param  addr     The first address to read
    **
    **  @param  data     The array of bytes to read into
    **
    **  @param  timeout  The maximum time to wait (ms), or 0 for no timeout
    **
    **  @return  The number of bytes read.
    **
    ***************************************************************************
    */
    public int readMemory(int addr, byte[] data, int timeout)
        throws UsbException
    {
        return com.read(RTYP_VENDOR_READ, RQST_CYPRESS, addr, 0, data, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Reads byte array from memory
    **
    **  @param  addr     The first address to read
    **
    **  @param  data     The array of bytes to read into
    **
    **  @param  offs     The offset in data to the first byte to read into
    **
    **  @param  leng     The number of bytes to read
    **
    **  @param  timeout  The maximum time to wait (ms), or 0 for no timeout
    **
    **  @return  The number of bytes read.
    **
    ***************************************************************************
    */
    public int readMemory(int addr, byte[] data, int offs, int leng,
                          int timeout) throws UsbException
    {
        return com.read(RTYP_VENDOR_READ, RQST_CYPRESS, addr, 0, data, offs,
                        leng, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Writes byte array to memory
    **
    **  @param  addr  The first address to write
    **
    **  @param  data  The array of bytes to write
    **
    **  @return  The number of bytes written.
    **
    ***************************************************************************
    */
    public int writeMemory(int addr, byte[] data) throws UsbException
    {
        return com.write(RTYP_VENDOR_WRITE, RQST_CYPRESS, addr, 0, data);
    }


   /**
    ***************************************************************************
    **
    **  Writes byte array to memory
    **
    **  @param  addr  The first address to write
    **
    **  @param  data  The array of bytes to write
    **
    **  @param  offs  The offset in data to the first byte to write from
    **
    **  @param  leng  The number of bytes to write
    **
    **  @return  The number of bytes written.
    **
    ***************************************************************************
    */
    public int writeMemory(int addr, byte[] data, int offs, int leng)
        throws UsbException
    {
        return com.write(RTYP_VENDOR_WRITE, RQST_CYPRESS, addr, 0, data, offs,
                         leng);
    }


   /**
    ***************************************************************************
    **
    **  Loads memory from standard file
    **
    **  @param  force  If true, force the load even if memory appears to be
    **                 already loaded correctly.
    **
    ***************************************************************************
    */
    public void load(boolean force) throws UsbException
    {
        load(STD_LOAD_FILE, force);
    }


   /**
    ***************************************************************************
    **
    **  Loads memory from a file
    **
    **  @param  name  The name of the "hex" file to load
    **
    **  @param  force  If true, force the load even if memory appears to be
    **                 already loaded correctly.
    **
    ***************************************************************************
    */
    public void load(String name, boolean force) throws UsbException
    {
        if (!name.equals(loadFile))
            cacheFile(name);
        if (force || !checkCache())
            loadCache();
    }


   /**
    ***************************************************************************
    **
    **  Caches a load file's contents
    **
    **  @param  name  The name of the "hex" file to cache
    **
    ***************************************************************************
    */
    private void cacheFile(String name) throws UsbException
    {
        byte[] data;
        int fileSize = 0, nRead;
        try {
            InputStream file = Ad7747Eval.class.getResourceAsStream(name);
            if (file == null)
                throw new FileNotFoundException(name);
            data = new byte[1024];
            while (true) {
                nRead = file.read(data);
                if (nRead < 0) break;
                fileSize += nRead;
            }
            file.close();
        }
        catch (Exception e) {
            throw new Ad7747Exception("Error reading file: " + e);
        }
        try {
            InputStream file = Ad7747Eval.class.getResourceAsStream(name);
            if (file == null)
                throw new FileNotFoundException(name);
            data = new byte[fileSize];
            nRead = file.read(data);
            file.close();
        }
        catch (Exception e) {
            throw new Ad7747Exception("Error reading file: " + e);
        }
        if (nRead != fileSize)
            throw new Ad7747Exception("Inconsistent data amount read");

        loadFile = null;
        for (int j = 0; j < loadMap.length; j++)
            loadMap[j] = 0;
        int posn = 0;
        while (posn >= 0 && posn < fileSize)
            posn = processLine(data, posn);
        if (posn >= 0)
            throw new Ad7747Exception("EOF record missing");
        if (-posn < fileSize)
            throw new Ad7747Exception("EOF record not last in file");
        loadFile = name;
    }


   /**
    ***************************************************************************
    **
    **  Processes a line of data
    **
    **  @param  data  Array containing the complete file data
    **
    **  @param  posn  The current position in the data
    **
    **  @return  The updated position in the data
    **
    ***************************************************************************
    */
    private int processLine(byte[] data, int posn) throws UsbException
    {
        if (posn + 11 > data.length) {
            throw new Ad7747Exception("Record too short");
        }
        if (data[posn] != 0x3a) {
            throw new Ad7747Exception("Beginning-of-line marker not found");
        }
        int leng = decodeHex(data, posn + 1);
        if (posn + 11 + 2 * leng > data.length) {
            throw new Ad7747Exception("Record too short");
        }

        posn += 3;
        int checkSum = leng, addr = 0, type = 0;
        for (int j = 0; j < leng + 4; j++, posn += 2) {
            int value = decodeHex(data, posn);
            if (j == 0)
                addr = value << 8;
            else if (j == 1)
                addr |= value;
            else if (j == 2) {
                type = value;
                if (type != 0 && type != 1)
                    throw new Ad7747Exception("Invalid record type");
            }
            else if (j < leng + 3) {
                int ad = addr + j - 3;
                if (ad >= CACHE_SIZE)
                    throw new Ad7747Exception("Address too large");
                loadImage[ad] = (byte)value;
                loadMap[ad >> 5] |= 1 << (ad & 0x1f);
            }
            checkSum += value;
        }
        if ((checkSum & 0xff) != 0)
            throw new Ad7747Exception("Checksum failed");

        if (data[posn] == 13 || data[posn] == 10) {
            posn++;
            if (data[posn] == 13 || data[posn] == 10) posn++;
        }

        return (type == 0) ? posn : -posn;
    }


   /**
    ***************************************************************************
    **
    **  Decodes two hexadecimal bytes into an integer
    **
    **  @param  data  Array containing the data bytes
    **
    **  @param  offs  The offset to the bytes to decode
    **
    ***************************************************************************
    */
    private int decodeHex(byte[] data, int offs) throws UsbException
    {
        try {
            return Integer.parseInt(new String(data, offs, 2), 16);
        }
        catch (Exception e) {
            throw new Ad7747Exception("Invalid hexadecimal number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Loads memory from the cache
    **
    ***************************************************************************
    */
    private void loadCache() throws UsbException
    {
        setReset(true);

        int start = 0;
        boolean skip = true;
        for (int j = 0; j < loadImage.length; j++) {
            boolean set = (loadMap[j >> 5] & (1 << (j & 0x1f))) != 0;
            if (skip ^ set) continue;
            if (skip)
                start = j;
            else {
                int leng = j - start;
                while (leng > 0) {
                    int wLeng = leng > PACKET_SIZE ? PACKET_SIZE : leng;
                    writeMemory(start, loadImage, start, wLeng);
                    leng -= wLeng;
                    start += wLeng;
                }
            }
            skip = !skip;
        }

        setReset(false);
    }


   /**
    ***************************************************************************
    **
    **  Checks whether memory contents matches the cache
    **
    ***************************************************************************
    */
    private boolean checkCache() throws UsbException
    {
        setReset(true);

        boolean match = true, skip = true;
        int start = 0;
        for (int j = 0; j < loadImage.length; j++) {
            boolean set = (loadMap[j >> 5] & (1 << (j & 0x1f))) != 0;
            if (skip ^ set) continue;
            if (skip)
                start = j;
            else {
                int leng = j - start;
                if (leng > PACKET_SIZE) leng = PACKET_SIZE;
                byte[] data = new byte[leng];
                readMemory(start, data, 1000);
                for (int k = 0; k < leng; k++) {
                    if (data[k] != loadImage[start + k])
                        match = false;
                }
                break;
            }
            skip = !skip;
        }

        setReset(false);

        return match;
    }


   /**
    ***************************************************************************
    **
    **  Waits for the board to become ready after a conversion
    **
    **  @param  mask  Mask of bits that must be clear
    **
    ***************************************************************************
    */
    private void waitReady(int mask) throws UsbException
    {
        int count;
        for (count = 0;
             count < 50 && (readRegister(STATUS_REG) & mask) != 0; count++) {
            try {
                Thread.sleep(5);
            }
            catch (InterruptedException e) {
            }
        }
        if (count >= 50)
            throw new Ad7747Exception("Conversion timed out");
    }

}
