package org.lsst.ccs.drivers.abb;

import org.lsst.ccs.drivers.modbus.Modbus;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling an ABB CMS device.
 *
 *  @author  Owen Saxton
 */
public class Cms extends Modbus {

    /**
     *  Public constants.
     */
    public static final int
        DEFAULT_BAUD_RATE = 19200,
        NUM_SENSORS       = 64;

    /**
     *  Private data.
     */
    private static final short
        ADDR_TRIGGER   = 0x3010,
        ADDR_SHOW      = 0x3011,
        CMD_TRIGGER    = 0x0010,
        CMD_RESET      = 0x0001,
        CMD_BLINK      = 0x1000,
        ZERO = 0,
        ONE = 1;

    private static final int
        BASE_ADDR_INST    = 0x0000,
        BASE_ADDR_MIN     = 0x0400,
        BASE_ADDR_MAX     = 0x0800,
        BASE_ADDR_HOLD    = 0x0c00,
        OFF_ADDR_RMS      = 0x0000,
        OFF_ADDR_AC       = 0x0100,
        OFF_ADDR_DC       = 0x0200;

    /**
     *  Constructor.
     */
    public Cms()
    {
        setOptions(Option.NO_NET);
        setDefaultBaud(DEFAULT_BAUD_RATE);
    }


    /**
     *  Opens a connection to the device.
     *
     *  @param  type   The type of connection to make (ignored)
     *  @param  ident  The serial device name
     *  @param  baud   The baud rate
     *  @param  dchar  The data characteristics (ignored)
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType type, String ident, int baud, int dchar)
        throws DriverException
    {
        super.open(ConnType.SERIAL, ident, baud, makeDataCharacteristics(DataBits.EIGHT, StopBits.TWO, Parity.EVEN, FlowCtrl.NONE));
        super.setAddressMode(true);
    }


    /**
     *  Opens a connection to the device.
     *
     *  @param  ident  The serial device name
     *  @param  baud   The baud rate
     *  @throws  DriverException
     */
    public void open(String ident, int baud) throws DriverException
    {
        openSerial(ident, baud);
    }


    /**
     *  Sets whether entity numbers are direct addresses.
     *
     *  This routine is a no-op since address mode is always set and should
     *  not be changed.
     *
     *  @param  mode  Whether or not address mode is to be set
     */
    @Override
    public void setAddressMode(boolean mode)
    {
    }


    /**
     *  Reads an RMS current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readRMS(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_INST + OFF_ADDR_RMS + sensor - 1);
    }


    /**
     *  Reads an AC current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readAC(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_INST + OFF_ADDR_AC + sensor - 1);
    }


    /**
     *  Reads a DC current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readDC(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_INST + OFF_ADDR_DC + sensor - 1);
    }


    /**
     *  Reads a maximum RMS current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readMaxRMS(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_MAX + OFF_ADDR_RMS + sensor - 1);
    }


    /**
     *  Reads a maximum AC current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readMaxAC(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_MAX + OFF_ADDR_AC + sensor - 1);
    }


    /**
     *  Reads a maximum DC current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readMaxDC(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_MAX + OFF_ADDR_DC + sensor - 1);
    }


    /**
     *  Reads a minimum RMS current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readMinRMS(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_MIN + OFF_ADDR_RMS + sensor - 1);
    }


    /**
     *  Reads a minimum AC current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readMinAC(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_MIN + OFF_ADDR_AC + sensor - 1);
    }


    /**
     *  Reads a minimum DC current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readMinDC(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_MIN + OFF_ADDR_DC + sensor - 1);
    }


    /**
     *  Reads a held RMS current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readHeldRMS(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_HOLD + OFF_ADDR_RMS + sensor - 1);
    }


    /**
     *  Reads a held AC current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readHeldAC(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_HOLD + OFF_ADDR_AC + sensor - 1);
    }


    /**
     *  Reads a held DC current value
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readHeldDC(int bAddr, int sensor) throws DriverException
    {
        checkSensor(sensor);
        return readSensor(bAddr, BASE_ADDR_HOLD + OFF_ADDR_DC + sensor - 1);
    }


    /**
     *  Resets all minimum and maximum values
     *
     *  @param  bAddr   The Modbus address
     *  @throws  DriverException
     */
    public void resetExtrema(int bAddr) throws DriverException
    {
        writeRegister((short)bAddr, ADDR_TRIGGER, CMD_RESET);
    }


    /**
     *  Triggers a hold operation
     *
     *  @param  bAddr   The Modbus address
     *  @throws  DriverException
     */
    public void triggerHold(int bAddr) throws DriverException
    {
        writeRegister((short)bAddr, ADDR_TRIGGER, CMD_TRIGGER);
    }


    /**
     *  Sets the show state of a sensor
     *
     *  @param  bAddr   The Modbus address
     *  @param  sensor  The sensor number (1 - 64)
     *  @param  show    Whether to show the sensor
     *  @throws  DriverException
     */
    public void showSensor(int bAddr, int sensor, boolean show) throws DriverException
    {
        checkSensor(sensor);
        writeRegister((short)bAddr, ADDR_SHOW, (short)((show ? CMD_BLINK : 0) | sensor));
    }


    /**
     *  Reads a current value
     *
     *  @param  bAddr  The Modbus address
     *  @param  addr   The module address
     *  @return  The current value
     *  @throws  DriverException
     */
    private double readSensor(int bAddr, int addr) throws DriverException
    {
        return readRegisters((short)bAddr, (short)addr, ONE)[0] * 0.01;
    }


    /**
     *  Checks a sensor number for validity.
     *
     *  @param  sensor  The sensor number
     *  @throws  DriverException
     */
    private void checkSensor(int sensor) throws DriverException
    {
        if (sensor <= 0 || sensor > NUM_SENSORS) {
            throw new DriverException("Invalid sensor number");
        }
    }

}
