package org.lsst.ccs.drivers.abb;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.modbus.TestModbus;
import java.util.LinkedHashMap;
import java.util.Map;
import org.lsst.ccs.drivers.modbus.Modbus;

/**
 *  Program to test the ABB CMS device driver
 * 
 *  @author Owen Saxton
 */
public class TestCms extends TestModbus {

    private final Cms cms;


   /**
    *  Constructor
    */
    public TestCms()
    {
        super(new Cms());
        cms = (Cms)mod;
    }


   /**
    *  Opens a connection to the device.
    *
    *  @param  type   The type of connection to make
    *  @param  ident  The device identifier:
    *                   serial number for FTDI device;
    *                   port name for serial
    *  @param  baud   The baud rate, or 0 for default (19200)
    *  @throws  DriverException
    */
    @Override
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     Modbus.ConnType type,
                     @Argument(name="ident", description="Device identifier")
                     String ident,
                     @Argument(name="baud", description="Baud rate")
                     int baud) throws DriverException
    {
        cms.open(type, ident, baud);
    }


   /**
    *  Shows a DC current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readDC", description="Show DC current")
    public double readDC(@Argument(name="baddr", description="Modbus address")
                         int bAddr,
                         @Argument(name="sensor", description="Sensor number")
                         int sensor) throws DriverException
    {
        return cms.readDC(bAddr, sensor);
    }


   /**
    *  Shows an AC current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readAC", description="Show AC current")
    public double readAC(@Argument(name="baddr", description="Modbus address")
                         int bAddr,
                         @Argument(name="sensor", description="Sensor number")
                         int sensor) throws DriverException
    {
        return cms.readAC(bAddr, sensor);
    }


   /**
    *  Shows an RMS current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readRMS", description="Show RMS current")
    public double readRMS(@Argument(name="baddr", description="Modbus address")
                          int bAddr,
                          @Argument(name="sensor", description="Sensor number")
                          int sensor) throws DriverException
    {
        return cms.readRMS(bAddr, sensor);
    }


   /**
    *  Shows the maximum of a DC current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readMaxDC", description="Show maximum DC current")
    public double readMaxDC(@Argument(name="baddr", description="Modbus address")
                            int bAddr,
                            @Argument(name="sensor", description="Sensor number")
                            int sensor) throws DriverException
    {
        return cms.readMaxDC(bAddr, sensor);
    }


   /**
    *  Shows the maximum of an AC current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readMaxAC", description="Show maximum AC current")
    public double readMaxAC(@Argument(name="baddr", description="Modbus address")
                           int bAddr,
                           @Argument(name="sensor", description="Sensor number")
                           int sensor) throws DriverException
    {
        return cms.readMaxAC(bAddr, sensor);
    }


   /**
    *  Shows the maximum of an RMS current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readMaxRMS", description="Show maximum RMS current")
    public double readMaxRMS(@Argument(name="baddr", description="Modbus address")
                             int bAddr,
                             @Argument(name="sensor", description="Sensor number")
                             int sensor) throws DriverException
    {
        return cms.readMaxRMS(bAddr, sensor);
    }


   /**
    *  Shows the minimum of a DC current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readMinDC", description="Show minimum DC current")
    public double readMinDC(@Argument(name="baddr", description="Modbus address")
                            int bAddr,
                            @Argument(name="sensor", description="Sensor number")
                            int sensor) throws DriverException
    {
        return cms.readMinDC(bAddr, sensor);
    }


   /**
    *  Shows the minimum of an AC current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readMinAC", description="Show minimum AC current")
    public double readMinAC(@Argument(name="baddr", description="Modbus address")
                           int bAddr,
                           @Argument(name="sensor", description="Sensor number")
                           int sensor) throws DriverException
    {
        return cms.readMinAC(bAddr, sensor);
    }


   /**
    *  Shows the minimum of an RMS current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readMinRMS", description="Show minimum RMS current")
    public double readMinRMS(@Argument(name="baddr", description="Modbus address")
                             int bAddr,
                             @Argument(name="sensor", description="Sensor number")
                             int sensor) throws DriverException
    {
        return cms.readMinRMS(bAddr, sensor);
    }


   /**
    *  Shows the held DC current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readHeldDC", description="Show held DC current")
    public double readHeldDC(@Argument(name="baddr", description="Modbus address")
                             int bAddr,
                             @Argument(name="sensor", description="Sensor number")
                             int sensor) throws DriverException
    {
        return cms.readHeldDC(bAddr, sensor);
    }


   /**
    *  Shows the held AC current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readHeldAC", description="Show held AC current")
    public double readHeldAC(@Argument(name="baddr", description="Modbus address")
                            int bAddr,
                            @Argument(name="sensor", description="Sensor number")
                            int sensor) throws DriverException
    {
        return cms.readHeldAC(bAddr, sensor);
    }


   /**
    *  Shows the held RMS current.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readHeldRMS", description="Show held RMS current")
    public double readHeldRMS(@Argument(name="baddr", description="Modbus address")
                              int bAddr,
                              @Argument(name="sensor", description="Sensor number")
                              int sensor) throws DriverException
    {
        return cms.readHeldRMS(bAddr, sensor);
    }


   /**
    *  Shows all DC currents.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readAllDC", description="Show all DC currents")
    public Map readAllDC(@Argument(name="baddr", description="Modbus address")
                         int bAddr,
                         @Argument(name="sensor", description="Sensor number")
                         int sensor) throws DriverException
    {
        Map<String, Double> values = new LinkedHashMap<>();
        values.put("Curr", cms.readDC(bAddr, sensor));
        values.put("Min", cms.readMinDC(bAddr, sensor));
        values.put("Max", cms.readMaxDC(bAddr, sensor));
        values.put("Held", cms.readHeldDC(bAddr, sensor));
        return values;
    }


   /**
    *  Shows all AC currents.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readAllAC", description="Show all AC currents")
    public Map readAllAC(@Argument(name="baddr", description="Modbus address")
                         int bAddr,
                         @Argument(name="sensor", description="Sensor number")
                         int sensor) throws DriverException
    {
        Map<String, Double> values = new LinkedHashMap<>();
        values.put("Curr", cms.readAC(bAddr, sensor));
        values.put("Min", cms.readMinAC(bAddr, sensor));
        values.put("Max", cms.readMaxAC(bAddr, sensor));
        values.put("Held", cms.readHeldAC(bAddr, sensor));
        return values;
    }


   /**
    *  Shows all RMS currents.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @return The current reading
    *  @throws  DriverException
    */
    @Command(name="readAllRMS", description="Show all RMS currents")
    public Map readAllRMS(@Argument(name="baddr", description="Modbus address")
                          int bAddr,
                          @Argument(name="sensor", description="Sensor number")
                          int sensor) throws DriverException
    {
        Map<String, Double> values = new LinkedHashMap<>();
        values.put("Curr", cms.readRMS(bAddr, sensor));
        values.put("Min", cms.readMinRMS(bAddr, sensor));
        values.put("Max", cms.readMaxRMS(bAddr, sensor));
        values.put("Held", cms.readHeldRMS(bAddr, sensor));
        return values;
    }


   /**
    *  Sets the show state of a sensor.
    * 
    *  @param  bAddr   The Modbus address
    *  @param  sensor  The sensor number
    *  @param  show    Whether to show the sensor
    *  @throws  DriverException
    */
    @Command(name="showSensor", description="Set whether sensor shows itself")
    public void showSensor(@Argument(name="baddr", description="Modbus address")
                           int bAddr,
                           @Argument(name="sensor", description="Sensor number")
                           int sensor,
                           @Argument(name="show", description="Whether to show")
                           boolean show) throws DriverException
    {
        cms.showSensor(bAddr, sensor, show);
    }


   /**
    *  Triggers a hold operation.
    * 
    *  @param  bAddr   The Modbus address
    *  @throws  DriverException
    */
    @Command(name="triggerHold", description="Trigger hold operation")
    public void triggerHold(@Argument(name="baddr", description="Modbus address")
                            int bAddr) throws DriverException
    {
        cms.triggerHold(bAddr);
    }


   /**
    *  Resets minimum and maximum values.
    * 
    *  @param  bAddr   The Modbus address
    *  @throws  DriverException
    */
    @Command(name="resetExtrema", description="Reset minima and maxima")
    public void resetExtrema(@Argument(name="baddr", description="Modbus address")
                             int bAddr) throws DriverException
    {
        cms.resetExtrema(bAddr);
    }

}
