package org.lsst.ccs.daq.utilities;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.utilities.image.FitsHeadersSpecificationsBuilder;
import org.lsst.ccs.utilities.image.HeaderSpecification;


/**
 * An instance of a <TT>FitsServiceConfiguration</TT> object is responsible
 * for configuring all the FitsServices that are running within an application.
 * 
 * An instance of this class must be present in the groovy tree.
 * 
 * All instances of FitsServices will pick up the FitsServiceConfiguration
 * object via the LookupField annotation.
 *
 * This class requires a list of header files provided via the
 * "headerFilesList" configuration parameter.
 *
 * This list contains a list of file names to be loaded. Each file name can be
 * specified as "fileName:fitsHeaderName". For example "ts8_primary:primary" to
 * assign the ts8_primary.spec file to the primary header.
 *
 * If the name is already the name of the header, then the file name alone will
 * suffice.
 *
 * @author The LSST CCS Team
 */
public class FitsServiceConfiguration implements HasLifecycle {

    @ConfigurationParameter(isFinal = true, maxLength = 15, description = "List of spec files to be read by this FITS service", units="unitless")
    private final List<String> headerFilesList = new ArrayList<>();

    private static final Logger LOGGER = Logger.getLogger(FitsServiceConfiguration.class.getName());

    private final FitsHeadersSpecificationsBuilder headerSpecsBuilder = new FitsHeadersSpecificationsBuilder();

    @ConfigurationParameterChanger(propertyName = "headerFilesList")
    public void setHeaderFilesList(List<String> list) {
        LOGGER.log(Level.FINE, "Configuring Fits Header Service with {0}", list);
        headerFilesList.clear();
        headerFilesList.addAll(list);
    }

    @Override
    public void init() {
        for ( String headerFile : headerFilesList ) {
            int index = headerFile.indexOf(":");
            String fileName = index >= 0 ? headerFile.substring(0,index) : headerFile;
            String headerName = index >=0 ? headerFile.substring(index+1) : null;
            if ( headerName == null ) {
                headerName = fileName.replace(".spec", "");
            }
            LOGGER.log(Level.FINE, "Loading spec file: {0} for extension {1}", new Object[]{fileName, headerName});
            headerSpecsBuilder.addSpecFile(fileName, headerName);
        }
    }
    
    
    
    Map<String,HeaderSpecification> getHeaderSpecificationMap() {
        Map<String,HeaderSpecification> resMap = new LinkedHashMap<>();
        for (Entry<String,HeaderSpecification> e : headerSpecsBuilder.getHeaderSpecifications().entrySet()) {
            resMap.put(e.getKey(), new HeaderSpecification(e.getValue()));
        }
        return resMap;
    }

    @Command(type = Command.CommandType.QUERY, category = Command.CommandCategory.SYSTEM)
    public String printHeaderSpecifications() {
        StringBuilder sb = new StringBuilder("Header specifications\n");
        Map<String, HeaderSpecification> config = headerSpecsBuilder.getHeaderSpecifications();
        for (String header : config.keySet()) {
            sb.append("***************************\n");
            sb.append("     Header: ").append(header).append("\n");
            sb.append("***************************\n");
            HeaderSpecification spec = config.get(header);
            for (HeaderSpecification.HeaderLine line : spec.getHeaders()) {
                sb.append("## ").append(line.getKeyword()).append(" ").append(line.getMetaName()).append(" ").append(line.getDataType()).append(" ").append(line.getComment()).append("\n");
            }
        }
        return sb.toString();
    }
    
}
