package org.lsst.ccs.daq.utilities;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.ccd.WFCCDType;
import org.lsst.ccs.utilities.image.FitsHeaderMetadataProvider;
import org.lsst.ccs.utilities.image.HeaderSpecification;
import org.lsst.ccs.utilities.image.MetaDataSet;

/**
 * The subcomponent of FITS service that deals with a single REB
 * 
 * @author The LSST CCS Team
 */
class PerRebFitsService implements FitsServiceInterface {

    private static final Logger LOGGER = Logger.getLogger(PerRebFitsService.class.getName());
    private final Map<String, MetaDataSet> geometryReplacementMetaDataSetMap = new HashMap<>();
    
    //Header keyword values related objects
    private final Map<String, Map<String, HeaderKeywordValue>> headerKeywordValuesMap = new HashMap<>();

    private final Reb reb;
    private final AgentStatusAggregatorService aggregatorService;
    private final String agentName;
    private final Map<String, HeaderSpecification> headerSpecificationsMap;


    PerRebFitsService(String agentName, AgentStatusAggregatorService aggregatorService, Map<String, HeaderSpecification> headerSpecificationsMap, Reb reb) {
        LOGGER.log(Level.FINE, "Configuring Fits Header for geometry {0}", reb.getUniqueId());

        this.reb = reb;
        this.aggregatorService = aggregatorService;
        this.agentName = agentName;
        this.headerSpecificationsMap = headerSpecificationsMap;

        for (String headerName : headerSpecificationsMap.keySet()) {
            if (!headerKeywordValuesMap.containsKey(headerName)) {
                headerKeywordValuesMap.put(headerName, new HashMap<>());
            }
        }

        fillHeaderKeywordMapsForReb(reb);

        String raftName = "";
        if ( reb.getRaft()!= null ) {
            raftName = reb.getRaft().getName();
        }
        String rebName = reb.getName();
        geometryReplacementMetaDataSetMap.clear();

        MetaDataSet mds = new MetaDataSet();
        mds.addMetaData("", "RAFT", raftName);
        mds.addMetaData("", "REB", rebName);        
        LOGGER.log(Level.FINE, "Adding default replacements for REB and RAFT: {0} {1}", new Object[]{mds.getValue("REB"), mds.getValue("RAFT")});
         //Add the CCD and Reb Replacements for each CCD unique id
        for ( CCD ccd : reb.getCCDs()) {

            MetaDataSet ccd_mds = new MetaDataSet();
            ccd_mds.addMetaDataSet(mds);
            
            String ccdName = ccd.getName();
            String ccdTrending = ccdName;
            if ( ccd.getType() instanceof WFCCDType ) {            
                ccdTrending = "SW";
            }
            ccd_mds.addMetaData("", "CCD", ccdName);
            ccd_mds.addMetaData("", "CCD_TRENDING", ccdTrending);
            ccd_mds.addMetaData("", "SENSOR", ccd.getSerialPosition());
            geometryReplacementMetaDataSetMap.put(ccd.getUniqueId(),ccd_mds);            
            LOGGER.log(Level.FINE, "Adding default replacements for CCD: {0}: {1}", new Object[]{ccd.getUniqueId(), ccd_mds.getValue("CCD")});
        }
    }

    public Reb getReb() {
        return reb;
    }
    
    /**
     * These methods to initialize the header keywords are to guarantee that
     * the provided header keywords are consistent with the provided geometry.
     *
     */

    private void fillHeaderKeywordMapsForReb(Reb reb) {
        for (CCD ccd : reb.getCCDs()) {
            if (!headerKeywordValuesMap.containsKey(ccd.getUniqueId())) {
                headerKeywordValuesMap.put(ccd.getUniqueId(), new HashMap<>());
            }
        }
    }
    
    
    private Map<String, HeaderKeywordValue> getHeaderKeywordMapForHeader(String headerName) {
        if (!headerKeywordValuesMap.containsKey(headerName)) {
            LOGGER.log(Level.WARNING, "The FitsService is not configured to store data for header {0}", headerName);
            headerKeywordValuesMap.put(headerName, new HashMap<>());
        }
        return headerKeywordValuesMap.get(headerName);
    }

    @Override
    public FitsServiceFitsHeaderMetadataProvider getFitsHeaderMetadataProvider(String source) {
        return new FitsServiceFitsHeaderMetadataProvider(source);
    }

    @Override
    public Map<String, HeaderSpecification> getHeaderSpecificationMap() {
        return headerSpecificationsMap;
    }
    
    // Package private class used to set Header Keyword values.
    private class HeaderKeywordValue {

        private final Object value;
        private final boolean sticky;

        HeaderKeywordValue(Object value, boolean sticky) {
            this.value = value;
            this.sticky = sticky;
        }

        Object getValue() {
            return value;
        }

        boolean isSticky() {
            return sticky;
        }
    }
    
    private class FitsServiceFitsHeaderMetadataProvider implements FitsHeaderMetadataProvider {

        private final String ccdUniqueId;
        private final Map<String, MetaDataSet> storedMetaDataSets = new HashMap<>();

        public FitsServiceFitsHeaderMetadataProvider(String ccdUniqueId) {
            this.ccdUniqueId = ccdUniqueId;
        }

        private MetaDataSet getMetaDataSet(String key) {
            MetaDataSet result = storedMetaDataSets.get(key);
            if (result == null) {
                if (key.equals("statusAggregator")) {
                    result = getStatusAggregatorMetaDataSet();
                    result.addMetaData("", "AGENT_NAME", agentName);
                } else if (key.equals("geometry")) {
                    result = geometryReplacementMetaDataSetMap.get(ccdUniqueId);
                } else {
                    result = buildMetaDataSetForExtension(key);
                }
                storedMetaDataSets.put(key, result);
            }
            return result;
        }

        @Override
        public MetaDataSet getAdditionalExtendedHeaderMetadata(String extendedKeyword) {
            MetaDataSet r = buildMetaDataSetForExtension(extendedKeyword);
            r.addMetaDataSet(getMetaDataSet(ccdUniqueId));
            r.addMetaDataSet(getMetaDataSet("statusAggregator"));
            r.addMetaDataSet(getMetaDataSet("geometry"));
            return r;
        }

        @Override
        public MetaDataSet getDataExtendedHeaderMetadata(String imageExtensionName) {
            MetaDataSet r = new MetaDataSet();
            r.addMetaDataSet(getMetaDataSet(ccdUniqueId));
            r.addMetaDataSet(getMetaDataSet("statusAggregator"));
            r.addMetaDataSet(getMetaDataSet("geometry"));
            return r;
        }

        @Override
        public MetaDataSet getPrimaryHeaderMetadata() {
            MetaDataSet r = new MetaDataSet();
            r.addMetaDataSet(getMetaDataSet("primary"));
            r.addMetaDataSet(getMetaDataSet(ccdUniqueId));
            r.addMetaDataSet(getMetaDataSet("statusAggregator"));
            r.addMetaDataSet(getMetaDataSet("geometry"));
            return r;
        }

    }
    
    private MetaDataSet buildMetaDataSetForExtension(String extension) {
        Map<String, HeaderKeywordValue> headerKeywordMap = getHeaderKeywordMapForHeader(extension);
        Map<String, Object> valuesMap = new HashMap<>();
        for (String headerKeyword : headerKeywordMap.keySet()) {
            valuesMap.put(headerKeyword, headerKeywordMap.get(headerKeyword).getValue());
        }

        MetaDataSet result = new MetaDataSet();
        result.addMetaDataMap(extension, valuesMap);
        return result;
    }

    
    private MetaDataSet getStatusAggregatorMetaDataSet() {
        MetaDataSet m = new MetaDataSet();
        Map<String, Object> aggrMap = aggregatorService.getAllLast();
        m.addMetaDataMap("StatusAggregator", aggrMap);
        return m;
    }

    
}
