package org.lsst.ccs.daq.utilities;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.Agent;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.image.HeaderSpecification;
import org.lsst.ccs.utilities.location.Location;

/**
 * This service is designed to be a singleton, instantiated from groovy, and
 * with a single FitsServiceConfiguration as a child. REBs can be dynamically
 * added and removed from this service at any time. It is designed to delegate most
 * functionality to one or more PerRebFitsService instances.
 *
 * @author The LSST CCS Team
 */
public class FitsService implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, FitsServiceConfiguration> configs = new LinkedHashMap<>();
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStatusAggregatorService aggregatorService;

    private final Map<Location, PerRebFitsService> perRebFitsServices = new ConcurrentHashMap<>();
    private final Map<Reb, String> rebs = new HashMap<>();
    private boolean hasBeenStarted = false;


    @Override
    public void start() {
        for (Map.Entry<Reb, String> reb : rebs.entrySet()) {
            createPerRebFitsService(reb.getKey(), reb.getValue());
        }
        hasBeenStarted = true;
    }

    public void addReb(Reb reb, String configName) {
        rebs.put(reb, configName);
        if (hasBeenStarted) {
            createPerRebFitsService(reb, configName);
        }
    }

    public void clearAllRebs() {
        perRebFitsServices.clear();
        rebs.clear();
    }
    
    public FitsServiceInterface getFitsServiceForReb(Reb reb) {
        return perRebFitsServices.get(reb.getLocation());
    }
    
    public Map<String, HeaderSpecification> getHeaderSpecificationMap(String configName) {
        return getConfigurationByName(configName).getHeaderSpecificationMap();
    }

    private FitsServiceConfiguration getConfigurationByName(String configName) {
        FitsServiceConfiguration config = configs.get(configName);
        if (config == null) {
            throw new RuntimeException("Unknown config name "+configName);
        }
        return config;
    }
    
    private void createPerRebFitsService(Reb reb, String configName) {
        PerRebFitsService perRebFitsService = new PerRebFitsService(agent.getName(), aggregatorService, getConfigurationByName(configName).getHeaderSpecificationMap(), reb);
        perRebFitsServices.put(reb.getLocation(), perRebFitsService);
    }
}
