package org.lsst.ccs.imagenaming.service;

import java.sql.SQLException;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.imagenaming.Controller;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.imagenaming.ImageNameAssigner;
import org.lsst.ccs.imagenaming.Source;

/**
 * A configurable service for fetching image names.
 *
 * @author tonyj
 */
@SuppressWarnings("FieldMayBeFinal")
public class ImageNameService implements HasLifecycle, ConfigurationBulkChangeHandler {

    @ConfigurationParameter(isFinal = true, units="unitless", description="Image Name Database URL")
    private volatile String dbURL;

    //We have here two variables to represent the Controller;
    //The String variable is used so that it can be used by the configuration
    //service. The Controller variable is for internal use.
    @ConfigurationParameter(isFinal = true, name = "controller", description = "The controller of this image name service (CCS, OCS, etc)", units = "unitless")
    private volatile String controllerStr = Controller.CCS.name();
    private Controller controller = Controller.CCS;

    //We have here two variables to represent the Source;
    //The String variable is used so that it can be used by the configuration
    //service. The Source variable is for internal use.
    @ConfigurationParameter(isFinal = true, name = "source", description = "The source of this image name service MainCamera, ComCam, AuxTel, TestStand", units = "unitless")
    private volatile String sourceStr;
    private Source source;

    @ConfigurationParameter(isFinal = true, description = "The timezone used to assign the date (e.g. UTC)", units = "unitless")
    private volatile String timeZoneId = "UTC";
    private TimeZone timeZone = TimeZone.getTimeZone("UTC");

    @ConfigurationParameter(isFinal = true, description = "The offset from midnight when the data transition occurs (e.g. 12 hours)", units = "unitless")
    private volatile Duration offset = Duration.ofHours(12);
    
    private Instant testTime;
    private ImageNameAssigner imageNameAssigner;
    
    /**
     * Create a test instance of the ImageNameService
     * @param dbURL The dbURL to use
     * @param controller The controller to use
     * @param source The source to use
     * @param timeZoneId The timeZoneId to use
     * @param offset The offset to use
     * @param testTime A fake time to use to make tests more reproducible
     * @return The created and initialized ImageNameService
     */
    public static ImageNameService testInstance(String dbURL, Controller controller, Source source, String timeZoneId, Duration offset, Instant testTime) {
        ImageNameService imageNameService = new ImageNameService();
        imageNameService.dbURL = dbURL;
        imageNameService.controller = controller;
        imageNameService.source = source;
        imageNameService.timeZoneId = timeZoneId;
        imageNameService.offset = offset;
        imageNameService.testTime = testTime;
        imageNameService.postInit();
        return imageNameService;
    }

    @Override
    public void postInit() {
        if (controller == null || dbURL == null || source == null) {
            throw new RuntimeException("Invalid configuration for ImageNameService");
        }
        checkTimeZoneId(timeZoneId);
        timeZone = TimeZone.getTimeZone(timeZoneId);
        imageNameAssigner = new ImageNameAssigner(dbURL);
    }

    @Override
    public void shutdown() {
        try {
            imageNameAssigner.close();
        } catch (SQLException ex) {
            throw new RuntimeException("Error closing database connection for image naming", ex);
        }
    }

    @ConfigurationParameterChanger(propertyName = "source")
    public void setSource(String source) {
        this.source = Source.valueOf(source);
        sourceStr = source;
    }
    
    @ConfigurationParameterChanger(propertyName = "controller")
    public void setController(String controller) {
        this.controller = Controller.valueOf(controller);
        controllerStr = controller;
    }
    
    public ImageName getImageName() {
        try {
            return imageNameAssigner.assignImageName(source, controller, timeZone, offset, testTime);
        } catch (SQLException ex) {
            throw new RuntimeException("Error accessing image name database", ex);
        }
    }

    public List<ImageName> getImageNames(int n) {
        try {
            return imageNameAssigner.assignImageNames(source, controller, timeZone, offset, n, testTime);
        } catch (SQLException ex) {
            throw new RuntimeException("Error accessing image name database", ex);
        }
    }

    @Override
    public void setParameterBulk(Map<String, Object> parametersView) {
        if (parametersView.containsKey("timeZoneId")) {
            timeZone = TimeZone.getTimeZone(parametersView.get("timeZoneId").toString());
        }
    }

    @Override
    public void validateBulkChange(Map<String, Object> parametersView) {
        if (parametersView.containsKey("timeZoneId")) {
            checkTimeZoneId(parametersView.get("timeZoneId").toString());
        }
    }

    private void checkTimeZoneId(String timeZoneId) throws IllegalArgumentException {
        if (!Arrays.asList(TimeZone.getAvailableIDs()).contains(timeZoneId)) {
            throw new IllegalArgumentException("Invalid time zone id " + timeZoneId);
        }
    }

}
