package org.lsst.ccs.daq.utilities;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.Geometry;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.ccd.WFCCDType;
import org.lsst.ccs.utilities.image.FitsHeaderMetadataProvider;
import org.lsst.ccs.utilities.image.HeaderSpecification;
import org.lsst.ccs.utilities.image.MetaDataSet;


/**
 * An <TT>AgentFitsHeaderService</TT> is responsible to write out fits files
 * for a given Reb.
 *
 * @author The LSST CCS Team
 */
public class FitsService implements HasLifecycle, StatusMessageListener {

    private final Map<String, MetaDataSet> geometryReplacementMetaDataSetMap = new HashMap<>();

    private Reb geometry;
    private String uniqueId = "notSet";

    private static final Logger LOGGER = Logger.getLogger(FitsService.class.getName());

    //Header keyword values related objects
    private final Map<String, Map<String, HeaderKeywordValue>> headerKeywordValuesMap = new HashMap<>();

    private Map<String,HeaderSpecification> headerSpecificationsMap;    
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private FitsServiceConfiguration config;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStatusAggregatorService aggregatorService;

    public static final String COMMON_HEADER_NAME = "all";

    private Boolean enableAtStart = null;
    
    @Override
    public void start() {        
        headerSpecificationsMap = config.getHeaderSpecificationMap();
        for ( String headerName : headerSpecificationsMap.keySet() ) {
            if ( !headerKeywordValuesMap.containsKey(headerName) ) {
                headerKeywordValuesMap.put(headerName, new HashMap<>());
            }      
        }
        headerKeywordValuesMap.put(COMMON_HEADER_NAME, new HashMap<>());
        if ( enableAtStart != null ) {
            setEnabled(enableAtStart);
        }
        
    }    
    //This method can be invoked from groovy. If that happens we have to postpone adding listeners 
    //to the start method.
    public void setEnabled(boolean enabled) {        
        if (agent != null) {
            LOGGER.log(Level.INFO, "{0} FitService for Reb {1}", new Object[]{enabled ? "Enabling" : "Disabling", geometry.getFullName()});
            if (enabled) {
                agent.getMessagingAccess().addStatusMessageListener(this, BusMessageFilterFactory.messageClass(StatusSubsystemData.class));
            } else {
                agent.getMessagingAccess().removeStatusMessageListener(this);
            }
        } else {
            enableAtStart = enabled;
        }
    }
    
    protected Reb getGeometry() {
        return geometry;
    }
    
    //This is to be used by groovy
    public void setReb(Reb reb) {
        setGeometry(reb);
    }
    
    public void setGeometry(Reb reb) {
        LOGGER.log(Level.FINE, "Configuring Fits Header for geometry {0}", reb.getUniqueId());
        
        if ( ! (reb instanceof Reb) ) {
            throw new RuntimeException("A FitsService must be configured for a Reb geometry");
        }

        this.geometry = (Reb)reb;
        this.uniqueId = geometry.getUniqueId();
        fillHeaderKeywordMapsForReb(geometry);

        String raftName = "";
        if ( geometry.getRaft()!= null ) {
            raftName = geometry.getRaft().getName();
        }
        String rebName = geometry.getName();
        geometryReplacementMetaDataSetMap.clear();

        MetaDataSet mds = new MetaDataSet();
        mds.addMetaData("", "RAFT", raftName);
        mds.addMetaData("", "REB", rebName);        
        LOGGER.log(Level.FINE, "Adding default replacements for REB and RAFT: {0} {1}", new Object[]{mds.getValue("REB"), mds.getValue("RAFT")});
         //Add the CCD and Reb Replacements for each CCD unique id
        for ( CCD ccd : geometry.getCCDs()) {

            MetaDataSet ccd_mds = new MetaDataSet();
            ccd_mds.addMetaDataSet(mds);
            
            String ccdName = ccd.getName();
            String ccdTrending = ccdName;
            if ( ccd.getType() instanceof WFCCDType ) {            
                ccdTrending = "SW";
            }
            ccd_mds.addMetaData("", "CCD", ccdName);
            ccd_mds.addMetaData("", "CCD_TRENDING", ccdTrending);
            geometryReplacementMetaDataSetMap.put(ccd.getUniqueId(),ccd_mds);            
            LOGGER.log(Level.FINE, "Adding default replacements for CCD: {0}: {1}", new Object[]{ccd.getUniqueId(), ccd_mds.getValue("CCD")});
        }
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {
        StatusSubsystemData d = (StatusSubsystemData) msg;
        if ( d.getDataKey().equals(FitsHeaderKeywordData.DATA_KEY) ) {
            FitsHeaderKeywordData fitsHeaderKeywordData = (FitsHeaderKeywordData) d.getObject().getValue();
            String headerKeywordDataId = fitsHeaderKeywordData.getDataId();
            if (headerKeywordDataId != null && !headerKeywordDataId.isEmpty() && !headerKeywordDataId.startsWith(uniqueId)) {
                return;
            }
            for (FitsHeaderKeywordData.HeaderKeywordValue value : fitsHeaderKeywordData.getHeaderKeywordValues()) {
                setHeaderKeywordValue(value);
            }

        }
    }

    
    public Map<String,HeaderSpecification> getHeaderSpecificationMap() {
        return headerSpecificationsMap;
    }

    /**
     * Set a key for the Primary header of all CCDs.
     *
     * @param headerKeywordName The name of the Header to set
     * @param headerKeywordValue The corresponding value
     * @param sticky Boolean value to specify if the provided Header keyword
     * value should be used across exposures. The default value is true. If
     * false is provided, the provided Header keyword value will be reset after
     * each exposure.
     */
    public void setHeaderKeywordValue(String headerKeywordName, Object headerKeywordValue, boolean sticky) {
        setHeaderKeywordValue(COMMON_HEADER_NAME,headerKeywordName, headerKeywordValue, sticky);
    }
    public void setHeaderKeywordValue(String headerKeywordName, Object headerKeywordValue) {
        setHeaderKeywordValue(COMMON_HEADER_NAME, headerKeywordName, headerKeywordValue, false);
    }
    public void setHeaderKeywordValue(String headerName, String headerKeywordName, Object headerKeywordValue) {
        setHeaderKeywordValue(headerName, headerKeywordName, headerKeywordValue, false);
    }

    public void setHeaderKeywordValue(String headerName, String headerKeywordName, Object headerKeywordValue, boolean sticky) {
        getHeaderKeywordMapForHeader(headerName).put(headerKeywordName, new HeaderKeywordValue(headerKeywordValue, sticky));
    }

    private void setHeaderKeywordValue(FitsHeaderKeywordData.HeaderKeywordValue value) {
        setHeaderKeywordValue(value.getHeaderName(), value.getHeaderKeywordName(), value.getHeaderKeywordValue(), value.isSticky());
    }

    /**
     * These methods to initialize the header keywords are to guarantee that
     * the provided header keywords are consistent with the provided geometry.
     *
     */
    private void fillHeaderKeywordMaps(Geometry geometry) {
        if (geometry instanceof Raft) {
            Raft raft = (Raft) geometry;
            for (Reb reb : raft.getRebs()) {
                fillHeaderKeywordMapsForReb(reb);
            }
        } else {
            throw new RuntimeException("This class is currently designed to support a single Raft.");
        }
    }
    private void fillHeaderKeywordMapsForReb(Reb reb) {
        for (CCD ccd : reb.getCCDs()) {
            if (!headerKeywordValuesMap.containsKey(ccd.getUniqueId())) {
                headerKeywordValuesMap.put(ccd.getUniqueId(), new HashMap<>());
            }
        }
    }

    private Map<String,HeaderKeywordValue> getHeaderKeywordMapForHeader(String headerName) {
        if ( !headerKeywordValuesMap.containsKey(headerName) ) {
            LOGGER.log(Level.WARNING, "The FitsService is not configured to store data for header {0}", headerName);
            headerKeywordValuesMap.put(headerName, new HashMap<>());
        }
        return headerKeywordValuesMap.get(headerName);
    }

    private MetaDataSet buildMetaDataSetForExtension(String extension) {
        Map<String,HeaderKeywordValue> headerKeywordMap = getHeaderKeywordMapForHeader(extension);
        Map<String, Object> valuesMap = new HashMap<>();
        for (String headerKeyword : headerKeywordMap.keySet()) {
            valuesMap.put(headerKeyword, headerKeywordMap.get(headerKeyword).getValue());
        }
        
        MetaDataSet result = new MetaDataSet();
        result.addMetaDataMap(extension, valuesMap);
        return result;
    }

    /**
     * Method to clear the non-sticky header keyword values.
     *
     */
    public void clearNonStickyHeaderKeywordValues() {
        for (Map<String, HeaderKeywordValue> map : headerKeywordValuesMap.values()) {
            clearNonStickyHeaderKeywordValuesFromMap(map);
        }
    }

    private void clearNonStickyHeaderKeywordValuesFromMap(Map<String, HeaderKeywordValue> map) {
        Iterator<Map.Entry<String, HeaderKeywordValue>> iter = map.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<String, HeaderKeywordValue> entry = iter.next();
            if (!entry.getValue().isSticky()) {
                iter.remove();
            }
        }
    }
    
    public FitsHeaderMetadataProvider getFitsHeaderMetadataProvider(String source) {
        return new FitsServiceFitsHeaderMetadataProvider(source);
    }

    private class FitsServiceFitsHeaderMetadataProvider implements FitsHeaderMetadataProvider {
    
        private final String ccdUniqueId;
        private final Map<String,MetaDataSet> storedMetaDataSets = new HashMap<>();
        

        private MetaDataSet getMetaDataSet(String key) {
            MetaDataSet result = storedMetaDataSets.get(key);
            if ( result == null ) {
                if (key.equals("statusAggregator")) {
                    result = getStatusAggregatorMetaDataSet();
                    result.addMetaData("", "AGENT_NAME", agent.getName());                    
                } else if (key.equals("geometry")) {
                    result = geometryReplacementMetaDataSetMap.get(ccdUniqueId);
                } else {
                    result = buildMetaDataSetForExtension(key);
                }
                storedMetaDataSets.put(key, result);
            }
            return result;
        }

        public FitsServiceFitsHeaderMetadataProvider(String ccdUniqueId) {
            this.ccdUniqueId = ccdUniqueId;
        }
        
        @Override
        public MetaDataSet getAdditionalExtendedHeaderMetadata(String extendedKeyword) {
            MetaDataSet r = buildMetaDataSetForExtension(extendedKeyword);
            r.addMetaDataSet(getMetaDataSet("all"));
            r.addMetaDataSet(getMetaDataSet(ccdUniqueId));
            r.addMetaDataSet(getMetaDataSet("statusAggregator"));
            r.addMetaDataSet(getMetaDataSet("geometry"));
            return r;
        }

        @Override
        public MetaDataSet getDataExtendedHeaderMetadata(String imageExtensionName) {
            MetaDataSet r = new MetaDataSet();
            r.addMetaDataSet(getMetaDataSet("all"));
            r.addMetaDataSet(getMetaDataSet(ccdUniqueId));
            r.addMetaDataSet(getMetaDataSet("statusAggregator"));
            r.addMetaDataSet(getMetaDataSet("geometry"));
            return r;
        }

        @Override
        public MetaDataSet getPrimaryHeaderMetadata() {
            MetaDataSet r = new MetaDataSet();
            r.addMetaDataSet(getMetaDataSet("primary"));
            r.addMetaDataSet(getMetaDataSet("all"));
            r.addMetaDataSet(getMetaDataSet(ccdUniqueId));
            r.addMetaDataSet(getMetaDataSet("statusAggregator"));
            r.addMetaDataSet(getMetaDataSet("geometry"));
            return r;
        }
        
        

    }

    private MetaDataSet getStatusAggregatorMetaDataSet() {
        MetaDataSet m = new MetaDataSet();
        Map<String, Object> aggrMap = aggregatorService.getAllLast();
        m.addMetaDataMap("StatusAggregator", aggrMap);
        return m;
    }

    //Private class used to set Header Keyword values.
    private class HeaderKeywordValue {

        private final Object value;
        private final boolean sticky;

        HeaderKeywordValue(Object value, boolean sticky) {
            this.value = value;
            this.sticky = sticky;
        }

        Object getValue() {
            return value;
        }

        boolean isSticky() {
            return sticky;
        }
    }

    public static class FitsServiceKeyReplacement {
        private final String match;
        private final String replace;

        public FitsServiceKeyReplacement(String configStr) {
            int index = configStr.indexOf(":");
            match = "(?i)" + Pattern.quote(configStr.substring(0, index));
            replace = configStr.substring(index + 1);
        }

        public String manipulate(String input) {
            return input.replaceAll(match, replace);
        }

        @Override
        public String toString() {
            return match+":"+replace;
        }

    }

    
}
