package org.lsst.ccs.daq.utilities;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.utilities.FitsService.FitsServiceKeyReplacement;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.Geometry;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.ccd.WFCCDType;
import org.lsst.ccs.utilities.image.FitsHeaderMetadataProvider;
import org.lsst.ccs.utilities.image.FitsHeadersSpecificationsBuilder;
import org.lsst.ccs.utilities.image.HeaderSpecification;
import org.lsst.ccs.utilities.image.ImageSet;
import org.lsst.ccs.utilities.image.MetaDataSet;


/**
 * An <TT>AgentFitsHeaderService</TT> is responsible to load spec files for
 * writing out fits headers.
 *
 * This Service is activated when a list of header files is provided via the
 * "headerFilesList" configuration parameter.
 *
 * This list contains a list of file names to be loaded. Each file name can be
 * specified as "fileName:fitsHeaderName". For example "ts8_primary:primary" to
 * assign the ts8_primary.spec file to the primary header.
 *
 * If the name is already the name of the header, then the file name alone will
 * suffice.
 *
 *
 * @author The LSST CCS Team
 */
public class FitsService implements HasLifecycle, StatusMessageListener {

    @ConfigurationParameter(isFinal = true)
    private final List<String> headerFilesList = new ArrayList<>();

    private final Map<String, MetaDataSet> geometryReplacementMetaDataSetMap = new HashMap<>();

    private Reb geometry;
    private String uniqueId = "notSet";

    private static final Logger LOGGER = Logger.getLogger(FitsService.class.getName());

    private final FitsHeadersSpecificationsBuilder headerSpecsBuilder = new FitsHeadersSpecificationsBuilder();

    //Header keyword values related objects
    private final Map<String, Map<String, HeaderKeywordValue>> headerKeywordValuesMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStatusAggregatorService aggregatorService;

    public static final String COMMON_HEADER_NAME = "all";

    @ConfigurationParameterChanger(propertyName = "headerFilesList")
    public void setHeaderFilesList(List<String> list) {
        LOGGER.log(Level.FINE, "Configuring Fits Header Service with {0}", list);
        headerFilesList.clear();
        headerFilesList.addAll(list);
    }

    @Override
    public void postInit() {
        for ( String headerFile : headerFilesList ) {
            int index = headerFile.indexOf(":");
            String fileName = index >= 0 ? headerFile.substring(0,index) : headerFile;
            String headerName = index >=0 ? headerFile.substring(index+1) : null;
            if ( headerName == null ) {
                headerName = fileName.replace(".spec", "");
            }
            LOGGER.log(Level.FINE, "Loading spec file: {0} for extension {1}", new Object[]{fileName, headerName});
            headerSpecsBuilder.addSpecFile(fileName, headerName);
            if ( !headerKeywordValuesMap.containsKey(headerName) ) {
                headerKeywordValuesMap.put(headerName, new HashMap<>());
            }      
        }
        headerKeywordValuesMap.put(COMMON_HEADER_NAME, new HashMap<>());

    }    
    
    protected Reb getGeometry() {
        return geometry;
    }
    
    //This is to be used by groovy
    public void setReb(Reb reb) {
        setGeometry(reb);
    }
    
    public void setGeometry(Reb reb) {
        LOGGER.log(Level.FINE, "Configuring Fits Header for geometry {0}", reb.getUniqueId());
        
        if ( ! (reb instanceof Reb) ) {
            throw new RuntimeException("A FitsService must be configured for a Reb geometry");
        }

        this.geometry = (Reb)reb;
        this.uniqueId = geometry.getUniqueId();
        fillHeaderKeywordMapsForReb(geometry);

        String raftName = "";
        if ( geometry.getRaft()!= null ) {
            raftName = geometry.getRaft().getName();
        }
        String rebName = geometry.getName();
        geometryReplacementMetaDataSetMap.clear();

        MetaDataSet mds = new MetaDataSet();
        mds.addMetaData("", "RAFT", raftName);
        mds.addMetaData("", "REB", rebName);
        LOGGER.log(Level.FINE, "Adding default replacements for REB and RAFT: {0} {1}", new Object[]{mds.getValue("REB"), mds.getValue("RAFT")});
         //Add the CCD and Reb Replacements for each CCD unique id
        for ( CCD ccd : geometry.getCCDs()) {

            MetaDataSet ccd_mds = new MetaDataSet();
            ccd_mds.addMetaDataSet(mds);
            
            String ccdName = ccd.getName();
            String ccdTrending = ccdName;
            if ( ccd.getType() instanceof WFCCDType ) {            
                ccdTrending = "SW";
            }
            ccd_mds.addMetaData("", "CCD", ccdName);
            ccd_mds.addMetaData("", "CCD_TRENDING", ccdTrending);
            geometryReplacementMetaDataSetMap.put(ccd.getUniqueId(),ccd_mds);            
            LOGGER.log(Level.FINE, "Adding default replacements for CCD: {0}: {1}", new Object[]{ccd.getUniqueId(), ccd_mds.getValue("CCD")});
        }
    }

    @Override
    public void start() {
        agent.getMessagingAccess().addStatusMessageListener(this,BusMessageFilterFactory.messageClass(StatusSubsystemData.class));
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {
        StatusSubsystemData d = (StatusSubsystemData) msg;
        if ( d.getDataKey().equals(FitsHeaderKeywordData.DATA_KEY) ) {
            FitsHeaderKeywordData fitsHeaderKeywordData = (FitsHeaderKeywordData) d.getObject().getValue();
            String headerKeywordDataId = fitsHeaderKeywordData.getDataId();
            if (headerKeywordDataId != null && !headerKeywordDataId.isEmpty() && !headerKeywordDataId.startsWith(uniqueId)) {
                return;
            }
            for (FitsHeaderKeywordData.HeaderKeywordValue value : fitsHeaderKeywordData.getHeaderKeywordValues()) {
                setHeaderKeywordValue(value);
            }

        }
    }

    
    
    
    public Map<String,HeaderSpecification> getHeaderSpecificationMap() {
        return headerSpecsBuilder.getHeaderSpecifications();
    }

    @Command(type = Command.CommandType.QUERY, category = Command.CommandCategory.SYSTEM)
    public String printHeaderSpecifications() {
        StringBuilder sb = new StringBuilder("Header specifications\n");
        Map<String, HeaderSpecification> config = headerSpecsBuilder.getHeaderSpecifications();
        for (String header : config.keySet()) {
            sb.append("***************************\n");
            sb.append("     Header: ").append(header).append("\n");
            sb.append("***************************\n");
            HeaderSpecification spec = config.get(header);
            for (HeaderSpecification.HeaderLine line : spec.getHeaders()) {
                sb.append("## ").append(line.getKeyword()).append(" ").append(line.getMetaName()).append(" ").append(line.getDataType()).append(" ").append(line.getComment()).append("\n");
            }
        }
        return sb.toString();
    }

    
    /**
     * Set a key for the Primary header of all CCDs.
     *
     * @param headerKeywordName The name of the Header to set
     * @param headerKeywordValue The corresponding value
     * @param sticky Boolean value to specify if the provided Header keyword
     * value should be used across exposures. The default value is true. If
     * false is provided, the provided Header keyword value will be reset after
     * each exposure.
     */
    public void setHeaderKeywordValue(String headerKeywordName, Object headerKeywordValue, boolean sticky) {
        setHeaderKeywordValue(COMMON_HEADER_NAME,headerKeywordName, headerKeywordValue, sticky);
    }
    public void setHeaderKeywordValue(String headerKeywordName, Object headerKeywordValue) {
        setHeaderKeywordValue(COMMON_HEADER_NAME, headerKeywordName, headerKeywordValue, false);
    }
    public void setHeaderKeywordValue(String headerName, String headerKeywordName, Object headerKeywordValue) {
        setHeaderKeywordValue(headerName, headerKeywordName, headerKeywordValue, false);
    }

    public void setHeaderKeywordValue(String headerName, String headerKeywordName, Object headerKeywordValue, boolean sticky) {
        getHeaderKeywordMapForHeader(headerName).put(headerKeywordName, new HeaderKeywordValue(headerKeywordValue, sticky));
    }

    private void setHeaderKeywordValue(FitsHeaderKeywordData.HeaderKeywordValue value) {
        setHeaderKeywordValue(value.getHeaderName(), value.getHeaderKeywordName(), value.getHeaderKeywordValue(), value.isSticky());
    }

    /**
     * These methods to initialize the header keywords are to guarantee that
     * the provided header keywords are consistent with the provided geometry.
     *
     */
    private void fillHeaderKeywordMaps(Geometry geometry) {
        if (geometry instanceof Raft) {
            Raft raft = (Raft) geometry;
            for (Reb reb : raft.getRebs()) {
                fillHeaderKeywordMapsForReb(reb);
            }
        } else {
            throw new RuntimeException("This class is currently designed to support a single Raft.");
        }
    }
    private void fillHeaderKeywordMapsForReb(Reb reb) {
        for (CCD ccd : reb.getCCDs()) {
            if (!headerKeywordValuesMap.containsKey(ccd.getUniqueId())) {
                headerKeywordValuesMap.put(ccd.getUniqueId(), new HashMap<>());
            }
        }
    }

    private Map<String,HeaderKeywordValue> getHeaderKeywordMapForHeader(String headerName) {
        if ( !headerKeywordValuesMap.containsKey(headerName) ) {
            LOGGER.log(Level.WARNING, "The FitsService is not configured to store data for header {0}", headerName);
            headerKeywordValuesMap.put(headerName, new HashMap<>());
        }
        return headerKeywordValuesMap.get(headerName);
    }

    private MetaDataSet buildMetaDataSetForExtension(String extension) {
        Map<String,HeaderKeywordValue> headerKeywordMap = getHeaderKeywordMapForHeader(extension);
        Map<String, Object> valuesMap = new HashMap<>();
        for (String headerKeyword : headerKeywordMap.keySet()) {
            valuesMap.put(headerKeyword, headerKeywordMap.get(headerKeyword).getValue());
        }
        
        MetaDataSet result = new MetaDataSet();
        result.addMetaDataMap(extension, valuesMap);
        return result;
    }

    /**
     * Method to clear the non-sticky header keyword values.
     *
     */
    public void clearNonStickyHeaderKeywordValues() {
        for (Map<String, HeaderKeywordValue> map : headerKeywordValuesMap.values()) {
            clearNonStickyHeaderKeywordValuesFromMap(map);
        }
    }

    private void clearNonStickyHeaderKeywordValuesFromMap(Map<String, HeaderKeywordValue> map) {
        Iterator<Map.Entry<String, HeaderKeywordValue>> iter = map.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<String, HeaderKeywordValue> entry = iter.next();
            if (!entry.getValue().isSticky()) {
                iter.remove();
            }
        }
    }
    
    public FitsHeaderMetadataProvider getFitsHeaderMetadataProvider(String source) {
        return new FitsServiceFitsHeaderMetadataProvider(source);
    }

    private class FitsServiceFitsHeaderMetadataProvider implements FitsHeaderMetadataProvider {
    
        private final String ccdUniqueId;

        private MetaDataSet getMetaDataSet(String key) {
            if (key.equals("statusAggregator")) {
                return getStatusAggregatorMetaDataSet();
            } else if (key.equals("geometry")) {
                return geometryReplacementMetaDataSetMap.get(ccdUniqueId);
            } else {
                return buildMetaDataSetForExtension(key);
            }
        }

        public FitsServiceFitsHeaderMetadataProvider(String ccdUniqueId) {
            this.ccdUniqueId = ccdUniqueId;
        }
        
        @Override
        public MetaDataSet getAdditionalExtendedHeaderMetadata(String extendedKeyword) {
            MetaDataSet r = buildMetaDataSetForExtension(extendedKeyword);
            r.addMetaDataSet(getMetaDataSet("all"));
            r.addMetaDataSet(getMetaDataSet(ccdUniqueId));
            r.addMetaDataSet(getMetaDataSet("statusAggregator"));
            r.addMetaDataSet(getMetaDataSet("geometry"));
            return r;
        }

        @Override
        public MetaDataSet getDataExtendedHeaderMetadata(int extendedIndex) {
            MetaDataSet r = new MetaDataSet();
            r.addMetaDataSet(getMetaDataSet("all"));
            r.addMetaDataSet(getMetaDataSet(ccdUniqueId));
            r.addMetaDataSet(getMetaDataSet("statusAggregator"));
            r.addMetaDataSet(getMetaDataSet("geometry"));
            return r;
        }

        @Override
        public MetaDataSet getPrimaryHeaderMetadata() {
            MetaDataSet r = new MetaDataSet();
            r.addMetaDataSet(getMetaDataSet("primary"));
            r.addMetaDataSet(getMetaDataSet("all"));
            r.addMetaDataSet(getMetaDataSet(ccdUniqueId));
            r.addMetaDataSet(getMetaDataSet("statusAggregator"));
            r.addMetaDataSet(getMetaDataSet("geometry"));
            return r;
        }

    }

    private MetaDataSet getStatusAggregatorMetaDataSet() {
        MetaDataSet m = new MetaDataSet();
        Map<String, Object> aggrMap = aggregatorService.getAllLast();
        m.addMetaDataMap("StatusAggregator", aggrMap);
        return m;
    }

    //Private class used to set Header Keyword values.
    private class HeaderKeywordValue {

        private final Object value;
        private final boolean sticky;

        HeaderKeywordValue(Object value, boolean sticky) {
            this.value = value;
            this.sticky = sticky;
        }

        Object getValue() {
            return value;
        }

        boolean isSticky() {
            return sticky;
        }
    }

    public static class FitsServiceKeyReplacement {
        private final String match;
        private final String replace;

        public FitsServiceKeyReplacement(String configStr) {
            int index = configStr.indexOf(":");
            match = "(?i)" + Pattern.quote(configStr.substring(0, index));
            replace = configStr.substring(index + 1);
        }

        public String manipulate(String input) {
            return input.replaceAll(match, replace);
        }

        @Override
        public String toString() {
            return match+":"+replace;
        }

    }

    
}
