package org.lsst.ccs.daq.utilities;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.utilities.FitsService.FitsServiceKeyReplacement;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.Geometry;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.image.FitsHeaderMetadataProvider;
import org.lsst.ccs.utilities.image.FitsHeadersSpecificationsBuilder;
import org.lsst.ccs.utilities.image.HeaderSpecification;
import org.lsst.ccs.utilities.image.ImageSet;
import org.lsst.ccs.utilities.image.MetaDataSet;


/**
 * An <TT>AgentFitsHeaderService</TT> is responsible to load spec files for
 * writing out fits headers.
 *
 * This Service is activated when a list of header files is provided via the
 * "headerFilesList" configuration parameter.
 * 
 * This list contains a list of file names to be loaded. Each file name can be
 * specified as "fileName:fitsHeaderName". For example "ts8_primary:primary" to
 * assign the ts8_primary.spec file to the primary header.
 * 
 * If the name is already the name of the header, then the file name alone will 
 * suffice.
 *
 * 
 * @author The LSST CCS Team
 */
public class FitsService implements HasLifecycle, StatusMessageListener, FitsHeaderMetadataProvider {

    @ConfigurationParameter(isFinal = true)
    private List<String> headerFilesList = new ArrayList<>();

    @ConfigurationParameter(isFinal = true)
    private List<String> replacements = new CopyOnWriteArrayList<>();
    
    private final List<FitsServiceKeyReplacement> keyReplacements = new ArrayList<>();
    
    private Geometry geometry;
    private String uniqueId = "notSet";

    private static final Logger LOGGER = Logger.getLogger(FitsService.class.getName());

    private final FitsHeadersSpecificationsBuilder headerSpecsBuilder = new FitsHeadersSpecificationsBuilder();                

    //Header keyword values related objects
    private final Map<String, Map<String, HeaderKeywordValue>> headerKeywordValuesMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStatusAggregatorService aggregatorService;


    @ConfigurationParameterChanger(propertyName = "headerFilesList")
    public void setHeaderFilesList(List<String> list) {
        LOGGER.info("Configuring Fits Header Service with "+list);        
        headerFilesList.addAll(list);        
    }

    public void setGeometry(Geometry geometry) {
        LOGGER.info("Configuring Fits Header for geometry "+geometry.getUniqueId());        
        this.geometry = geometry;
        this.uniqueId = geometry.getUniqueId();
        fillHeaderKeywordMaps(geometry);
    }
        
    @Override
    public void postInit() {
        
        for ( String headerFile : headerFilesList ) {
            int index = headerFile.indexOf(":");
            String fileName = index >= 0 ? headerFile.substring(0,index) : headerFile;
            String headerName = index >=0 ? headerFile.substring(index+1) : null;
            if ( headerName == null ) {
                headerName = fileName.replace(".spec", "");
            }
            LOGGER.info("Loading spec file: "+fileName+" for extension "+headerName);
            headerSpecsBuilder.addSpecFile(fileName, headerName);
            if ( !headerKeywordValuesMap.containsKey(headerName) ) {
                headerKeywordValuesMap.put(headerName, new HashMap<>());
            }      
        }
        
        if (!replacements.isEmpty()) {
            LOGGER.info("FitsService is configured to perform the following replacements:");
            for (String r : replacements) {
                FitsServiceKeyReplacement obj = new FitsServiceKeyReplacement(r);
                keyReplacements.add(obj);
                LOGGER.info(obj.toString());
            }

        }
        
    }
    
    @Override
    public void start() {
        agent.getMessagingAccess().addStatusMessageListener(this,BusMessageFilterFactory.messageClass(StatusSubsystemData.class));
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {
        StatusSubsystemData d = (StatusSubsystemData) msg;
        if ( d.getDataKey().equals(FitsHeaderKeywordData.DATA_KEY) ) {
            FitsHeaderKeywordData fitsHeaderKeywordData = (FitsHeaderKeywordData) d.getObject().getValue();
            String headerKeywordDataId = fitsHeaderKeywordData.getDataId();
            if ( headerKeywordDataId != null && !headerKeywordDataId.isEmpty() && !headerKeywordDataId.equals(uniqueId)) {
                return;
            }
            for (FitsHeaderKeywordData.HeaderKeywordValue value : fitsHeaderKeywordData.getHeaderKeywordValues()) {
                setHeaderKeywordValue(value);
            }
            
        }
    }
    
    
    
    
    public Map<String,HeaderSpecification> getHeaderSpecificationMap() {
        return headerSpecsBuilder.getHeaderSpecifications();
    }
    
    protected void printHeaderSpecifications() {
        StringBuilder sb = new StringBuilder();
        Map<String, HeaderSpecification> config = headerSpecsBuilder.getHeaderSpecifications();
        for (String header : config.keySet()) {
            sb.append("***************************\n");
            sb.append("     Header: ").append(header).append("\n");
            sb.append("***************************\n");
            HeaderSpecification spec = config.get(header);
            for (HeaderSpecification.HeaderLine line : spec.getHeaders()) {
                sb.append("## ").append(line.getKeyword()).append(" ").append(line.getMetaName()).append(" ").append(line.getDataType()).append(" ").append(line.getComment()).append("\n");
            }
        }
        System.out.println("Header specifications\n"+sb.toString());
    }
    
    
    /**
     * Set a key for the Primary header of all CCDs.
     *
     * @param headerName The name of the Header to set
     * @param headerValue The corresponding value
     * @param sticky Boolean value to specify if the provided Header keyword
     * value should be used across exposures. The default value is true. If
     * false is provided, the provided Header keyword value will be reset after
     * each exposure.
     */    
    public void setHeaderKeywordValue(String headerKeywordName, Object headerKeywordValue) {
        setHeaderKeywordValue("primary",headerKeywordName, headerKeywordValue, false);
    }
    public void setHeaderKeywordValue(String headerKeywordName, Object headerKeywordValue, boolean sticky) {
        setHeaderKeywordValue("primary",headerKeywordName, headerKeywordValue, sticky);
    }
    public void setHeaderKeywordValue(String headerName, String headerKeywordName, Object headerKeywordValue) {
        setHeaderKeywordValue(headerName, headerKeywordName, headerKeywordValue, false);
    }
    public void setHeaderKeywordValue(String headerName, String headerKeywordName, Object headerKeywordValue, boolean sticky) {
        LOGGER.info("Setting for header: "+headerName+" ("+headerKeywordName+","+headerKeywordValue+") "+sticky);        
        getHeaderKeywordMapForHeader(headerName).put(headerKeywordName, new HeaderKeywordValue(headerKeywordValue, sticky));
    }
    
    private void setHeaderKeywordValue(FitsHeaderKeywordData.HeaderKeywordValue value) {
        setHeaderKeywordValue(value.getHeaderName(), value.getHeaderKeywordName(), value.getHeaderKeywordValue(), value.isSticky());
    }
    
    /**
     * These methods to initialize the header keywords are to guarantee that
     * the provided header keywords are consistent with the provided geometry.
     * 
     */
    private void fillHeaderKeywordMaps(Geometry geometry) {
        if (geometry instanceof Raft) {
            Raft raft = (Raft) geometry;
            for (Reb reb : raft.getChildrenList()) {
                fillHeaderKeywordMapsForReb(reb);
            }
        } else if (geometry instanceof Reb) {
            fillHeaderKeywordMapsForReb((Reb) geometry);
        } else {
            throw new RuntimeException("This class is currently designed to support a single Raft.");
        }
    }    
    private void fillHeaderKeywordMapsForReb(Reb reb) {
        for (CCD ccd : reb.getChildrenList()) {
            if ( ! headerKeywordValuesMap.containsKey(ccd.getUniqueId()) ) {
                headerKeywordValuesMap.put(ccd.getUniqueId(), new HashMap<>());
            }
        }        
    }
    
    private Map<String,HeaderKeywordValue> getHeaderKeywordMapForHeader(String headerName) {
        if ( !headerKeywordValuesMap.containsKey(headerName) ) {
            LOGGER.warning("The FitsService is not configured to store data for header "+headerName);
            headerKeywordValuesMap.put(headerName, new HashMap<>());
        }
        return headerKeywordValuesMap.get(headerName);
    }

    private Map<String, Object> buildMapOfHeaderKeywordValues(Map<String, HeaderKeywordValue> map) {
        Map<String, Object> result = new HashMap<>();
        for (String headerKeyword : map.keySet()) {
            result.put(headerKeyword, map.get(headerKeyword).getValue());
        }
        return result;
    }
    
    /**
     * Method to clear the non-sticky header keyword values.
     * 
     */
    public void clearNonStickyHeaderKeywordValues() {
        for (Map<String, HeaderKeywordValue> map : headerKeywordValuesMap.values()) {
            clearNonStickyHeaderKeywordValuesFromMap(map);
        }
    }

    private void clearNonStickyHeaderKeywordValuesFromMap(Map<String, HeaderKeywordValue> map) {
        Iterator<Map.Entry<String, HeaderKeywordValue>> iter = map.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<String, HeaderKeywordValue> entry = iter.next();
            if (!entry.getValue().isSticky()) {
                iter.remove();
            }
        }
    }
    
    
    @Override
    public MetaDataSet getAdditionalExtendedHeaderMetadata(ImageSet imageSet, String extendedKeyword) {
        return buildMetaDataSetForExtension(extendedKeyword, imageSet);
    }

    @Override
    public MetaDataSet getDataExtendedHeaderMetadata(ImageSet imageSet, int extendedIndex) {
        return buildMetaDataSetForExtension(imageSet.getCCD().getUniqueId(), imageSet);
    }

    @Override
    public MetaDataSet getPrimaryHeaderMetadata(ImageSet imageSet) {
        return buildMetaDataSetForExtension("primary", imageSet);
    }

    @Override
    public void completedHeaderMetadata(ImageSet imageSet) {
        statusAggregatorDataMap.remove(imageSet);
    }
    
    private final Map<ImageSet, Map<String,Object>> statusAggregatorDataMap = new ConcurrentHashMap<>();
    
    private MetaDataSet buildMetaDataSetForExtension(String extension, ImageSet imageSet) {
        MetaDataSet m = new MetaDataSet();
        synchronized(statusAggregatorDataMap) {
            if ( !statusAggregatorDataMap.containsKey(imageSet) ) {                
                Map<String,Object> aggrMap = aggregatorService.getAllLast();
                Map<String,Object> toAdd = new HashMap<>();
        
                for(Entry<String,Object> e : aggrMap.entrySet()) {                    
                    String key = e.getKey();
                    for(FitsServiceKeyReplacement r : keyReplacements) {
                        key = r.manipulate(key);
                    }
                    if (!key.equals(e.getKey())) {
                        if (toAdd.containsKey(key) || aggrMap.containsKey(key)) {
                            LOGGER.log(Level.WARNING, "Key {0} already exists in metadata set from Status Aggregator. This might lead to problems.", key);
                        }
                        toAdd.put(key, e.getValue());
                    }
                }
                aggrMap.putAll(toAdd);
                statusAggregatorDataMap.put(imageSet, aggrMap);
            }
            m.addMetaDataMap("StatusAggregator", statusAggregatorDataMap.get(imageSet));
        }
        
        m.addMetaDataMap(extension, buildMapOfHeaderKeywordValues(getHeaderKeywordMapForHeader(extension)));        
        return m;
    }
    
    //Private class used to set Header Keyword values.
    private class HeaderKeywordValue {

        private final Object value;
        private final boolean sticky;

        HeaderKeywordValue(Object value, boolean sticky) {
            this.value = value;
            this.sticky = sticky;
        }

        Object getValue() {
            return value;
        }

        boolean isSticky() {
            return sticky;
        }
    }

    public static class FitsServiceKeyReplacement {
        private final String match;
        private final String replace;

        public FitsServiceKeyReplacement(String configStr) {
            int index = configStr.indexOf(":");
            match = configStr.substring(0,index);
            replace = configStr.substring(index+1);
        }
        
        public String manipulate(String input) {
            return input.replaceAll(match, replace);
        }
        
        @Override
        public String toString() {
            return match+":"+replace;
        }
        
    }
    
    
}
