package org.lsst.ccs.config.utilities;

import groovy.util.Eval;
import org.lsst.ccs.Subsystem;
import org.lsst.gruth.jutils.HollowParm;
import org.lsst.gruth.types.GArray;
import org.lsst.gruth.types.GList;
import org.lsst.gruth.types.GMap;
import org.lsst.gruth.types.GStruct;

import java.io.*;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * Stores as String various types in a property store.
 * <BR/>
 * The way these properties are stored is abstract: though the usual (and default) way is to use
 * a file, it can be modified to use something else (such as a database).
 * <P>
 * The type of the stored data is not obvious and should be known the the caller code.
 * That's why there are different method to store/retrieve data (each method for each type).
 *
 * <P/>
 * <B>important note</B> : once you have stored some data with a key you cannot re-use this key for another type.
 * Or if you want to do that then put the property to <TT>null</TT> before.
 *
 * <P/>
 * <B>Important note</B> : for the time being (this may change later) instances of this class are not garbage collected
 *
 * @author bamade
 */
// Date: 08/11/12

public class PersistentProperties implements Serializable, Cloneable {
    /**
     * this class defines the way the properties are made persistent and read.
     * A <TT>Store</TT> could be a properties file or a database for example.
     */
    public static interface Store {
        /**
         * should be normally part of the constructor: call this method only once!
         * @param props
         */
        void storeIsFor(PersistentProperties props) ;

        /**
         * will save the properties registered for this store
         * @throws IOException
         */
        void commit() throws IOException;

        /**
         * will read the Properties registered by this store
         * @throws IOException
         */
        void update() throws IOException;
    }

    public static class FileStore implements Store {
        String storeName ;
        PersistentProperties persistentProperties ;
        File currentFile ;

        public void storeIsFor(PersistentProperties props ) {
            if(storeName != null) {
                throw new IllegalStateException("name already set for Store") ;
            }
            persistentProperties = props ;
            this.storeName = props.name;
        }

        @Override
        public void commit() throws IOException {
            //System.out.println(" storing " + persistentProperties.properties);
            checkForFile();
            FileOutputStream fos = new FileOutputStream(currentFile) ;
            persistentProperties.properties.store(fos, "persistent properties store for " +
                    persistentProperties.name);
            try {
                fos.close();
            } catch (IOException exc) {/*IGNORE*/ }
            persistentProperties.modified = false ;
        }

        private void checkForFile() throws IOException {
            if(currentFile == null) {
                URL url = Subsystem.class.getResource(storeName + ".properties") ;
                if(url != null) {
                    try {
                        currentFile = new File(url.toURI());
                    } catch (URISyntaxException e) {
                        throw new IOException(e) ;
                    }
                } else {
                    currentFile = new File(storeName + ".properties") ;
                }
            }

        }

            @Override
            public void update() throws IOException {
                checkForFile();
                FileInputStream fis = new FileInputStream(currentFile) ;
                persistentProperties.properties.load(fis);
                try {
                    fis.close() ;
                } catch (IOException exc) {
                    /*IGNORE*/
                }
                //is it relevant?
                persistentProperties.modified = false ;
        }
    }

    ///TODO use SoftReferences and a queue
    private static ArrayList<PersistentProperties> activeProperties =
            new ArrayList<PersistentProperties>();

    static {
        // start a Runtime shutdown hook
        Runtime.getRuntime().addShutdownHook(new Thread() {
            public void run() {
                //read the activeProperties
                for (PersistentProperties props : activeProperties) {
                    Store store = props.store;
                    if (store != null) {
                        if (props.modified) {
                            try {
                                store.commit();
                            } catch (IOException exc) {
                                //TODO log
                            }
                        }
                    }
                }
            }
        });
    }
    private Store store  ;
    private String name ;
    private Properties properties = new Properties();
    private volatile boolean modified;

    /**
     * creates a <TT>PersistenProperties</TT> with a specific store.
     * @param name (beware should be consistent with that are legal for your store!)
     * @param store
     */
    public PersistentProperties(String name, Store store) {
        this.store = store;
        this.name = name;
        store.storeIsFor(this);
        activeProperties.add(this) ;
    }

    /**
     * uses a fileStore.
     * @param name (beware should be consistent with a fileName)
     */
    public PersistentProperties(String name) {
        this(name, new FileStore()) ;
    }


    // private utilities methods


    private Integer toInteger(String intString) {
        if (intString == null) return null;
        return new Integer(intString);
    }

    private Double toDouble(String doubleString) {
        if (doubleString == null) return null;
        return new Double(doubleString);
    }

    private Float toFloat(String floatString) {
        if (floatString == null) return null;
        return new Float(floatString);
    }

    private Boolean toBoolean(String booleanString) {
        if (booleanString == null) return null;
        return new Boolean(booleanString);
    }

    private Map toMap(String mapString) {
        if (mapString == null) return null;
        return GMap.valueOf(mapString);
    }

    private List toList(String listString) {
        if (listString == null) return null;
        return GList.valueOf(listString);
    }

    private <T> T toObject(Class<T> clazz, String objectString) {
        if (objectString == null) return null;
        if (clazz.isArray()) {
            return (T) GArray.valueOf(objectString, clazz.getName());
        } else {
            Object obj = Eval.me(objectString);
            if (obj instanceof List) {
                return (T) GStruct.valueOf(clazz, (List) obj);
            }
            if (obj instanceof Map) {
                return (T) GStruct.valueOf(clazz, (Map) obj);
            }
        }
        throw new IllegalArgumentException(" format not available : " + objectString + "for " + clazz);
    }

    ///////////// methods for simple

    /**
     * put an Integer value in the Properties
     * @param key
     * @param value
     * @return the previous Integer value or null
     */
    public Integer putInteger(String key, int value) {
        Object res = properties.setProperty(key, String.valueOf(value));
        Integer resInt = toInteger((String) res);
        modified = true;
        return resInt;
    }


    /**
     * get a Integer value from the Properties
     * @param key
     * @return the value or null if value is not set (or set to null)
     */
    public Integer getInteger(String key) {
        return toInteger(properties.getProperty(key));
    }

    /**
     * put a Boolean value in the properties
     * @param key
     * @param bool
     * @return the previous Boolean value or null
     */
    public Boolean putBoolean(String key, boolean bool) {
        Object res = properties.setProperty(key, String.valueOf(bool));
        Boolean resBool = toBoolean((String) res);
        modified = true;
        return resBool;
    }

    /**
     * gets a Boolean value stored with this key
     * @param key
     * @return
     */
    public Boolean getBoolean(String key) {
        return toBoolean(properties.getProperty(key));
    }

    /**
     * puts a Double value in the properties
     * @param key
     * @param val
     * @return previous double value or null
     */
    public Double putDouble(String key, double val) {
        Object res = properties.setProperty(key, String.valueOf(val));
        Double resDbl = toDouble((String) res);
        modified = true;
        return resDbl;
    }

    /**
     * get a double value associated with the key
     * @param key
     * @return value or null if no such property
     */
    public Double getDouble(String key) {
        return toDouble(properties.getProperty(key));
    }

    /**
     * registers a Float as a Property
     * @param key
     * @param val
     * @return previous value or null
     */
    public Float putFloat(String key, float val) {
        Object res = properties.setProperty(key, String.valueOf(val));
        Float resDbl = toFloat((String) res);
        modified = true;
        return resDbl;
    }

    /**
     * gets a Float from a value in the properties
     * @param key
     * @return
     */
    public Float getFloat(String key) {
        return toFloat(properties.getProperty(key));
    }

    ///////////////// OBJECTS

    private Object rawPutObject(String key, Object obj) {
        if (obj == null) {
            return properties.setProperty(key, null);
        }
        Object res = properties.setProperty(key, HollowParm.stringify(obj));
        return res;
    }

    // define String, Array, list, map, struct

    /**
     * registers a String in the properties
     * @param key
     * @param value
     * @return previous value or null if there was none
     */
    public String putString(String key, String value) {
        String res = (String) properties.setProperty(key, value);
        modified = true;
        return res;
    }

    /**
     * gets a String associated with a key
     * @param key
     * @return value or null if none was found
     */
    public String getString(String key) {
        return properties.getProperty(key);
    }

    /**
     *  registers an array in the properties.
     *  You'll have to declare an array class which is compatible with the <TT>value</TT>
     *  argument : if the result is not null the returned Object will be of this type.
     *  <P/>
     *  So for example :
     *  <PRE>
     *      Double[] arrDbl = {2.44, 6.66} ;
     *      Number[] arrNumber = myProperties.putArray("valueArray", Number[].class, arrDbl) ;
     *  </PRE>
     *  Doing this:
     *  <PRE>
     *      Number[] arrNumber = myProperties.putArray("valueArray", Double[].class, arrDbl) ;
     *  </PRE>
     *  will work but you may end up with an <TT>ArrayStoreException</TT>
     *  when dealing later with <TT>arrNumber</TT>
     * @param key
     * @param arrayClass such as String[].class
     * @param value could be an array which is "assignable to"
     *              this array elements should be of a primitive type, wrapper types, String, BigDecimal or Arrays of arrays of these types.
     *              Arrays of other types of Objects are not supported (some "structs" -see below-would work)
     * @param <T> type of Array with elements of type X
     * @param <V> array of any type Y that is assignment compatible to X
     * @return
     */
    public <T, V extends T> T putArray(String key, Class<T> arrayClass, V value) {
        Object rawres = rawPutObject(key, value);
        T res = toObject(arrayClass, (String) rawres);
        modified = true;
        return res;
    }

    /**
     * returns a value stored in the properties as an array.
     * @param key
     * @param arrayClass
     * @param <T>
     * @return the array which is represented by the String stored or null if there is none
     */
    public <T> T getArray(String key, Class<T> arrayClass) {
        return toObject(arrayClass , properties.getProperty(key));
    }

    /**
     * This will store objects that stick to a convention (and known as <TT>struct</TT>.
     * The values are stored as a String version of a <TT>List</TT> or of a <TT>Map</TT>
     * depending on the definition of the Object:
     * <UL>
     *     <LI/> if the object has a full fledged constructor, then a List will be used
     *     <LI/> if the object is a <TT>bean</TT> with only <I>getters/setters</I>
     *     then a Map will be used (each pair will be : Name, value)
     * </UL>
     * In both cases the <TT>toString</TT> method of these object should exactly produce
     * a List or Map litteral (with the conventions <TT>[1,'world',2.8]</TT> for lists
     * and <TT>[number:1, name:'world', max:2.8]</TT> for Maps.
     * @param key
     * @param structClass the name of the class that will be used to analyse previous value
     * @param value any object that sticks to this <TT>struct</TT> convention
     * @param <T>
     * @return
     */
    public <T> T putStruct(String key, Class<T> structClass, T value) {
        Object rawres = rawPutObject(key, value);
        T res = toObject(structClass, (String) rawres);
        modified = true;
        return res;
    }

    /**
     * returns an object stored with the <TT>struct</TT> convention from the properties.
     * @param key
     * @param structClass
     * @param <T>
     * @return
     */
    public <T> T getStruct(String key, Class<T> structClass) {
        return toObject(structClass , properties.getProperty(key));
    }

    /**
     * registers a List to the properties.
     * Note that the elements of the list can be only wrappers (Double, Integer,...), and
     * String objects or List of these lists (or Maps of this simple data).
     * <P/>
     * <B>Warning</B> : a floating point value stored will be returned as a <TT>BigDecimal</TT>
     * this may change in future releases (known feature)
     * @param key
     * @param list
     * @return
     */
    public List putList(String key, List list) {
        Object rawres = rawPutObject(key, list);
        List res = toList((String)rawres) ;
        modified = true ;
        return res;
    }

    /**
     * returns a list stored in the properties.
     * <P/>
     * <B>Warning</B> : a floating point value stored will be returned as a <TT>BigDecimal</TT>
     * this may change in future releases (known feature)
     *
     * @param key
     * @return
     */
    public List getList(String key) {
        return toList( properties.getProperty(key));
    }

    /**
     * stores a <TT>Map</TT> as a String in the properties.
     * Note that the values of the map can be only wrappers (Double, Integer,...), and
     * String objects  (or lists or Maps of such objects).
     *
     * <B>Warning</B> : a floating point value stored will be returned as a <TT>BigDecimal</TT>
     * this may change in future releases (known feature)
     *
     * @param key
     * @param map
     * @return
     */
    public Map putMap(String key, Map map) {
        Object rawres = rawPutObject(key, map);
        Map res = toMap((String)rawres) ;
        modified = true ;
        return res;
    }

    /**
     * returns a <TT>Map</TT> stored in the properties.
     * <B>Warning</B> : a floating point value stored will be returned as a <TT>BigDecimal</TT>
     * this may change in future releases (known feature)
     *
     * @param key
     * @return
     */
    public Map getMap(String key) {
        return toMap( properties.getProperty(key));
    }

    private Object evalIt(String str){
        String trimStr = str.trim() ;
       // if first character is not [ or number then returns the arg
        char first = trimStr.charAt(0) ;
        Object res = str ;
        if(Character.isDigit(first) || first == '[') {
            res = Eval.me(trimStr) ;
        } else if ("true".equalsIgnoreCase(trimStr)){
            return new Boolean(true) ;
        } else if ("false".equalsIgnoreCase(trimStr)) {
            return new Boolean(false) ;
            // numeric that start with a dot
            // see if groovy has such a test ....
        }
        return res ;
    }

    /**
     * Puts an object with no type specification in the  properties.
     * Without type specification some unwanted effect might happen when reading
     * the previous value:
     * <UL>
     *     <LI/> floating point values will be returned as BigDecimal
     *     <LI/> Strings that start with a numerical character will fire an attempt to
     *     create a number (which may fail if you have a String such as "200x300")
     *     <LI/> String that start with a "[" character will fire an attempt to read a List or
     *     Map (which may fail!)
     *     <LI/> String "true" or "false" will return a Boolean!
     *     <LI/> <TT>struct</TT> objects won't be analysed
     *     <LI/> arrays will be returned as Lists!
     * </UL>
     * @param key
     * @param obj
     * @return
     */
    public Object putObject(String key, Object obj){
        Object rawres = rawPutObject(key, obj);
        if(rawres != null) {
            rawres = evalIt((String)rawres) ;
        }
        modified = true ;
        return rawres;
    }

    /**
     * tries to return an object out of a String.
     * See remarks with method <TT>putObject</TT>
     * @param key
     * @return
     */
    public Object getObject(String key) {
        String rawres = properties.getProperty(key) ;
        Object res = null ;
        if(rawres != null) {
            res = evalIt(rawres) ;
        }
        return res ;
    }
}
