package org.lsst.ccs.config.remote;

import org.lsst.ccs.bus.BusMessage;
import org.lsst.ccs.bus.StatusForEnd;
import org.lsst.ccs.bus.StatusForStart;
import org.lsst.ccs.config.*;
import org.lsst.ccs.framework.BasicModularSubSystem;

import java.util.logging.Level;

/**
 * A subsystem that listens on the buses for request for boot data, or for other subsystems
 * status notifications fired when they start with a configuration (or give up);
 * it is also a full-fledged Configuration service that can receive and execute commands.
 * <p/>
 * The system property "org.lsst.ccs.testcontext" tells the server if it is only in test context.
 * In that case it will handle only boot requests that are in test context (property <TT>inTestContext</TT>:
 * this can be used to create a "dummy" file based server that handles tests.
 *
 * @author bamade
 */
// Date: 26/06/12

public class ConfigurationServerSystem extends BasicModularSubSystem {
    ConfigurationServiceImpl service;
    ConfigurationFacade facade;
    boolean forTestsOnly;

    public ConfigurationServerSystem(ConfigurationFacade facade) {
        super("configServer");
        this.facade = facade;
        forTestsOnly = Boolean.getBoolean("org.lsst.ccs.testcontext");
        this.service = new ConfigurationServiceImpl(facade);
        // add module main as ConfigurationserverModule
        this.addModule(new ConfigurationServerModule(service));
        this.setStatusBroadcastPeriod(24 * 60 * 60 * 1000);
        this.setListenToStatus(true);
    }

    @Override
    public void onStatus(BusMessage message) {
        String initiatorName = message.getOrigin();
        //if bootStatus
        try {
            if (message instanceof BootStatus) {
                // beware: initiator is not a subsystem
                BootStatus bootStatus = (BootStatus) message;
                // call appropriate method
                String requestedSubsystem = bootStatus.getRequestedSubsystem();
                String requestedConfiguration = bootStatus.getRequestedConfiguration();
                String requestedTag = bootStatus.getRequestedTag();
                String origin = bootStatus.getOrigin();
                boolean forTestContext = bootStatus.isInTestContext();
                // the server is not for test AND the message is not for Test
                // OR the server is for test AND the message is for Test
                if (!(forTestsOnly ^ forTestContext)) {
                    Object startupObject = null;
                    BootCommand cmd = null;
                    if (requestedSubsystem != null && !"".equals(requestedSubsystem)) {

                    } else {
                        // based on machineConfiguration
                        MachineConfiguration machineConfiguration = facade.getMachineConfiguration(origin);
                        requestedSubsystem = machineConfiguration.getSubsystemName() ;
                        requestedConfiguration = machineConfiguration.getConfigName() ;
                        requestedTag = machineConfiguration.getTag() ;
                    }
                    // facade request for active combination
                    //if configuration send configuration
                    if(requestedConfiguration!= null && ! "".equals(requestedConfiguration)) {
                        startupObject = facade.getActiveConfigProfile(requestedConfiguration, requestedTag) ;
                    } else {
                        // else if description send description only
                        startupObject = facade.getActiveSubsystemDescription(requestedSubsystem, requestedTag);
                    }
                    cmd = new BootCommand(origin, requestedSubsystem, requestedConfiguration, requestedTag, startupObject);
                    // and send Command
                    if (cmd != null) {
                        fac.sendCommand(cmd);
                    } else {
                        PackCst.CURLOG.log(Level.WARNING, "no descriptions for ", message);
                        //TODO :tell subsystem startup we are in error!
                    }
                }

                //
            } else if (message instanceof StatusForStart) {
                StatusForStart startStatus = (StatusForStart) message;
                //TODO: this is a patch due to the fact this is a subsystem: modify!
                if(startStatus.getOrigin().equals(getName())) {
                    return ;
                }
                boolean forTestContext = startStatus.isInTestContext();
                if (!(forTestsOnly ^ forTestContext)) {
                    facade.registerRun(initiatorName, startStatus.getConfigurationName(),
                            startStatus.getTag(), message.getTimeStamp());
                    //register RUN
                }
            } else if (message instanceof StatusForEnd) {
                //register end run
                StatusForEnd endStatus = (StatusForEnd) message;
                //TODO: this is a patch due to the fact this is a subsystem: modify!
                if(endStatus.getOrigin().equals(getName())) {
                    return ;
                }
                boolean forTestContext = endStatus.isInTestContext();
                if (!(forTestsOnly ^ forTestContext)) {
                    facade.endRun(initiatorName, endStatus.getConfigurationName(),
                            endStatus.getTag(), endStatus.getTimeStamp());
                }
            }
        } catch (Exception exc) {
            PackCst.CURLOG.log(Level.SEVERE, "on status message " + message, exc);
            System.err.println("ERROR :" + exc);
        }
    }

    public static void main(String[] args) {
        System.setProperty("lsst.messaging.factory", "org.lsst.ccs.bus.BusMessagingFactory");
        ConfigurationFacade facade;
        if (Boolean.getBoolean("org.lsst.ccs.testcontext")) {
            facade = new ConfigurationFacade(new FIleBasedDAO());
        } else {
            facade = new ConfigurationFacade(new HibernateDAO());
        }
        ConfigurationServerSystem system = new ConfigurationServerSystem(facade);
        system.start();
    }
}
