package org.lsst.ccs.config;

import org.hibernate.annotations.Immutable;
import org.lsst.gruth.jutils.ComponentNode;
import org.lsst.gruth.jutils.HollowParm;

import javax.persistence.*;
import java.util.*;

/**
 * An actual configuration profile.
 * Once registered instances of this class are immutable except for  comments
 * and registration of method calls (to be specified).
 *
 * @author bamade
 */
// Date: 12/04/12

@Table(name = "AConfigProfile",
        uniqueConstraints = {@UniqueConstraint(columnNames = {"name", "tag"})}
)
@Entity
@Immutable
class AConfigProfile extends ConfigProfile implements Cloneable {
    @Id
    @GeneratedValue
    private long id; //generated
    @ManyToOne (fetch=FetchType.EAGER)
    private /*@NotNull*/ ASubsystemDescription subsystemDesc;

    /**
     * duplicate information : helps when in an intermediate state where
     * the subsystemDescription is being deprecated ;
     */
    private long subsystemId;


    //TODO: A map<Path,XXX> instead
    @OneToMany(cascade = CascadeType.ALL, fetch=FetchType.EAGER)
    private /*@NotNull*/ Set<AParameterConfiguration> parameterConfigurations =
            // just in case! not ideal when handled as a bean
            new HashSet<AParameterConfiguration>();

    @Transient
    private /*@NotNull*/ Set<AParameterConfiguration> unModifiableConfigurations =
            Collections.unmodifiableSet(parameterConfigurations);

    ////////////////////////////////// CONSTRUCTORS

    AConfigProfile() {
    }

    /**
     * creates a config profile pointing to a Subsystem Description that must be registered in the database.
     *
     * @param subsystemDesc
     * @param name
     * @param tag
     * @param userName
     * @param level
     * @throws IllegalArgumentException if the subsystem description is not registered or is deprecated
     */
    public AConfigProfile(ASubsystemDescription subsystemDesc, String name, String tag, String userName, int level) {
        super(name, tag, userName, level);
        this.subsystemDesc = subsystemDesc;
        this.subsystemId = subsystemDesc.getId();
        if (subsystemId == 0) {
            throw new IllegalArgumentException("Subsystem description should have been registered in database");
        }
        //TODO: tautological all instances of ASubsystemDescription have a STILL_VALID date!
        long endDate = subsystemDesc.getEndTimestamp();
        // when in remote operation this tet may not suffice
        if (!(endDate == PackCst.STILL_VALID)) {
            throw new IllegalArgumentException("using deprecated subsystem");
        }
    }


    /**
     * creates a copy of an active ConfigProfile (for instance to start an engineering mode or to validate
     * a new tagged configuration from an engineering mode configuration).
     *
     * @param anotherConfig
     * @param name
     * @param tag
     * @param userName
     * @param level
     * @param toEngineering
     */
    public AConfigProfile(AConfigProfile anotherConfig, String name, String tag,
                          String userName, int level, boolean toEngineering) {
        this(anotherConfig.getSubsystemDesc(), name, tag, userName, level);
        setBeenInEngineeringMode(toEngineering);
        for (AParameterConfiguration parmConfig : anotherConfig.getParameterConfigurations()) {
            AParameterConfiguration copyParm = AParameterConfiguration.copyWithoutEvents(parmConfig);
            if (anotherConfig.isBeenInEngineeringMode()) {
                List<AValueEvent> eventList = parmConfig.valueEvents;
                if (toEngineering) {
                    //copy the history (use case?)
                    if (eventList != null) {
                        for (AValueEvent event : eventList) {
                            copyParm.addValueEvent(event);
                        }
                    }
                    copyParm.setCopy(true);

                } else {
                    //move last in history to value
                    if (eventList != null) {
                        int size = eventList.size();
                        // "belt and suspenders" strategy
                        if (size > 0) {
                            AValueEvent event = eventList.get(size - 1);
                            copyParm.setValue(event.getValue());
                        }
                    }
                }
            }
            this.addParameterConfigurations(copyParm);
        }
    }

    /////////////////////////////////// ACCESSORS/MUTATORS
    @Override
    public long getId() {
        return id;
    }

    @Override
    protected void setId(long id) {
        this.id = id;
    }


    @Override
    public SubsystemDescription getSubsystemDescription() {
        return subsystemDesc;
    }


    /**
     * return <TT>getParameterConfigurations</TT>
     *
     * @return
     */
    @Override
    public Set<? extends ParameterConfiguration> getModifiedParameters() {
        return getParameterConfigurations();
    }

    /**
     * changes a parameter in Engineering mode.
     * @param parameter
     * @param time
     * @param value
     */
    @Override
    public void temporaryChangeConfigurationValue(ParameterConfiguration parameter, long time, String value) {
        // if not in engineering mode stops
        if (!isBeenInEngineeringMode()) {
            throw new IllegalStateException("ConfigProfile not in Engineering mode");
        }
        // if parameter is null ->  exception
        if (null == parameter) {
            throw new IllegalArgumentException("no such ParameterConfiguration");
        }
        //  cast to AParemeterConfiguration
        // wrong AParameterConfiguration aParameterConfiguration = (AParameterConfiguration) parameter;
        AParameterConfiguration aParameterConfiguration = (AParameterConfiguration) this.fetch(parameter);
        // TODO if(aParameterConfiguration == null) {
        aParameterConfiguration.addValueEvent(new AValueEvent(time, value));
    }

    public   ParameterConfiguration temporaryChangeConfigurationValue(String  parameterPath, long time, String value) {

        ParameterPath path = ParameterPath.valueOf(parameterPath) ;
        // fetch the ParameterConfiguration
        ParameterConfiguration parameter = this.fetch(path) ;
        if(parameter == null) {
          //creates a new Parameter
            AParameterDescription description = (AParameterDescription)subsystemDesc.fetch(path) ;
          // register it to the  configProfile
            parameter = new AParameterConfiguration(description) ;
            this.addParameterConfigurations(parameter);
        }
        //calls the other method
        temporaryChangeConfigurationValue(parameter, time, value);
        return parameter ;
    }

    ASubsystemDescription getSubsystemDesc() {
        return subsystemDesc;
    }

    void setSubsystemDesc(ASubsystemDescription subsystemDesc) {
        this.subsystemDesc = subsystemDesc;
    }

    /**
     * @return an unmodifiable set
     */
    public Set<AParameterConfiguration> getParameterConfigurations() {
        return unModifiableConfigurations;
    }


    void setParameterConfigurations(Set<AParameterConfiguration> parameterConfigurations) {
        this.parameterConfigurations = parameterConfigurations;
        this.unModifiableConfigurations = Collections.unmodifiableSet(parameterConfigurations);
    }

    public AConfigProfile clone() {
        AConfigProfile res = null;
        try {
            res = (AConfigProfile) super.clone();
            HashSet<AParameterConfiguration> newSet = new HashSet<AParameterConfiguration>();
            for (AParameterConfiguration parmConfig : parameterConfigurations) {
                newSet.add(parmConfig.clone());
            }
            res.setParameterConfigurations(newSet);
        } catch (CloneNotSupportedException e) { /*IGNORE*/}
        return res;

    }

    public long getSubsystemId() {
        return subsystemId;
    }

    void setSubsystemId(long subsystemId) {
        this.subsystemId = subsystemId;
    }

    public void setBeenInEngineeringMode(boolean beenInEngineeringMode) {
        super.setBeenInEngineeringMode(beenInEngineeringMode);
    }

    /**
     * the only public mutator
     * <p/>
     * TODO: no persistence of this for the moment
     *
     * @param remarks
     */
    public void setRemarks(String remarks) {
        //TODO add a transient modification boolean checked by the persistence
        super.setRemarks(remarks);
    }

    /**
     * Modifies the current object by using a properties object (built from a .properties file)
     * @param props
     */
    //TODO: what happens if a value is illegal right in the middle of a modification?
    @Override
    public void mergeProperties(Properties props) {
        if (getId() != 0L) {
            throw new ImmutableStateException("no modification of registered data");
        }
        // creates a Map of all ParameterDescription using name and simple names
        Map<String, ParameterDescription> map = subsystemDesc.generateDescriptionMap();
        for (String name : props.stringPropertyNames()) {
            ParameterDescription description = map.get(name);
            if (description != null) {
                // is there a modifiable configuration or not?
                ParameterConfiguration configuration = this.fetch(description);
                String value = props.getProperty(name);
                if (configuration != null) {
                    configuration.modifyValue(value);
                } else {
                    AParameterDescription realDescription = (AParameterDescription) description;
                    configuration = new AParameterConfiguration(realDescription, value);
                    this.addParameterConfigurations(configuration);
                }

            }
        }
    }
    //////////////////////////// IDENT METHODS
///////////////////////////////// UTILITIES METHODS

    /**
     * to build a list of Parameter configurations one needs a list of possible parameter descriptions
     * that can be used.
     *
     * @return
     */
    public Set<AParameterDescription> getPossibleParameters() {
        Set<AParameterDescription> res = new HashSet<AParameterDescription>();
        for (ParameterDescription parmDesc : this.getSubsystemDescription().getParamDescriptionSet()) {
            if (parmDesc.getLevel() <= this.getLevel()) {
                res.add(new AParameterDescription(parmDesc));
            }
        }
        return res;
    }

    /**
     * registers a list of parameter configurations
     *
     * @param parameterConfigurations
     * @throws ImmutableStateException  if the object is already registered in database
     * @throws IllegalArgumentException if levels are incompatibles
     * @throws IllegalArgumentException if the corresponding parameter descriptions are not alive in the current
     *                                  subsystem description
     */
    // TODO: check if Description not deprecated!
    public void addParameterConfigurations(AParameterConfiguration... parameterConfigurations) {
        if (isReadOnly()) {
            throw new ImmutableStateException("Parameter Configuration list");
        }
        for (AParameterConfiguration parameterConfiguration : parameterConfigurations) {
            AParameterDescription description = parameterConfiguration.getParameterDescription();
            // check for level
            if (description.getLevel() > this.getLevel()) {
                throw new IllegalArgumentException("incompatible levels. profile is : " + getLevel()
                        + " and description is :" + description.getLevel());
            }
            // check for deprecation
            this.parameterConfigurations.add(parameterConfiguration);
        }
    }

    /**
     * adds parameter configuration objects to this profile
     * @param parameterConfigurations
     */
    public void addParameterConfigurations(ParameterConfiguration... parameterConfigurations) {
        addParameterConfigurations(buildSafeArray(parameterConfigurations));
    }

    private AParameterConfiguration[] buildSafeArray(ParameterConfiguration... parameterConfigurations) {
        AParameterConfiguration[] parms = new AParameterConfiguration[parameterConfigurations.length];
        for (int ix = 0; ix < parameterConfigurations.length; ix++) {
            ParameterConfiguration conf = parameterConfigurations[ix];
            if (!(conf instanceof AParameterConfiguration)) {
                throw new IllegalArgumentException("deprecated parameter Configuration");
            }
            parms[ix] = (AParameterConfiguration) conf;

        }
        return parms;
    }

    /**
     * removes a list of parameter configurations
     *
     * @param parameterConfigurations
     * @throws ImmutableStateException if operated on an object registered to the database
     */
    public void removeParameterConfigurations(ParameterConfiguration... parameterConfigurations) {
        if (isReadOnly()) {
            throw new ImmutableStateException("Parameter Configuration list");
        }
        for (ParameterConfiguration parameterConfiguration : parameterConfigurations) {
            this.parameterConfigurations.remove(parameterConfiguration);
        }
    }


    /**
     * prepares a new executable configuration data with these modified parameters.
     * used by <TT>PreparedConfiguration</TT> objects.
     *
     * @return
     */
    public ComponentNode getModifiedConfigurationData() {
        ComponentNode res = null;
        ComponentNode componentNode = subsystemDesc.getTopComponentNode() ;
        componentNode = componentNode.clone();
        for (AParameterConfiguration parameterConfiguration : getParameterConfigurations()) {
            ParameterPath path = parameterConfiguration.getPath();
            String componentName = path.getComponentName();
            String codeName = path.getCodeName();
            if (codeName != null && !"".equals(codeName)) {
                throw new UnsupportedOperationException(" no change on methods yet --> " + codeName);
            }
            String parameterName = path.getParameterName();
            ComponentNode goalComponent = (ComponentNode) componentNode.getNodeByName(componentName);
            // get Parameter
            Object rawParm = goalComponent.getAttributes().get(parameterName);
            // not null
            if (rawParm instanceof HollowParm) {
                HollowParm hollow = (HollowParm) rawParm;
                hollow.modifyChecked(parameterConfiguration.getValue());
            } else {
                throw new IllegalArgumentException("parameter not modifiable" + rawParm);
            }
        }
        res = componentNode;

        return res;
    }

}
