package org.lsst.ccs.startup;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.config.*;
import org.lsst.gruth.jutils.ComponentNode;
import org.lsst.gruth.nodes.ComponentFactory;
import org.lsst.gruth.nodes.Utils;

import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.util.Properties;

/**
 *
 * A collection of static utilities linked to boot operations.
 * Note that:
 * <UL>
 *     <LI>
 *      a descriptive node represents a configuration description
 *      <LI/>
 *      an effective node is the real subsystem and module hierarchy (ready to run)
 *      <LI/>
 *      a "raw" subsystem is without configuration data
 * </UL>
 * @author bamade
 */
// Date: 05/06/12

public class BootUtils {
    public static class LocalBootObject {
        ComponentNode descriptiveNode;
        ComponentNode effectiveNode;
        Subsystem subsystem ;
        String subsystemName ;
        String configurationName ;
        String tagName ;

        public LocalBootObject(String pathName) throws Exception{
            setNames(pathName, null);
            descriptiveNode = BootUtils.getDescriptiveNode(pathName) ;
            subsystemFromDescriptionNode(descriptiveNode,subsystemName,configurationName,tagName, null);

        }

        public LocalBootObject(String pathName, String configFileName) throws Exception {
            setNames(pathName, configFileName);
            descriptiveNode = BootUtils.getDescriptiveNode(pathName) ;
            Properties configProps = null;
            if (configFileName != null) {
                configProps = new Properties();
                Reader reader = new FileReader(configFileName);
                configProps.load(reader);
            }
            subsystemFromDescriptionNode(descriptiveNode,subsystemName,configurationName,tagName, configProps);
        }

        private void setNames(String pathName, String configFileName) {
            NamesAndTag namesAndTag = BootUtils.namesFromPath(pathName) ;
            //split name
            subsystemName = namesAndTag.getSubsystemName() ;
            configurationName = namesAndTag.getConfigName() ;
            tagName = namesAndTag.getTag() ;
            if (configFileName != null ) {
                namesAndTag = BootUtils.namesFromPath(configFileName) ;
                if(! subsystemName.equals(namesAndTag.getSubsystemName())) {
                    throw new IllegalArgumentException(" subsystem names differs :" + subsystemName + " <> " + namesAndTag.getSubsystemName()) ;
                }
                String newConfigName = namesAndTag.getConfigName() ;
                String newTagName = namesAndTag.getTag() ;
                if(configurationName.length()!=0 && newConfigName.length()!= 0) {
                    if(! configurationName.equals(newConfigName)) {
                        throw new IllegalArgumentException(" configuration name clash :" + configurationName + " <> " + newConfigName) ;
                    }
                }
                if(tagName.length()!=0 && newTagName.length()!= 0) {
                    if(! tagName.equals(newTagName)) {
                        throw new IllegalArgumentException(" tag name clash :" + tagName + " <> " + newTagName) ;
                    }
                }
                if(newConfigName.length() != 0) {
                    configurationName = newConfigName ;
                }
                if(newTagName.length() != 0) {
                    tagName = newTagName ;
                }
            }

        }

        public ComponentNode getDescriptiveNode() {
            return descriptiveNode;
        }

        public ComponentNode getEffectiveNode() {
            return effectiveNode;
        }

        public Subsystem getSubsystem() {
            return subsystem;
        }

        public String getSubsystemName() {
            return subsystemName;
        }

        public String getConfigurationName() {
            return configurationName;
        }

        public String getTagName() {
            return tagName;
        }


        public Object getRunningObject(String key) {
            return effectiveNode.getIndirect(key) ;
        }

        private void subsystemFromDescriptionNode(ComponentNode componentNode, String subsystemName, String configName,
                                                             String tag, Properties configProps) throws Exception{

            SubsystemDescription description = Factories.createSubsystemDescription(subsystemName, tag, userName, version,
                    componentNode, DataFlavour.PURE_OBJECT_TREE, SubsystemDescription.DEFAULT_TREE_PARAMETER_FILTER);
            MemoryDAO dao = new MemoryDAO() ;
            dao.saveAbstractDescription(description);
            // if no configuration create an empty configuration
            ConfigProfile configProfile = Factories.createConfigProfile(description,configName,tag, userName, level) ;
            if(configProps == null) {
            } else {
                // else merge with properties
                configProfile.mergeProperties(configProps);
                componentNode = configProfile.getModifiedConfigurationData();
            }
            dao.saveAbstractConfig(configProfile);
            effectiveNode = BootUtils.getEffectiveNode(componentNode) ;
            //system.start()
            subsystem =  getRawSubsystemFromEffectiveNode(effectiveNode, configProfile, subsystemName,configName, tag) ;
            // set up configuration access
            LocalConfigurationProxy configurationAccess = new LocalConfigurationProxy(configProfile) ;
            subsystem.setConfigurationProxy(configurationAccess);

        }
    }
    //TODO .... set?
    // defult values
    static String userName = "" ;
    static String version = "1.0" ;
    static int level = PackCst.DESIGNER_LEVEL ;

    /**
     * gets a descriptive node from a file
     * @param pathName
     * @return
     * @throws Exception
     */
    public static ComponentNode getDescriptiveNode(String pathName) throws Exception {
        ComponentNode componentNode = null ;
        InputStream is = getInput(BootUtils.class, pathName) ;
        if(pathName.endsWith(".ser")) {
            ObjectInputStream ois = new ObjectInputStream(new BufferedInputStream(is));
            componentNode = (ComponentNode) ois.readObject();

        } else if( pathName.endsWith(".groovy") || pathName.endsWith(".groo")){
            //TODO: handle the encoding!
            componentNode = Utils.getComponentNodeFromGroovy(is, "ISO-8859-1");

        }
        return componentNode ;
    }

    /**
     * gets an effective node from a description node
     * @param descriptionNode
     * @return
     * @throws InvocationTargetException
     * @throws NoSuchMethodException
     * @throws IllegalAccessException
     */
    public static ComponentNode getEffectiveNode(ComponentNode descriptionNode) throws InvocationTargetException, NoSuchMethodException, IllegalAccessException {
        ComponentFactory fact = new ComponentFactory(descriptionNode) ;
        ComponentNode top = fact.buildObjectTreeFrom(descriptionNode) ;
        top.doCallsFromTop() ;
        return top ;
    }


    /**
     * gets an effective node from a file
     * @param pathName
     * @return
     * @throws Exception
     */
    public static ComponentNode getEffectiveNodeFromFile(String pathName) throws Exception {
        ComponentNode descriptiveNode = getDescriptiveNode(pathName) ;
        return getEffectiveNode(descriptiveNode)  ;
    }



    public static Subsystem getRawSubsystemFromEffectiveNode(ComponentNode effectiveNode, ConfigProfile profile , String subsystemName,
                                                             String configName, String tag) {
        LocalConfigurationProxy configurationAccess = new LocalConfigurationProxy(profile) ;
        NodeModularSubsystem system = new NodeModularSubsystem(subsystemName, configurationAccess) ;
        system.registerNodes(effectiveNode) ;
        //system.setConfiguration(configName);
        //system.setTag(tag);
        return system ;
    }

    @Deprecated
    public static Subsystem getRawSubsystemFromEffectiveNode(ComponentNode effectiveNode,ConfigProfile profile, String pathName) {
        NamesAndTag namesAndTag = BootUtils.namesFromPath(pathName) ;
        //split name
        String subsystemName = namesAndTag.getSubsystemName() ;
        String configName = namesAndTag.getConfigName() ;
        String tag = namesAndTag.getTag() ;
        return getRawSubsystemFromEffectiveNode(effectiveNode, profile, subsystemName, configName, tag) ;
    }

    public static Subsystem rawSubsystemFromDescriptionNode(ComponentNode descriptionNode,
                                                            ConfigProfile profile ,
                                                            String subsystemName, String configName,
                                                            String tag) throws Exception {
        ComponentNode top = getEffectiveNode(descriptionNode) ;
        //system.start()
        return getRawSubsystemFromEffectiveNode(top, profile, subsystemName, configName, tag) ;
    }

    @Deprecated
    public static void bootSubsystemFromDescriptionNode(ComponentNode node, ConfigProfile profile,  String subsystemName,
                                                        String configName, String tag) throws Exception {
        Subsystem system = rawSubsystemFromDescriptionNode(node, profile, subsystemName, configName, tag) ;
        system.start() ;
    }

    @Deprecated
    public static void bootSubsystemFromEffectiveNode (ComponentNode effectiveNode,ConfigProfile profile, String subsystemName,
                                                String configName, String tag) {
        Subsystem system = getRawSubsystemFromEffectiveNode(effectiveNode, profile,  subsystemName, configName, tag) ;
        system.start() ;
    }

    @Deprecated
    public static void bootSubsystemFromEffectiveNode(ComponentNode effectiveNode,ConfigProfile profile, String pathName) throws Exception {
        NamesAndTag namesAndTag = BootUtils.namesFromPath(pathName) ;
        //split name
        String subsystemName = namesAndTag.getSubsystemName() ;
        String configName = namesAndTag.getConfigName() ;
        String tag = namesAndTag.getTag() ;
        bootSubsystemFromEffectiveNode(effectiveNode, profile, subsystemName, configName, tag);
    }


    public static Subsystem getSubsystemFromFile(String pathName ) throws Exception {
        /** WAS
        ComponentNode descriptiveNode = getDescriptiveNode(pathName) ;
        NamesAndTag namesAndTag = BootUtils.namesFromPath(pathName) ;
        //split name
        String subsystemName = namesAndTag.getSubsystemName() ;
        String configName = namesAndTag.getConfigName() ;
        String tag = namesAndTag.getTag() ;
        Subsystem res = rawSubsystemFromDescriptionNode(descriptiveNode, subsystemName, configName, tag) ;
        return res;
        */
        return getSubsystemFromFile(pathName, null) ;
    }

    public static Subsystem getSubsystemFromFile(String pathName, String propertiesFileName) throws Exception {
        LocalBootObject localBootObject ;
        if(propertiesFileName == null) {
            localBootObject = new LocalBootObject(pathName) ;
        } else {

            localBootObject = new LocalBootObject(pathName, propertiesFileName) ;
        }
        return localBootObject.getSubsystem() ;

        /*
        * if there is a second argument it is the configuration file
        * in that case the first argument is only a raw subsystem description without configuration or tag name
       * */

        /*
        Properties configProps = null;
        if (propertiesFileName != null) {
            configProps = new Properties();
            Reader reader = new FileReader(propertiesFileName);
            configProps.load(reader);
        }
        NamesAndTag namesAndTag = BootUtils.namesFromPath(pathName);
        //split name
        String subsystemName = namesAndTag.getSubsystemName();
        String configName = namesAndTag.getConfigName();
        String tag = namesAndTag.getTag();
        // if configProps not null then configuration name should be empty
        if ((configProps != null) && !"".equals(configName)) {
            throw new IllegalArgumentException("cannot configure with " + propertiesFileName
                    + " over a file which contains already a configuration " + configName);
        }
        // set up a new config Name
        if (configProps != null) {
            NamesAndTag configNamesAndTag = BootUtils.namesFromPath(pathName);
            String configSubsystemName = configNamesAndTag.getSubsystemName();
            if (!subsystemName.equals(configSubsystemName)) {
                throw new IllegalArgumentException(" configuration file " + propertiesFileName
                        + "apparently not configuring subsystem " + subsystemName + " (based on files names)");
            }
            configName = configNamesAndTag.getConfigName();
            tag = configNamesAndTag.getTag();
        }
        ComponentNode componentNode = null ;
        InputStream is = BootUtils.getInput(BootUtils.class, pathName) ;
        // now we distinguish between .groovy files or serialized objects
        if(pathName.endsWith(".ser")) {
            ObjectInputStream ois = new ObjectInputStream(new BufferedInputStream(is));
            componentNode = (ComponentNode) ois.readObject();

        } else if( pathName.endsWith(".groovy") || pathName.endsWith(".groo")){
            //TODO: handle the encoding!
            componentNode = Utils.getComponentNodeFromGroovy(is, "ISO-8859-1");
        } else {
            throw new IllegalArgumentException(" file name extension not recognised :" + pathName) ;
        }
        // end duplication

        Subsystem subsystem = subsystemFromDescriptionNode(componentNode,subsystemName,configName,tag,configProps) ;
        return subsystem ;
        */
    }

    /*
    public static Subsystem subsystemFromDescriptionNode(ComponentNode componentNode, String subsystemName, String configName,
                                                         String tag, Properties configProps) throws Exception{

        SubsystemDescription description = Factories.createSubsystemDescription(subsystemName, tag, userName, version,
                componentNode, DataFlavour.PURE_OBJECT_TREE, SubsystemDescription.DEFAULT_TREE_PARAMETER_FILTER);
        MemoryDAO dao = new MemoryDAO() ;
        dao.saveAbstractDescription(description);
        // if no configuration create an empty configuration
        ConfigProfile configProfile = Factories.createConfigProfile(description,configName,tag, userName, level) ;
        if(configProps == null) {
        } else {
            // else merge with properties
            configProfile.mergeProperties(configProps);
            componentNode = configProfile.getModifiedConfigurationData();
        }
        dao.saveAbstractConfig(configProfile);

        //boot susbystem
        Subsystem subsystem = BootUtils.rawSubsystemFromDescriptionNode(componentNode, subsystemName, configName, tag) ;
        // set up configuration access
        LocalConfigurationProxy configurationProxy = new LocalConfigurationProxy(configProfile) ;
        subsystem.setConfigurationProxy(configurationProxy);
        return subsystem ;
    } */

    /*
    public static Subsystem subsystemFromDescriptionNode(ComponentNode componentNode, String subsystemName, String configName,
                                                         String tag) throws Exception{
        return subsystemFromDescriptionNode(componentNode,subsystemName,configName,tag,null) ;
    }
    */





    /**
     * saves the latest run configuration in a local cache.
     * This is saved in a file (see naming  convention handled by <TT>baseNameFromNames</TT>
     * that will reside in tha cache directory.
     * The name of this file will be reported in another text file in this directory named "latest.txt".
     * <P/>
     * TODO: The path of the cache directory is to be determined.
     * @param node
     * @param subsystemName
     * @param configName
     * @param tag
     */
    public static void saveInCache(ComponentNode node, String subsystemName,
                                   String configName, String tag) {
        //TODO: implement saveInCache
    }

    public static ComponentNode getLatestInCache() {
        return null ;
    }

    public static void bootFromCache() {
        throw new UnsupportedOperationException("bootFromCache") ;
    }

    /**
     * searches for a file as a local file or a resource linked to a class or a global resource.
     * @param clazz
     * @param pathName
     * @return
     * @throws IOException
     */
    // TODO: possible duplication of code in core utils
    public static InputStream getInput(Class clazz ,String pathName) throws IOException {
        InputStream is = null ;
        File file = new File(pathName);
        if(file.exists()) {
            is = new FileInputStream(pathName);

        } else {
            is = clazz.getResourceAsStream(pathName) ;
            if(is == null) {
                is = clazz.getResourceAsStream("/"+pathName) ;
            }
            if(is == null) {
                throw new IllegalArgumentException("no resource "+ pathName);
            }
        }
        return is ;
    }

    /**
     * gets subsystem name, configName and tag name from pathName
     * @param pathName
     * @return
     */
    public static NamesAndTag namesFromPath(String pathName) {
        //split name
        String subsystemName = "" ;
        String configName = "" ;
        String tag = "" ;
        int indexDot =  pathName.lastIndexOf('.');
        int lastPath = pathName.lastIndexOf('/');
        String rawName = pathName.substring(lastPath + 1, indexDot);
        String[] elements = rawName.split("_");
        //falltrough !
        switch(elements.length) {
            case 3: tag = elements[2] ;
            case 2: configName = elements[1] ;
            case 1: subsystemName = elements[0] ;
        }
        return new NamesAndTag(subsystemName,configName,tag) ;
    }

    /**
     * creates a fileName from a NamesAndTag object (reverse operation of <TT>namesFromPath</TT>
     * @param namesAndTag
     * @return
     */
    public static String baseNameFromNames(NamesAndTag namesAndTag) {
        return String.format("%s_%s_%s", namesAndTag.getSubsystemName(), namesAndTag.getConfigName(),
                namesAndTag.getTag()) ;
    }
}
